@extends('layouts.app')

@section('title', 'Nova Bacia Hidrográfica')

@section('content')
<div class="space-y-6">
    <!-- Header -->
    <div class="bg-white rounded-lg shadow p-6">
        <div class="flex items-center justify-between">
            <div>
                <h1 class="text-2xl font-bold text-gray-800">
                    <i class="fas fa-plus mr-3 text-cyan-600"></i>
                    Nova Bacia Hidrográfica
                </h1>
                <p class="text-gray-600 mt-2">Adicionar uma nova bacia hidrográfica ao sistema</p>
            </div>
            <div class="flex space-x-4">
                <a href="{{ route('bacias.index') }}" 
                   class="bg-gray-600 text-white px-4 py-2 rounded-md hover:bg-gray-700 transition-colors">
                    <i class="fas fa-arrow-left mr-2"></i>
                    Voltar à Lista
                </a>
            </div>
        </div>
    </div>

    <!-- Formulário -->
    <div class="bg-white rounded-lg shadow">
        <div class="px-6 py-4 border-b border-gray-200">
            <h3 class="text-lg font-medium text-gray-900">
                <i class="fas fa-water mr-2 text-cyan-600"></i>
                Informações da Bacia Hidrográfica
            </h3>
        </div>
        
        <form method="POST" action="{{ route('bacias.store') }}" class="p-6 space-y-6">
            @csrf
            
            <!-- Identificação -->
            <div class="bg-blue-50 border border-blue-200 rounded-lg p-4">
                <h4 class="text-sm font-semibold text-blue-800 uppercase tracking-wide mb-4">
                    <i class="fas fa-id-card mr-2"></i>
                    Identificação
                </h4>
                <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Nome da Bacia *</label>
                        <input type="text" name="nome" value="{{ old('nome') }}" required
                               class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-cyan-500 @error('nome') border-red-500 @enderror"
                               placeholder="Ex: Bacia do Rio Zambeze">
                        @error('nome')
                            <p class="text-red-500 text-xs mt-1">{{ $message }}</p>
                        @enderror
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Código *</label>
                        <input type="text" name="codigo" value="{{ old('codigo') }}" required
                               class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-cyan-500 @error('codigo') border-red-500 @enderror"
                               placeholder="Ex: BRZ-001">
                        @error('codigo')
                            <p class="text-red-500 text-xs mt-1">{{ $message }}</p>
                        @enderror
                    </div>
                </div>
            </div>

            <!-- Características Físicas -->
            <div class="bg-green-50 border border-green-200 rounded-lg p-4">
                <h4 class="text-sm font-semibold text-green-800 uppercase tracking-wide mb-4">
                    <i class="fas fa-mountain mr-2"></i>
                    Características Físicas
                </h4>
                <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Área da Bacia (km²)</label>
                        <input type="number" name="area_km2" value="{{ old('area_km2') }}" step="0.01"
                               class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-cyan-500 @error('area_km2') border-red-500 @enderror"
                               placeholder="Ex: 1570000">
                        @error('area_km2')
                            <p class="text-red-500 text-xs mt-1">{{ $message }}</p>
                        @enderror
                        <p class="text-xs text-gray-500 mt-1">Área total da bacia hidrográfica em quilômetros quadrados</p>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Comprimento do Rio Principal (km)</label>
                        <input type="number" name="comprimento_rio_principal" value="{{ old('comprimento_rio_principal') }}" step="0.01"
                               class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-cyan-500 @error('comprimento_rio_principal') border-red-500 @enderror"
                               placeholder="Ex: 2574">
                        @error('comprimento_rio_principal')
                            <p class="text-red-500 text-xs mt-1">{{ $message }}</p>
                        @enderror
                        <p class="text-xs text-gray-500 mt-1">Comprimento do curso principal de água</p>
                    </div>
                </div>
            </div>

            <!-- Abrangência Geográfica -->
            <div class="bg-purple-50 border border-purple-200 rounded-lg p-4">
                <h4 class="text-sm font-semibold text-purple-800 uppercase tracking-wide mb-4">
                    <i class="fas fa-map-marked-alt mr-2"></i>
                    Abrangência Geográfica
                </h4>
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-3">Províncias Abrangidas *</label>

                    @php
                        $provinciasPorRegiao = \App\Helpers\ProvinciaHelper::getProvinciasPorRegiao();
                        $selected = old('provincias_abrangidas', []);
                    @endphp

                    <!-- Interface melhorada com checkboxes organizados por região -->
                    <div class="space-y-4">
                        @foreach($provinciasPorRegiao as $regiao => $provincias)
                            <div class="bg-white rounded-lg border border-gray-200 p-3">
                                <h5 class="text-sm font-semibold text-gray-800 mb-2 flex items-center">
                                    @if($regiao == 'Norte')
                                        <i class="fas fa-arrow-up text-blue-500 mr-2"></i>
                                    @elseif($regiao == 'Centro')
                                        <i class="fas fa-arrows-alt-h text-green-500 mr-2"></i>
                                    @else
                                        <i class="fas fa-arrow-down text-orange-500 mr-2"></i>
                                    @endif
                                    Região {{ $regiao }}
                                    <span class="ml-2 text-xs text-gray-500">({{ count($provincias) }} províncias)</span>
                                </h5>
                                <div class="grid grid-cols-1 md:grid-cols-2 gap-2">
                                    @foreach($provincias as $provincia)
                                        <label class="flex items-center space-x-2 p-2 rounded-md hover:bg-gray-50 cursor-pointer transition-colors duration-150">
                                            <input type="checkbox"
                                                   name="provincias_abrangidas[]"
                                                   value="{{ $provincia }}"
                                                   {{ in_array($provincia, $selected) ? 'checked' : '' }}
                                                   class="w-4 h-4 text-purple-600 border-gray-300 rounded focus:ring-purple-500 focus:ring-2">
                                            <div class="flex-1">
                                                <span class="text-sm font-medium text-gray-900">{{ $provincia }}</span>
                                                @php
                                                    $info = \App\Helpers\ProvinciaHelper::getInfoProvincia($provincia);
                                                @endphp
                                                @if($info)
                                                    <div class="text-xs text-gray-500">
                                                        Capital: {{ $info['capital'] }}
                                                        @if($info['costa'])
                                                            <span class="inline-flex items-center px-1.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800 ml-1">
                                                                <i class="fas fa-water mr-1"></i>Costa
                                                            </span>
                                                        @endif
                                                    </div>
                                                @endif
                                            </div>
                                        </label>
                                    @endforeach
                                </div>
                            </div>
                        @endforeach
                    </div>

                    @error('provincias_abrangidas')
                        <p class="text-red-500 text-xs mt-2">{{ $message }}</p>
                    @enderror

                    <!-- Ações Rápidas -->
                    <div class="mt-4 pt-3 border-t border-purple-200">
                        <div class="flex flex-wrap gap-2">
                            <button type="button" onclick="selecionarRegiao('Norte')"
                                    class="text-xs px-3 py-1 bg-blue-100 text-blue-800 rounded-full hover:bg-blue-200 transition-colors">
                                <i class="fas fa-arrow-up mr-1"></i>Selecionar Norte
                            </button>
                            <button type="button" onclick="selecionarRegiao('Centro')"
                                    class="text-xs px-3 py-1 bg-green-100 text-green-800 rounded-full hover:bg-green-200 transition-colors">
                                <i class="fas fa-arrows-alt-h mr-1"></i>Selecionar Centro
                            </button>
                            <button type="button" onclick="selecionarRegiao('Sul')"
                                    class="text-xs px-3 py-1 bg-orange-100 text-orange-800 rounded-full hover:bg-orange-200 transition-colors">
                                <i class="fas fa-arrow-down mr-1"></i>Selecionar Sul
                            </button>
                            <button type="button" onclick="selecionarTodas()"
                                    class="text-xs px-3 py-1 bg-purple-100 text-purple-800 rounded-full hover:bg-purple-200 transition-colors">
                                <i class="fas fa-check-double mr-1"></i>Selecionar Todas
                            </button>
                            <button type="button" onclick="limparSelecao()"
                                    class="text-xs px-3 py-1 bg-gray-100 text-gray-800 rounded-full hover:bg-gray-200 transition-colors">
                                <i class="fas fa-times mr-1"></i>Limpar
                            </button>
                        </div>
                    </div>

                    <p class="text-xs text-gray-500 mt-2">
                        <i class="fas fa-info-circle mr-1"></i>
                        Selecione uma ou mais províncias onde a bacia hidrográfica se estende
                    </p>
                </div>
            </div>

            <!-- Descrição -->
            <div class="bg-gray-50 border border-gray-200 rounded-lg p-4">
                <h4 class="text-sm font-semibold text-gray-800 uppercase tracking-wide mb-4">
                    <i class="fas fa-file-text mr-2"></i>
                    Descrição e Observações
                </h4>
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">Descrição</label>
                    <textarea name="descricao" rows="4"
                              class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-cyan-500 @error('descricao') border-red-500 @enderror"
                              placeholder="Descrição detalhada da bacia hidrográfica, incluindo características importantes, afluentes principais, etc.">{{ old('descricao') }}</textarea>
                    @error('descricao')
                        <p class="text-red-500 text-xs mt-1">{{ $message }}</p>
                    @enderror
                </div>
            </div>

            <!-- Botões -->
            <div class="flex items-center justify-end space-x-4 pt-6 border-t border-gray-200">
                <a href="{{ route('bacias.index') }}" 
                   class="px-6 py-2 border border-gray-300 rounded-md text-gray-700 hover:bg-gray-50 transition-colors">
                    <i class="fas fa-times mr-2"></i>
                    Cancelar
                </a>
                <button type="submit"
                        class="px-6 py-2 bg-cyan-600 text-white rounded-md hover:bg-cyan-700 transition-colors">
                    <i class="fas fa-save mr-2"></i>
                    Criar Bacia Hidrográfica
                </button>
            </div>
        </form>
    </div>
</div>

<script>
// JavaScript para funcionalidades de seleção rápida de províncias
document.addEventListener('DOMContentLoaded', function() {
    // Mapeamento das províncias por região
    const regioes = {
        'Norte': ['Cabo Delgado', 'Nampula', 'Niassa'],
        'Centro': ['Manica', 'Sofala', 'Tete', 'Zambézia'],
        'Sul': ['Gaza', 'Inhambane', 'Maputo Província', 'Maputo Cidade']
    };

    // Função para selecionar todas as províncias de uma região
    window.selecionarRegiao = function(regiao) {
        const provincias = regioes[regiao];
        if (!provincias) return;

        provincias.forEach(provincia => {
            const checkbox = document.querySelector(`input[value="${provincia}"]`);
            if (checkbox) {
                checkbox.checked = true;
                // Adicionar efeito visual
                animarCheckbox(checkbox);
            }
        });

        // Mostrar feedback visual
        mostrarFeedback(`Região ${regiao} selecionada`, 'success');
    };

    // Função para selecionar todas as províncias
    window.selecionarTodas = function() {
        const checkboxes = document.querySelectorAll('input[name="provincias_abrangidas[]"]');
        checkboxes.forEach(checkbox => {
            checkbox.checked = true;
            animarCheckbox(checkbox);
        });
        mostrarFeedback('Todas as províncias selecionadas', 'success');
    };

    // Função para limpar todas as seleções
    window.limparSelecao = function() {
        const checkboxes = document.querySelectorAll('input[name="provincias_abrangidas[]"]');
        checkboxes.forEach(checkbox => {
            checkbox.checked = false;
            animarCheckbox(checkbox);
        });
        mostrarFeedback('Seleção limpa', 'info');
    };

    // Função para animar checkbox
    function animarCheckbox(checkbox) {
        const label = checkbox.closest('label');
        if (label) {
            label.classList.add('scale-105');
            setTimeout(() => {
                label.classList.remove('scale-105');
            }, 200);
        }
    }

    // Função para mostrar feedback visual
    function mostrarFeedback(mensagem, tipo) {
        // Remover feedback anterior se existir
        const feedbackAnterior = document.querySelector('.feedback-provincial');
        if (feedbackAnterior) {
            feedbackAnterior.remove();
        }

        // Criar elemento de feedback
        const feedback = document.createElement('div');
        feedback.className = `feedback-provincial fixed top-4 right-4 px-4 py-2 rounded-lg shadow-lg z-50 transition-all duration-300 transform translate-x-full`;

        // Definir cores baseado no tipo
        const cores = {
            'success': 'bg-green-500 text-white',
            'info': 'bg-blue-500 text-white',
            'warning': 'bg-yellow-500 text-gray-900'
        };

        feedback.className += ` ${cores[tipo] || cores.info}`;
        feedback.innerHTML = `
            <div class="flex items-center space-x-2">
                <i class="fas fa-${tipo === 'success' ? 'check' : tipo === 'info' ? 'info-circle' : 'exclamation-triangle'}"></i>
                <span class="font-medium">${mensagem}</span>
            </div>
        `;

        // Adicionar ao DOM
        document.body.appendChild(feedback);

        // Animar entrada
        setTimeout(() => {
            feedback.classList.remove('translate-x-full');
        }, 100);

        // Remover após 3 segundos
        setTimeout(() => {
            feedback.classList.add('translate-x-full');
            setTimeout(() => {
                if (feedback.parentNode) {
                    feedback.remove();
                }
            }, 300);
        }, 3000);
    }

    // Adicionar contador dinâmico de províncias selecionadas
    function atualizarContador() {
        const checkboxes = document.querySelectorAll('input[name="provincias_abrangidas[]"]:checked');
        const contador = checkboxes.length;

        // Encontrar ou criar elemento contador
        let contadorElement = document.querySelector('.contador-provincias');
        if (!contadorElement) {
            contadorElement = document.createElement('div');
            contadorElement.className = 'contador-provincias text-sm font-medium text-purple-600 mt-2';

            const container = document.querySelector('label[for="provincias_abrangidas"]')?.parentNode ||
                             document.querySelector('input[name="provincias_abrangidas[]"]')?.closest('.space-y-4')?.parentNode;

            if (container) {
                container.appendChild(contadorElement);
            }
        }

        // Atualizar texto do contador
        if (contador === 0) {
            contadorElement.innerHTML = '<i class="fas fa-info-circle mr-1"></i>Nenhuma província selecionada';
            contadorElement.className = 'contador-provincias text-sm font-medium text-gray-500 mt-2';
        } else if (contador === 1) {
            contadorElement.innerHTML = '<i class="fas fa-map-marker-alt mr-1"></i>1 província selecionada';
            contadorElement.className = 'contador-provincias text-sm font-medium text-purple-600 mt-2';
        } else {
            contadorElement.innerHTML = `<i class="fas fa-map-marked-alt mr-1"></i>${contador} províncias selecionadas`;
            contadorElement.className = 'contador-provincias text-sm font-medium text-purple-600 mt-2';
        }
    }

    // Adicionar listeners para atualizar contador
    document.querySelectorAll('input[name="provincias_abrangidas[]"]').forEach(checkbox => {
        checkbox.addEventListener('change', atualizarContador);
    });

    // Atualizar contador inicial
    atualizarContador();

    // Validação visual aprimorada
    const form = document.querySelector('form');
    if (form) {
        form.addEventListener('submit', function(e) {
            const checkboxes = document.querySelectorAll('input[name="provincias_abrangidas[]"]:checked');
            if (checkboxes.length === 0) {
                e.preventDefault();
                mostrarFeedback('Selecione pelo menos uma província', 'warning');

                // Scroll até a seção de províncias
                const provinciaSection = document.querySelector('.bg-purple-50');
                if (provinciaSection) {
                    provinciaSection.scrollIntoView({ behavior: 'smooth', block: 'center' });
                    provinciaSection.classList.add('ring-2', 'ring-red-500', 'ring-opacity-50');
                    setTimeout(() => {
                        provinciaSection.classList.remove('ring-2', 'ring-red-500', 'ring-opacity-50');
                    }, 3000);
                }
            }
        });
    }
});
</script>

@endsection