@extends('layouts.app')

@section('title', 'Curvas Cota-Volume - ' . $barragem->nome)

@section('content')
<div class="space-y-6">
    <!-- Header -->
    <div class="bg-white rounded-lg shadow p-6">
        <div class="flex items-center justify-between">
            <div>
                <h1 class="text-2xl font-bold text-gray-800">
                    <i class="fas fa-chart-area mr-3 text-blue-600"></i>
                    Curvas Cota-Volume - {{ $barragem->nome }}
                </h1>
                <p class="text-gray-600 mt-2">Gestão das curvas que relacionam cota com volume e área</p>
            </div>
            <div class="flex space-x-3">
                <button onclick="showNovaCoordenadasModal()" class="bg-green-600 hover:bg-green-700 text-white px-4 py-2 rounded-lg">
                    <i class="fas fa-plus mr-2"></i>
                    Novo Ponto
                </button>
                <a href="{{ route('barragens.show', $barragem) }}" class="bg-gray-600 hover:bg-gray-700 text-white px-4 py-2 rounded-lg">
                    <i class="fas fa-arrow-left mr-2"></i>
                    Voltar
                </a>
            </div>
        </div>
    </div>

    <!-- Explicação -->
    <div class="bg-blue-50 rounded-lg p-4">
        <div class="flex items-start">
            <i class="fas fa-info-circle text-blue-600 mt-1 mr-3"></i>
            <div>
                <h3 class="font-medium text-blue-900">Como Funciona</h3>
                <p class="text-sm text-blue-800 mt-1">
                    As curvas cota-volume relacionam o nível da água (cota) com o volume armazenado na barragem.
                    <br>• <strong>Cota:</strong> Altura da água em metros
                    <br>• <strong>Volume:</strong> Quantidade de água armazenada em Mm³
                    <br>• <strong>Área:</strong> Superfície do espelho d'água em km² (opcional)
                </p>
            </div>
        </div>
    </div>

    <div class="grid grid-cols-1 2xl:grid-cols-2 gap-8">
        <!-- Gráfico -->
        <div class="bg-white rounded-lg shadow overflow-hidden">
            <div class="px-6 py-4 border-b border-gray-200 bg-gray-50">
                <h3 class="text-lg font-medium text-gray-900">Gráfico da Curva</h3>
            </div>
            <div class="p-6">
                @if($curvas->count() > 1)
                <div class="h-80">
                    <canvas id="curvaVolumeChart" class="w-full h-full"></canvas>
                </div>
                @else
                <div class="h-80 flex items-center justify-center bg-gray-50 rounded-lg">
                    <div class="text-center">
                        <i class="fas fa-chart-line text-4xl text-gray-400 mb-4"></i>
                        <p class="text-gray-600">Adicione pelo menos 2 pontos para visualizar o gráfico</p>
                    </div>
                </div>
                @endif
            </div>
        </div>

        <!-- Lista de Coordenadas -->
        <div class="bg-white rounded-lg shadow overflow-hidden">
            <div class="px-6 py-4 border-b border-gray-200 bg-gray-50">
                <h3 class="text-lg font-medium text-gray-900">Pontos da Curva</h3>
            </div>

            @if($curvas->count() > 0)
            <div class="overflow-x-auto">
                <table class="min-w-full divide-y divide-gray-200">
                    <thead class="bg-gray-50">
                        <tr>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Cota (m)</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Volume (Mm³)</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Área (km²)</th>
                            <th class="px-6 py-3 text-center text-xs font-medium text-gray-500 uppercase">Ações</th>
                        </tr>
                    </thead>
                    <tbody class="bg-white divide-y divide-gray-200">
                        @foreach($curvas as $curva)
                        <tr class="hover:bg-gray-50">
                            <td class="px-6 py-4 text-sm font-medium text-gray-900">{{ number_format($curva->cota, 2) }}</td>
                            <td class="px-6 py-4 text-sm text-gray-900">{{ number_format($curva->volume, 2) }}</td>
                            <td class="px-6 py-4 text-sm text-gray-900">
                                {{ $curva->area ? number_format($curva->area, 2) : '--' }}
                            </td>
                            <td class="px-6 py-4 text-center">
                                <div class="flex justify-center space-x-2">
                                    <button onclick="editarCoordenadas({{ $curva->id }}, {{ $curva->cota }}, {{ $curva->volume }}, {{ $curva->area ?? 'null' }})"
                                            class="text-blue-600 hover:text-blue-900">
                                        <i class="fas fa-edit"></i>
                                    </button>
                                    <button onclick="removerCoordenadas({{ $curva->id }})"
                                            class="text-red-600 hover:text-red-900">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </div>
                            </td>
                        </tr>
                        @endforeach
                    </tbody>
                </table>
            </div>
            @else
            <div class="text-center py-12">
                <i class="fas fa-chart-area text-4xl text-gray-400 mb-4"></i>
                <h3 class="text-lg font-medium text-gray-900 mb-2">Nenhum ponto cadastrado</h3>
                <p class="text-gray-500 mb-6">Adicione pontos para criar a curva cota-volume desta barragem.</p>
                <button onclick="showNovaCoordenadasModal()" class="bg-green-600 hover:bg-green-700 text-white px-6 py-3 rounded-lg">
                    <i class="fas fa-plus mr-2"></i>
                    Adicionar Primeiro Ponto
                </button>
            </div>
            @endif
        </div>
    </div>
</div>

<!-- Modal Novo/Editar Ponto -->
<div id="coordenadasModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 hidden z-50">
    <div class="flex items-center justify-center min-h-screen p-4">
        <div class="bg-white rounded-xl shadow-2xl w-full max-w-md">
            <form id="coordenadasForm" method="POST" action="{{ route('barragens.curvas-volume.store', $barragem) }}">
                @csrf
                <input type="hidden" id="method" name="_method" value="POST">

                <div class="bg-gradient-to-r from-blue-600 to-blue-700 px-6 py-4 text-white rounded-t-xl">
                    <div class="flex justify-between items-center">
                        <h3 class="text-xl font-bold" id="modalTitle">Novo Ponto da Curva</h3>
                        <button type="button" onclick="hideCoordeanadasModal()" class="text-white hover:text-gray-200">
                            <i class="fas fa-times text-xl"></i>
                        </button>
                    </div>
                </div>

                <div class="p-6 space-y-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Cota (m) *</label>
                        <input type="number" name="cota" id="cota" step="0.001" required
                               class="w-full border-gray-300 rounded-lg focus:ring-blue-500 focus:border-blue-500">
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Volume (Mm³) *</label>
                        <input type="number" name="volume" id="volume" step="0.001" required
                               class="w-full border-gray-300 rounded-lg focus:ring-blue-500 focus:border-blue-500">
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Área (km²)</label>
                        <input type="number" name="area" id="area" step="0.001"
                               class="w-full border-gray-300 rounded-lg focus:ring-blue-500 focus:border-blue-500">
                        <p class="text-xs text-gray-500 mt-1">Campo opcional para calcular áreas</p>
                    </div>

                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Observações</label>
                        <textarea name="observacoes" id="observacoes" rows="2"
                                  class="w-full border-gray-300 rounded-lg focus:ring-blue-500 focus:border-blue-500"></textarea>
                    </div>
                </div>

                <div class="bg-gray-50 px-6 py-4 border-t flex justify-end space-x-3 rounded-b-xl">
                    <button type="button" onclick="hideCoordeanadasModal()" class="bg-gray-300 hover:bg-gray-400 text-gray-700 px-6 py-2 rounded-lg">
                        Cancelar
                    </button>
                    <button type="submit" class="bg-blue-600 hover:bg-blue-700 text-white px-6 py-2 rounded-lg">
                        <i class="fas fa-save mr-2"></i>
                        Salvar
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Include Chart.js -->
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>

<script>
let curvaChart = null;
let currentEditingId = null;

function showNovaCoordenadasModal() {
    currentEditingId = null;
    document.getElementById('modalTitle').textContent = 'Novo Ponto da Curva';
    document.getElementById('coordenadasForm').action = '{{ route('barragens.curvas-volume.store', $barragem) }}';
    document.getElementById('method').value = 'POST';

    // Limpar formulário
    document.getElementById('cota').value = '';
    document.getElementById('volume').value = '';
    document.getElementById('area').value = '';
    document.getElementById('observacoes').value = '';

    document.getElementById('coordenadasModal').classList.remove('hidden');
}

function editarCoordenadas(id, cota, volume, area) {
    currentEditingId = id;
    document.getElementById('modalTitle').textContent = 'Editar Ponto da Curva';
    document.getElementById('coordenadasForm').action = `{{ url('barragens/' . $barragem->id . '/curvas-volume') }}/${id}`;
    document.getElementById('method').value = 'PUT';

    // Preencher formulário
    document.getElementById('cota').value = cota;
    document.getElementById('volume').value = volume;
    document.getElementById('area').value = area || '';

    document.getElementById('coordenadasModal').classList.remove('hidden');
}

function removerCoordenadas(id) {
    if (confirm('Tem certeza que deseja remover este ponto da curva?')) {
        const form = document.createElement('form');
        form.method = 'POST';
        form.action = `{{ url('barragens/' . $barragem->id . '/curvas-volume') }}/${id}`;
        form.innerHTML = `
            @csrf
            @method('DELETE')
        `;
        document.body.appendChild(form);
        form.submit();
    }
}

function hideCoordeanadasModal() {
    document.getElementById('coordenadasModal').classList.add('hidden');
}

// Inicializar gráfico
function initializeCurvaChart() {
    const ctx = document.getElementById('curvaVolumeChart');
    if (!ctx) return;

    fetch('{{ route('api.barragens.curvas-volume.chart', $barragem) }}')
        .then(response => response.json())
        .then(data => {
            curvaChart = new Chart(ctx, {
                type: 'line',
                data: data,
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    scales: {
                        x: {
                            display: true,
                            title: {
                                display: true,
                                text: 'Cota (m)'
                            }
                        },
                        y: {
                            type: 'linear',
                            display: true,
                            position: 'left',
                            title: {
                                display: true,
                                text: 'Volume (Mm³)'
                            }
                        },
                        y1: {
                            type: 'linear',
                            display: data.datasets.length > 1,
                            position: 'right',
                            title: {
                                display: true,
                                text: 'Área (km²)'
                            },
                            grid: {
                                drawOnChartArea: false,
                            },
                        }
                    },
                    plugins: {
                        title: {
                            display: true,
                            text: 'Curva Cota-Volume'
                        },
                        legend: {
                            display: true,
                            position: 'top'
                        }
                    }
                }
            });
        })
        .catch(error => console.error('Erro ao carregar dados do gráfico:', error));
}

// Fechar modal ao clicar fora
document.getElementById('coordenadasModal')?.addEventListener('click', function(e) {
    if (e.target === this) {
        hideCoordeanadasModal();
    }
});

// Inicializar gráfico quando a página carrega
document.addEventListener('DOMContentLoaded', function() {
    @if($curvas->count() > 1)
    initializeCurvaChart();
    @endif
});
</script>
@endsection