@extends('layouts.app')

@section('title', 'Detalhes da Barragem - ' . $barragem->nome)

@section('content')
<div class="min-h-screen bg-gray-50">
    <!-- Top Navigation Bar with Actions -->
    <div class="bg-white shadow-sm border-b border-gray-200 sticky top-0 z-50">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex items-center justify-between h-16">
                <!-- Breadcrumb -->
                <nav class="flex items-center space-x-2 text-sm">
                    <a href="{{ route('dashboard') }}" class="text-gray-500 hover:text-blue-600 transition-colors">
                        <i class="fas fa-home mr-1"></i>Dashboard
                    </a>
                    <i class="fas fa-chevron-right text-gray-300"></i>
                    <a href="{{ route('barragens.index') }}" class="text-gray-500 hover:text-blue-600 transition-colors">Barragens</a>
                    <i class="fas fa-chevron-right text-gray-300"></i>
                    <span class="text-black font-extrabold text-lg">{{ $barragem->nome }}</span>
                </nav>

                <!-- Quick Actions Toolbar -->
                <div class="flex items-center space-x-2">
                    <!-- Primary Actions - Always visible -->
                    <button onclick="addReading()"
                            aria-label="Adicionar nova leitura da barragem"
                            class="inline-flex items-center px-3 py-2 sm:px-4 bg-green-600 hover:bg-green-700 focus:bg-green-700 focus:ring-2 focus:ring-green-500 focus:ring-offset-2 text-white text-sm font-medium rounded-lg transition-all duration-200 shadow-sm">
                        <i class="fas fa-plus mr-0 sm:mr-2" aria-hidden="true"></i>
                        <span class="hidden sm:inline">Nova Leitura</span>
                    </button>

                    <a href="{{ route('barragens.leituras', $barragem) }}"
                       aria-label="Ver todas as leituras da barragem"
                       class="inline-flex items-center px-3 py-2 sm:px-4 bg-blue-600 hover:bg-blue-700 focus:bg-blue-700 focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 text-white text-sm font-medium rounded-lg transition-all duration-200 shadow-sm">
                        <i class="fas fa-chart-line mr-0 sm:mr-2" aria-hidden="true"></i>
                        <span class="hidden sm:inline">Leituras</span>
                    </a>

                    <!-- Secondary Actions - Hidden on small screens, shown in dropdown -->
                    <div class="hidden lg:flex items-center space-x-2">
                        <a href="/graficos/barragens"
                           aria-label="Ver análises e gráficos da barragem"
                           class="inline-flex items-center px-4 py-2 bg-purple-600 hover:bg-purple-700 focus:bg-purple-700 focus:ring-2 focus:ring-purple-500 focus:ring-offset-2 text-white text-sm font-medium rounded-lg transition-all duration-200 shadow-sm">
                            <i class="fas fa-chart-area mr-2" aria-hidden="true"></i>Análises
                        </a>
                        <button onclick="generateReport()"
                                aria-label="Gerar relatório da barragem"
                                class="inline-flex items-center px-4 py-2 bg-orange-600 hover:bg-orange-700 focus:bg-orange-700 focus:ring-2 focus:ring-orange-500 focus:ring-offset-2 text-white text-sm font-medium rounded-lg transition-all duration-200 shadow-sm">
                            <i class="fas fa-file-alt mr-2" aria-hidden="true"></i>Relatório
                        </button>
                        <a href="{{ route('barragens.edit', $barragem) }}"
                           aria-label="Editar dados da barragem"
                           class="inline-flex items-center px-4 py-2 bg-gray-600 hover:bg-gray-700 focus:bg-gray-700 focus:ring-2 focus:ring-gray-500 focus:ring-offset-2 text-white text-sm font-medium rounded-lg transition-all duration-200 shadow-sm">
                            <i class="fas fa-edit mr-2" aria-hidden="true"></i>Editar
                        </a>
                    </div>

                    <!-- Mobile Dropdown Menu -->
                    <div class="relative lg:hidden">
                        <button onclick="toggleMobileMenu()"
                                aria-label="Menu de ações adicionais"
                                aria-expanded="false"
                                aria-haspopup="true"
                                id="mobile-menu-button"
                                class="inline-flex items-center justify-center w-10 h-10 bg-gray-600 hover:bg-gray-700 focus:bg-gray-700 focus:ring-2 focus:ring-gray-500 focus:ring-offset-2 text-white rounded-lg transition-all duration-200 shadow-sm">
                            <i class="fas fa-ellipsis-v" aria-hidden="true"></i>
                        </button>

                        <!-- Dropdown Panel -->
                        <div id="mobile-menu-panel"
                             class="hidden absolute right-0 mt-2 w-48 bg-white rounded-lg shadow-lg border border-gray-200 z-50"
                             role="menu"
                             aria-orientation="vertical"
                             aria-labelledby="mobile-menu-button">
                            <div class="py-1">
                                <a href="/graficos/barragens"
                                   class="group flex items-center px-4 py-3 text-sm text-gray-700 hover:bg-gray-50 transition-colors"
                                   role="menuitem">
                                    <i class="fas fa-chart-area mr-3 text-purple-500 group-hover:text-purple-600" aria-hidden="true"></i>
                                    <span>Análises</span>
                                </a>
                                <button onclick="generateReport()"
                                        class="group w-full flex items-center px-4 py-3 text-sm text-gray-700 hover:bg-gray-50 transition-colors text-left"
                                        role="menuitem">
                                    <i class="fas fa-file-alt mr-3 text-orange-500 group-hover:text-orange-600" aria-hidden="true"></i>
                                    <span>Relatório</span>
                                </button>
                                <a href="{{ route('barragens.edit', $barragem) }}"
                                   class="group flex items-center px-4 py-3 text-sm text-gray-700 hover:bg-gray-50 transition-colors"
                                   role="menuitem">
                                    <i class="fas fa-edit mr-3 text-gray-500 group-hover:text-gray-600" aria-hidden="true"></i>
                                    <span>Editar</span>
                                </a>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">

        <!-- Hero Card -->
        <div class="bg-gradient-to-r from-blue-600 to-blue-800 rounded-2xl shadow-xl mb-8 overflow-hidden">
            <div class="px-8 py-8">
                <div class="flex items-center justify-between">
                    <div class="flex items-center space-x-6">
                        <div class="w-20 h-20 bg-gray-800/90 rounded-2xl flex items-center justify-center backdrop-blur">
                            <i class="fas fa-dam text-white text-3xl"></i>
                        </div>
                        <div>
                            <h1 class="text-4xl font-black text-gray-900 mb-2 bg-white px-4 py-2 rounded-lg shadow-md">{{ $barragem->nome }}</h1>
                            <p class="text-blue-100 text-lg">Código: <span class="font-bold text-gray-900">{{ $barragem->codigo }}</span></p>
                        </div>
                    </div>

                    <!-- Status Indicators -->
                    <div class="flex items-center space-x-4">
                        @php
                            $estadoConfig = [
                                'activa' => ['class' => 'bg-green-500 text-white border-green-400', 'icon' => 'fas fa-check-circle', 'label' => 'ATIVA'],
                                'inactiva' => ['class' => 'bg-red-500 text-white border-red-400', 'icon' => 'fas fa-times-circle', 'label' => 'INATIVA'],
                                'manutencao' => ['class' => 'bg-yellow-500 text-white border-yellow-400', 'icon' => 'fas fa-wrench', 'label' => 'MANUTENÇÃO']
                            ];
                            $config = $estadoConfig[$barragem->estado] ?? ['class' => 'bg-gray-500 text-white border-gray-400', 'icon' => 'fas fa-question-circle', 'label' => strtoupper($barragem->estado)];
                        @endphp

                        <div class="text-right">
                            <div class="text-sm text-blue-400 font-bold">Status da Barragem</div>
                            <div class="inline-flex items-center px-3 py-1 rounded-full text-sm font-bold {{ $config['class'] }} border-2 backdrop-blur">
                                <i class="{{ $config['icon'] }} mr-2"></i>{{ $config['label'] }}
                            </div>
                        </div>

                        @if($barragem->ultima_leitura)
                        <div class="text-right">
                            <div class="text-sm text-blue-400 font-bold">Última Leitura</div>
                            <div class="text-lg font-bold text-gray-100">{{ $barragem->ultima_leitura->data_leitura->format('d/m/Y') }}</div>
                        </div>
                        @endif
                    </div>
                </div>

                <!-- Métricas Principais - Notifications Config Style -->
                @if($barragem->ultima_leitura)
                <div class="mt-8">
                    @php
                        // Cálculos para indicadores de status
                        $cotaStatus = 'normal';
                        $cotaColor = 'blue';
                        if ($barragem->cota_nmc && $barragem->ultima_leitura->cota_actual >= $barragem->cota_nmc) {
                            $cotaStatus = 'critical';
                            $cotaColor = 'red';
                        } elseif ($barragem->cota_npa && $barragem->ultima_leitura->cota_actual >= $barragem->cota_npa) {
                            $cotaStatus = 'high';
                            $cotaColor = 'yellow';
                        }

                        $volumePercentage = $barragem->ultima_leitura->percentagem_enchimento ?? 0;
                        $volumeStatus = 'normal';
                        $volumeColor = 'green';
                        if ($volumePercentage >= 90) {
                            $volumeStatus = 'critical';
                            $volumeColor = 'red';
                        } elseif ($volumePercentage >= 75) {
                            $volumeStatus = 'high';
                            $volumeColor = 'yellow';
                        } elseif ($volumePercentage <= 30) {
                            $volumeStatus = 'low';
                            $volumeColor = 'orange';
                        }

                        $diasAtras = $barragem->ultima_leitura->data_leitura->diffInDays(now());
                        $dataStatus = $diasAtras == 0 ? 'current' : ($diasAtras <= 3 ? 'recent' : 'outdated');
                        $dataColor = $diasAtras == 0 ? 'green' : ($diasAtras <= 3 ? 'yellow' : 'red');
                    @endphp

                    <div class="flex flex-row gap-4 w-full">
                        <!-- Cota Atual -->
                        <div class="flex-1 bg-white rounded-lg shadow p-6">
                            <div class="flex items-center justify-between">
                                <div>
                                    <p class="text-sm font-medium text-gray-600 mb-2">COTA ATUAL</p>
                                    <p class="text-3xl font-bold text-blue-600">{{ number_format($barragem->ultima_leitura->cota_actual, 1) }}</p>
                                    <p class="text-xs text-gray-500">metros</p>
                                </div>
                                <div class="bg-blue-100 p-3 rounded-lg">
                                    <i class="fas fa-ruler-vertical text-2xl text-blue-600"></i>
                                </div>
                            </div>
                        </div>

                        <!-- Volume -->
                        <div class="flex-1 bg-white rounded-lg shadow p-6">
                            <div class="flex items-center justify-between">
                                <div>
                                    <p class="text-sm font-medium text-gray-600 mb-2">VOLUME</p>
                                    <p class="text-3xl font-bold text-green-600">{{ $barragem->ultima_leitura->volume_actual ? number_format($barragem->ultima_leitura->volume_actual, 1) : '--' }}</p>
                                    <p class="text-xs text-gray-500">{{ $barragem->volume_maximo ? 'Máx: ' . number_format($barragem->volume_maximo, 1) . 'Mm³' : 'Mm³' }}</p>
                                </div>
                                <div class="bg-green-100 p-3 rounded-lg">
                                    <i class="fas fa-tint text-2xl text-green-600"></i>
                                </div>
                            </div>
                        </div>

                        <!-- Enchimento -->
                        <div class="flex-1 bg-white rounded-lg shadow p-6">
                            <div class="flex items-center justify-between">
                                <div>
                                    <p class="text-sm font-medium text-gray-600 mb-2">ENCHIMENTO</p>
                                    <p class="text-3xl font-bold text-{{ $volumeColor }}-600">{{ number_format($volumePercentage, 1) }}%</p>
                                    <p class="text-xs text-gray-500">
                                        @if($volumePercentage >= 90)
                                            📊 Crítico
                                        @elseif($volumePercentage >= 75)
                                            📈 Alto
                                        @elseif($volumePercentage <= 30)
                                            📉 Baixo
                                        @else
                                            ✅ Normal
                                        @endif
                                    </p>
                                </div>
                                <div class="bg-{{ $volumeColor }}-100 p-3 rounded-lg">
                                    <i class="fas fa-percentage text-2xl text-{{ $volumeColor }}-600"></i>
                                </div>
                            </div>
                        </div>

                        <!-- Atualização dos Dados -->
                        <div class="flex-1 bg-white rounded-lg shadow p-6">
                            <div class="flex items-center justify-between">
                                <div>
                                    <p class="text-sm font-medium text-gray-600 mb-2">DADOS</p>
                                    <p class="text-3xl font-bold text-{{ $dataColor }}-600">
                                        @if($diasAtras == 0)
                                            Hoje
                                        @elseif($diasAtras == 1)
                                            Ontem
                                        @else
                                            {{ $diasAtras }}d
                                        @endif
                                    </p>
                                    <p class="text-xs text-gray-500">{{ $barragem->ultima_leitura->data_leitura->format('d/m/Y') }}</p>
                                </div>
                                <div class="bg-{{ $dataColor }}-100 p-3 rounded-lg">
                                    <i class="fas fa-{{ $diasAtras == 0 ? 'wifi' : 'clock' }} text-2xl text-{{ $dataColor }}-600"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                @else
                <!-- No data state -->
                <div class="mt-8 bg-white/95 backdrop-blur rounded-xl p-8 border border-gray-300 text-center shadow-lg"
                     role="alert"
                     aria-live="polite">
                    <div class="w-16 h-16 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-4">
                        <i class="fas fa-database text-gray-600 text-2xl" aria-hidden="true"></i>
                    </div>
                    <h3 class="text-lg font-semibold text-gray-900 mb-2">Sem Dados Disponíveis</h3>
                    <p class="text-gray-600 text-sm mb-4">Nenhuma leitura foi registrada para esta barragem.</p>
                    <button onclick="addReading()"
                            class="inline-flex items-center px-4 py-2 bg-green-600 hover:bg-green-700 focus:ring-2 focus:ring-green-500 text-white rounded-lg font-medium transition-colors">
                        <i class="fas fa-plus mr-2" aria-hidden="true"></i>Adicionar Primeira Leitura
                    </button>
                </div>
                @endif
            </div>
        </div>

        <!-- Main Dashboard Grid -->
        <div class="grid grid-cols-12 gap-8">

            <!-- Left Column - Charts and Analysis -->
            <div class="col-span-12 lg:col-span-8 space-y-8">

                <!-- Interactive Chart -->
                <div class="bg-white rounded-2xl shadow-lg overflow-hidden">
                    <div class="border-b border-gray-200 px-6 py-4">
                        <div class="flex items-center justify-between">
                            <div>
                                <h2 class="text-xl font-bold text-gray-900">Análise Temporal</h2>
                                <p class="text-gray-600 text-sm">Evolução dos parâmetros ao longo do tempo</p>
                            </div>
                            <div class="flex items-center space-x-2">
                                <span class="text-xs text-gray-500 font-medium">PERÍODO:</span>
                                <button onclick="updateChartPeriod('7d')"
                                        class="period-btn px-3 py-1 bg-blue-600 text-white rounded-lg text-xs font-bold hover:bg-blue-700 transition-colors" data-period="7d">7d</button>
                                <button onclick="updateChartPeriod('30d')"
                                        class="period-btn px-3 py-1 bg-gray-200 text-gray-700 rounded-lg text-xs font-bold hover:bg-gray-300 transition-colors" data-period="30d">30d</button>
                                <button onclick="updateChartPeriod('1y')"
                                        class="period-btn px-3 py-1 bg-gray-200 text-gray-700 rounded-lg text-xs font-bold hover:bg-gray-300 transition-colors" data-period="1y">1a</button>
                            </div>
                        </div>
                    </div>
                    <div class="p-6">
                        <div class="h-80">
                            <canvas id="damChart" class="w-full h-full"></canvas>
                        </div>
                    </div>
                </div>

                <!-- Technical Data Grid - Reorganizado por Criticidade -->
                <div class="bg-white rounded-2xl shadow-lg overflow-hidden">
                    <div class="border-b border-gray-200 px-6 py-4">
                        <h2 class="text-xl font-bold text-gray-900">Especificações Técnicas</h2>
                        <p class="text-gray-600 text-sm">Informações organizadas por ordem de criticidade</p>
                    </div>
                    <div class="p-6 space-y-8">

                        <!-- SEÇÃO 1: SEGURANÇA E ALERTAS - CRÍTICO -->
                        <div class="space-y-4">
                            <div class="flex items-center mb-4">
                                <div class="flex items-center space-x-2">
                                    <span class="bg-red-500 text-white text-xs font-bold px-2 py-1 rounded-full flex items-center">
                                        <i class="fas fa-exclamation-triangle mr-1"></i>CRÍTICO
                                    </span>
                                    <h3 class="text-lg font-bold text-gray-900">Segurança e Alertas</h3>
                                </div>
                            </div>

                            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">

                                <!-- Níveis de Segurança -->
                                @php
                                    $currentLevel = $barragem->ultima_leitura->cota_actual ?? null;
                                    $statusCritico = false;
                                    $statusAlerta = false;

                                    if ($currentLevel && $barragem->cota_nmc && $currentLevel >= $barragem->cota_nmc) {
                                        $statusCritico = true;
                                    } elseif ($currentLevel && $barragem->cota_npa && $currentLevel >= $barragem->cota_npa) {
                                        $statusAlerta = true;
                                    }
                                @endphp

                                <div class="border-2 {{ $statusCritico ? 'border-red-400 bg-red-50' : ($statusAlerta ? 'border-yellow-400 bg-yellow-50' : 'border-gray-300 bg-white') }} rounded-xl p-4 hover:shadow-lg transition-all">
                                    <div class="flex items-center mb-4">
                                        <div class="w-10 h-10 {{ $statusCritico ? 'bg-red-500' : ($statusAlerta ? 'bg-yellow-500' : 'bg-gray-500') }} rounded-lg flex items-center justify-center mr-3 relative">
                                            @if($statusCritico || $statusAlerta)
                                                <div class="absolute -top-1 -right-1 w-3 h-3 bg-red-500 rounded-full animate-pulse"></div>
                                            @endif
                                            <i class="fas fa-ruler-vertical text-white text-sm"></i>
                                        </div>
                                        <div class="flex-1">
                                            <h4 class="font-bold text-gray-900 flex items-center">
                                                Níveis de Segurança
                                                @if($statusCritico)
                                                    <span class="ml-2 bg-red-500 text-white text-xs px-2 py-1 rounded-full animate-pulse">
                                                        CRÍTICO
                                                    </span>
                                                @elseif($statusAlerta)
                                                    <span class="ml-2 bg-yellow-500 text-white text-xs px-2 py-1 rounded-full">
                                                        ALERTA
                                                    </span>
                                                @endif
                                            </h4>
                                        </div>
                                    </div>

                                    <div class="space-y-3 text-sm">
                                        @if($barragem->ultima_leitura)
                                        <div class="bg-blue-100 border border-blue-200 rounded-lg p-3">
                                            <div class="flex justify-between items-center">
                                                <span class="text-blue-700 font-medium">Nível Atual:</span>
                                                <span class="font-bold text-xl {{ $statusCritico ? 'text-red-600' : ($statusAlerta ? 'text-yellow-600' : 'text-blue-600') }}">
                                                    {{ number_format($barragem->ultima_leitura->cota_actual, 1) }}m
                                                </span>
                                            </div>
                                            <div class="mt-2 bg-white rounded-full h-3 overflow-hidden">
                                                @php
                                                    $maxLevel = max($barragem->cota_maxima ?? 100, $barragem->cota_nmc ?? 100, $barragem->cota_npa ?? 100);
                                                    $percentage = ($currentLevel / $maxLevel) * 100;
                                                @endphp
                                                <div class="h-full {{ $statusCritico ? 'bg-red-500' : ($statusAlerta ? 'bg-yellow-500' : 'bg-blue-500') }} transition-all duration-300"
                                                     style="width: {{ min(100, $percentage) }}%"></div>
                                            </div>
                                        </div>
                                        @endif

                                        @if($barragem->cota_nmc)
                                        <div class="flex justify-between items-center py-2 border-b border-red-200">
                                            <span class="text-gray-600 font-medium">NMC (Nível Máximo de Cheia):</span>
                                            <div class="flex items-center space-x-2">
                                                <span class="font-bold text-red-600">{{ number_format($barragem->cota_nmc, 1) }}m</span>
                                                @if($currentLevel && $currentLevel >= $barragem->cota_nmc)
                                                    <i class="fas fa-exclamation-triangle text-red-500 animate-pulse"></i>
                                                @endif
                                            </div>
                                        </div>
                                        @endif

                                        @if($barragem->cota_npa)
                                        <div class="flex justify-between items-center py-2 border-b border-yellow-200">
                                            <span class="text-gray-600 font-medium">NPA (Nível Pleno Armazenamento):</span>
                                            <div class="flex items-center space-x-2">
                                                <span class="font-bold text-yellow-600">{{ number_format($barragem->cota_npa, 1) }}m</span>
                                                @if($currentLevel && $currentLevel >= $barragem->cota_npa && $currentLevel < ($barragem->cota_nmc ?? PHP_FLOAT_MAX))
                                                    <i class="fas fa-exclamation-triangle text-yellow-500"></i>
                                                @endif
                                            </div>
                                        </div>
                                        @endif

                                        @if($barragem->cota_nme)
                                        <div class="flex justify-between items-center py-2">
                                            <span class="text-gray-600 font-medium">NME (Nível Mínimo Exploração):</span>
                                            <span class="font-bold text-green-600">{{ number_format($barragem->cota_nme, 1) }}m</span>
                                        </div>
                                        @endif
                                    </div>
                                </div>

                                <!-- Classificação de Risco -->
                                <div class="border-2 border-gray-300 rounded-xl p-4 hover:bg-gray-50 hover:shadow-lg transition-all">
                                    <div class="flex items-center mb-4">
                                        <div class="w-10 h-10 bg-orange-500 rounded-lg flex items-center justify-center mr-3">
                                            <i class="fas fa-shield-alt text-white text-sm"></i>
                                        </div>
                                        <h4 class="font-bold text-gray-900">Classificação de Risco</h4>
                                    </div>

                                    <div class="space-y-3 text-sm">
                                        @if($barragem->classe_risco)
                                        <div class="text-center py-4">
                                            <div class="inline-flex items-center px-4 py-2 rounded-full text-lg font-bold border-2
                                                @if($barragem->classe_risco == 'alto') bg-red-100 text-red-800 border-red-300
                                                @elseif($barragem->classe_risco == 'medio') bg-yellow-100 text-yellow-800 border-yellow-300
                                                @else bg-green-100 text-green-800 border-green-300 @endif">
                                                @if($barragem->classe_risco == 'alto')
                                                    <i class="fas fa-exclamation-triangle mr-2"></i>
                                                @elseif($barragem->classe_risco == 'medio')
                                                    <i class="fas fa-exclamation-circle mr-2"></i>
                                                @else
                                                    <i class="fas fa-check-circle mr-2"></i>
                                                @endif
                                                RISCO {{ strtoupper($barragem->classe_risco) }}
                                            </div>
                                        </div>
                                        @else
                                        <div class="text-center py-4">
                                            <span class="inline-flex items-center px-4 py-2 bg-gray-100 text-gray-600 rounded-full text-sm">
                                                <i class="fas fa-question-circle mr-2"></i>Não classificado
                                            </span>
                                        </div>
                                        @endif

                                        <div class="space-y-2">
                                            @if($barragem->categoria_dano)
                                            <div class="flex justify-between">
                                                <span class="text-gray-600">Categoria de Dano:</span>
                                                <span class="font-bold text-purple-600 capitalize">{{ $barragem->categoria_dano }}</span>
                                            </div>
                                            @endif
                                            @if($barragem->frequencia_monitoramento)
                                            <div class="flex justify-between">
                                                <span class="text-gray-600">Monitoramento:</span>
                                                <span class="font-bold text-blue-600 capitalize">{{ $barragem->frequencia_monitoramento }}</span>
                                            </div>
                                            @endif
                                        </div>
                                    </div>
                                </div>

                            </div>
                        </div>

                        <!-- SEÇÃO 2: PARÂMETROS OPERACIONAIS - OPERACIONAL -->
                        <div class="space-y-4 border-t border-gray-200 pt-6">
                            <div class="flex items-center mb-4">
                                <div class="flex items-center space-x-2">
                                    <span class="bg-blue-500 text-white text-xs font-bold px-2 py-1 rounded-full flex items-center">
                                        <i class="fas fa-tachometer-alt mr-1"></i>OPERACIONAL
                                    </span>
                                    <h3 class="text-lg font-bold text-gray-900">Parâmetros Operacionais</h3>
                                </div>
                            </div>

                            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">

                                <!-- Volumes -->
                                <div class="border-2 border-blue-200 bg-blue-50 rounded-xl p-4 hover:shadow-lg transition-all">
                                    <div class="flex items-center mb-4">
                                        <div class="w-10 h-10 bg-blue-500 rounded-lg flex items-center justify-center mr-3">
                                            <i class="fas fa-tint text-white text-sm"></i>
                                        </div>
                                        <h4 class="font-bold text-gray-900">Volumes e Capacidades</h4>
                                    </div>

                                    <div class="space-y-3 text-sm">
                                        @if($barragem->ultima_leitura && $barragem->ultima_leitura->volume_actual)
                                        <div class="bg-white border border-blue-200 rounded-lg p-3">
                                            <div class="flex justify-between items-center">
                                                <span class="text-blue-700 font-medium">Volume Atual:</span>
                                                <span class="font-bold text-xl text-blue-600">{{ number_format($barragem->ultima_leitura->volume_actual, 1) }} Mm³</span>
                                            </div>
                                            @if($barragem->volume_maximo)
                                            <div class="mt-2 bg-gray-100 rounded-full h-2 overflow-hidden">
                                                @php $volumePercentage = ($barragem->ultima_leitura->volume_actual / $barragem->volume_maximo) * 100; @endphp
                                                <div class="h-full bg-blue-500 transition-all duration-300" style="width: {{ min(100, $volumePercentage) }}%"></div>
                                            </div>
                                            @endif
                                        </div>
                                        @endif

                                        @if($barragem->volume_maximo)
                                        <div class="flex justify-between py-2 border-b border-blue-200">
                                            <span class="text-gray-600">Volume Máximo:</span>
                                            <span class="font-bold text-blue-600">{{ number_format($barragem->volume_maximo, 1) }} Mm³</span>
                                        </div>
                                        @endif

                                        @if($barragem->volume_util)
                                        <div class="flex justify-between py-2 border-b border-blue-200">
                                            <span class="text-gray-600">Volume Útil:</span>
                                            <span class="font-bold text-green-600">{{ number_format($barragem->volume_util, 1) }} Mm³</span>
                                        </div>
                                        @endif

                                        @if($barragem->volume_morto)
                                        <div class="flex justify-between py-2">
                                            <span class="text-gray-600">Volume Morto:</span>
                                            <span class="font-bold text-gray-600">{{ number_format($barragem->volume_morto, 1) }} Mm³</span>
                                        </div>
                                        @endif
                                    </div>
                                </div>

                                <!-- Cotas Operacionais -->
                                <div class="border-2 border-green-200 bg-green-50 rounded-xl p-4 hover:shadow-lg transition-all">
                                    <div class="flex items-center mb-4">
                                        <div class="w-10 h-10 bg-green-500 rounded-lg flex items-center justify-center mr-3">
                                            <i class="fas fa-arrows-alt-v text-white text-sm"></i>
                                        </div>
                                        <h4 class="font-bold text-gray-900">Cotas Operacionais</h4>
                                    </div>

                                    <div class="space-y-3 text-sm">
                                        @if($barragem->cota_maxima)
                                        <div class="flex justify-between py-2 border-b border-green-200">
                                            <span class="text-gray-600">Cota Máxima:</span>
                                            <span class="font-bold text-green-600">{{ number_format($barragem->cota_maxima, 1) }}m</span>
                                        </div>
                                        @endif

                                        @if($barragem->cota_minima)
                                        <div class="flex justify-between py-2 border-b border-green-200">
                                            <span class="text-gray-600">Cota Mínima:</span>
                                            <span class="font-bold text-green-600">{{ number_format($barragem->cota_minima, 1) }}m</span>
                                        </div>
                                        @endif

                                        @if($barragem->cota_maxima && $barragem->cota_minima)
                                        <div class="bg-white border border-green-200 rounded-lg p-3">
                                            <div class="flex justify-between items-center mb-2">
                                                <span class="text-green-700 font-medium">Amplitude:</span>
                                                <span class="font-bold text-green-600">{{ number_format($barragem->cota_maxima - $barragem->cota_minima, 1) }}m</span>
                                            </div>
                                            @if($barragem->ultima_leitura)
                                            <div class="bg-gray-100 rounded-full h-2">
                                                @php $cotaPercent = (($barragem->ultima_leitura->cota_actual - $barragem->cota_minima) / ($barragem->cota_maxima - $barragem->cota_minima)) * 100; @endphp
                                                <div class="h-full bg-green-500 rounded-full transition-all" style="width: {{ max(0, min(100, $cotaPercent)) }}%"></div>
                                            </div>
                                            @endif
                                        </div>
                                        @endif
                                    </div>
                                </div>

                                <!-- Capacidades -->
                                <div class="border-2 border-purple-200 bg-purple-50 rounded-xl p-4 hover:shadow-lg transition-all">
                                    <div class="flex items-center mb-4">
                                        <div class="w-10 h-10 bg-purple-500 rounded-lg flex items-center justify-center mr-3">
                                            <i class="fas fa-database text-white text-sm"></i>
                                        </div>
                                        <h4 class="font-bold text-gray-900"><i class="fas fa-database text-purple-600 mr-2"></i>Capacidades</h4>
                                    </div>

                                    <div class="space-y-3 text-sm">
                                        @if($barragem->capacidade_turbinamento)
                                        <div class="flex justify-between py-2 border-b border-purple-200">
                                            <span class="text-gray-600">Turbinamento:</span>
                                            <span class="font-bold text-purple-600">{{ number_format($barragem->capacidade_turbinamento, 1) }} m³/s</span>
                                        </div>
                                        @endif

                                        @if($barragem->capacidade_vertedouro)
                                        <div class="flex justify-between py-2 border-b border-purple-200">
                                            <span class="text-gray-600">Vertedouro:</span>
                                            <span class="font-bold text-purple-600">{{ number_format($barragem->capacidade_vertedouro, 1) }} m³/s</span>
                                        </div>
                                        @endif

                                        @if($barragem->capacidade_descarga)
                                        <div class="flex justify-between py-2">
                                            <span class="text-gray-600">Descarga:</span>
                                            <span class="font-bold text-purple-600">{{ number_format($barragem->capacidade_descarga, 1) }} m³/s</span>
                                        </div>
                                        @endif

                                        @if(!$barragem->capacidade_turbinamento && !$barragem->capacidade_vertedouro && !$barragem->capacidade_descarga)
                                        <div class="text-center py-4">
                                            <span class="text-gray-500 text-sm">Dados não disponíveis</span>
                                        </div>
                                        @endif
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- SEÇÃO 3: CARACTERÍSTICAS TÉCNICAS - TÉCNICO -->
                        <div class="space-y-4 border-t border-gray-200 pt-6">
                            <div class="flex items-center mb-4">
                                <div class="flex items-center space-x-2">
                                    <span class="bg-gray-600 text-white text-xs font-bold px-2 py-1 rounded-full flex items-center">
                                        <i class="fas fa-cog mr-1"></i>TÉCNICO
                                    </span>
                                    <h3 class="text-lg font-bold text-gray-900">Características Técnicas</h3>
                                </div>
                            </div>

                            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">

                                <!-- Características Físicas -->
                                <div class="border border-gray-300 rounded-xl p-4 hover:bg-gray-50 hover:shadow-lg transition-all">
                                    <div class="flex items-center mb-4">
                                        <div class="w-10 h-10 bg-purple-600 rounded-lg flex items-center justify-center mr-3">
                                            <i class="fas fa-ruler-combined text-white text-sm"></i>
                                        </div>
                                        <h4 class="font-bold text-gray-900"><i class="fas fa-ruler-combined text-purple-600 mr-2"></i>Dimensões Físicas</h4>
                                    </div>

                                    <div class="space-y-3 text-sm">
                                        @if($barragem->altura_barragem)
                                        <div class="flex justify-between py-2 border-b border-gray-200">
                                            <span class="text-gray-600">Altura da Barragem:</span>
                                            <span class="font-bold text-gray-900">{{ number_format($barragem->altura_barragem, 1) }}m</span>
                                        </div>
                                        @endif

                                        @if($barragem->comprimento_crista)
                                        <div class="flex justify-between py-2 border-b border-gray-200">
                                            <span class="text-gray-600">Comprimento da Crista:</span>
                                            <span class="font-bold text-gray-900">{{ number_format($barragem->comprimento_crista, 0) }}m</span>
                                        </div>
                                        @endif

                                        @if($barragem->largura_base)
                                        <div class="flex justify-between py-2">
                                            <span class="text-gray-600">Largura da Base:</span>
                                            <span class="font-bold text-gray-900">{{ number_format($barragem->largura_base, 1) }}m</span>
                                        </div>
                                        @endif
                                    </div>
                                </div>

                                <!-- Tipo e Construção -->
                                <div class="border border-gray-300 rounded-xl p-4 hover:bg-gray-50 hover:shadow-lg transition-all">
                                    <div class="flex items-center mb-4">
                                        <div class="w-10 h-10 bg-amber-600 rounded-lg flex items-center justify-center mr-3">
                                            <i class="fas fa-industry text-white text-sm"></i>
                                        </div>
                                        <h4 class="font-bold text-gray-900"><i class="fas fa-industry text-amber-600 mr-2"></i>Tipo e Construção</h4>
                                    </div>

                                    <div class="space-y-3 text-sm">
                                        @if($barragem->tipo_barragem)
                                        <div class="bg-indigo-50 border border-indigo-200 rounded-lg p-3 text-center">
                                            <span class="text-indigo-700 text-xs font-medium block mb-1">Tipo de Barragem</span>
                                            <span class="font-bold text-indigo-900 text-lg capitalize">{{ $barragem->tipo_barragem }}</span>
                                        </div>
                                        @endif

                                        @if($barragem->material_construcao)
                                        <div class="flex justify-between py-2 border-b border-gray-200">
                                            <span class="text-gray-600">Material:</span>
                                            <span class="font-bold text-gray-900 capitalize">{{ $barragem->material_construcao }}</span>
                                        </div>
                                        @endif

                                        @if($barragem->ano_construcao)
                                        <div class="flex justify-between py-2">
                                            <span class="text-gray-600">Ano de Construção:</span>
                                            <span class="font-bold text-gray-900">{{ $barragem->ano_construcao }}</span>
                                        </div>
                                        @endif
                                    </div>
                                </div>

                                <!-- Equipamentos -->
                                <div class="border border-gray-300 rounded-xl p-4 hover:bg-gray-50 hover:shadow-lg transition-all">
                                    <div class="flex items-center mb-4">
                                        <div class="w-10 h-10 bg-orange-600 rounded-lg flex items-center justify-center mr-3">
                                            <i class="fas fa-cog text-white text-sm"></i>
                                        </div>
                                        <h4 class="font-bold text-gray-900"><i class="fas fa-cog text-orange-600 mr-2"></i>Equipamentos</h4>
                                    </div>

                                    <div class="space-y-3 text-sm">
                                        @if($barragem->num_comportas)
                                        <div class="flex justify-between py-2 border-b border-gray-200">
                                            <span class="text-gray-600">Número de Comportas:</span>
                                            <span class="font-bold text-gray-900">{{ $barragem->num_comportas }}</span>
                                        </div>
                                        @endif

                                        @if($barragem->tipo_vertedouro)
                                        <div class="flex justify-between py-2 border-b border-gray-200">
                                            <span class="text-gray-600">Tipo Vertedouro:</span>
                                            <span class="font-bold text-gray-900 capitalize">{{ $barragem->tipo_vertedouro }}</span>
                                        </div>
                                        @endif

                                        @if($barragem->sistema_medicao)
                                        <div class="bg-teal-50 border border-teal-200 rounded-lg p-2 text-center">
                                            <span class="text-teal-700 text-xs font-medium">Sistema de Medição</span>
                                            <div class="font-bold text-teal-900 capitalize">{{ $barragem->sistema_medicao }}</div>
                                        </div>
                                        @endif
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- SEÇÃO 4: INFORMAÇÕES ADMINISTRATIVAS - ADMINISTRATIVO -->
                        <div class="space-y-4 border-t border-gray-200 pt-6">
                            <div class="flex items-center mb-4">
                                <div class="flex items-center space-x-2">
                                    <span class="bg-green-600 text-white text-xs font-bold px-2 py-1 rounded-full flex items-center">
                                        <i class="fas fa-file-alt mr-1"></i>ADMINISTRATIVO
                                    </span>
                                    <h3 class="text-lg font-bold text-gray-900">Informações Administrativas</h3>
                                </div>
                            </div>

                            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">

                                <!-- Localização -->
                                <div class="border border-green-300 rounded-xl p-4 hover:bg-green-50 hover:shadow-lg transition-all">
                                    <div class="flex items-center mb-4">
                                        <div class="w-10 h-10 bg-green-500 rounded-lg flex items-center justify-center mr-3">
                                            <i class="fas fa-map-marker-alt text-white text-sm"></i>
                                        </div>
                                        <h4 class="font-bold text-gray-900">Localização</h4>
                                    </div>

                                    <div class="space-y-3 text-sm">
                                        @if($barragem->provincia)
                                        <div class="flex justify-between py-2 border-b border-green-200">
                                            <span class="text-gray-600">Província:</span>
                                            <span class="font-bold text-green-700">{{ $barragem->provincia }}</span>
                                        </div>
                                        @endif

                                        @if($barragem->distrito)
                                        <div class="flex justify-between py-2 border-b border-green-200">
                                            <span class="text-gray-600">Distrito:</span>
                                            <span class="font-bold text-green-700">{{ $barragem->distrito }}</span>
                                        </div>
                                        @endif

                                        @if($barragem->bacia_hidrografica)
                                        <div class="flex justify-between py-2 border-b border-green-200">
                                            <span class="text-gray-600">Bacia Hidrográfica:</span>
                                            <span class="font-bold text-green-700">{{ $barragem->bacia_hidrografica->nome }}</span>
                                        </div>
                                        @endif

                                        @if($barragem->latitude && $barragem->longitude)
                                        <div class="mt-3 pt-3 border-t border-green-200">
                                            <x-coordenadas-clicaveis
                                                :latitude="$barragem->latitude"
                                                :longitude="$barragem->longitude"
                                                :nome="$barragem->nome"
                                                tipo="barragem"
                                                :id="$barragem->id"
                                                size="xs" />
                                        </div>
                                        @endif
                                    </div>
                                </div>

                                <!-- Projeto -->
                                @if($barragem->empresa_projetista || $barragem->empresa_construtora || $barragem->ano_projeto)
                                <div class="border border-green-300 rounded-xl p-4 hover:bg-green-50 hover:shadow-lg transition-all">
                                    <div class="flex items-center mb-4">
                                        <div class="w-10 h-10 bg-emerald-600 rounded-lg flex items-center justify-center mr-3">
                                            <i class="fas fa-drafting-compass text-white text-sm"></i>
                                        </div>
                                        <h4 class="font-bold text-gray-900">Dados do Projeto</h4>
                                    </div>

                                    <div class="space-y-3 text-sm">
                                        @if($barragem->empresa_projetista)
                                        <div class="border-b border-green-200 pb-2">
                                            <span class="text-gray-600 text-xs block mb-1">Empresa Projetista:</span>
                                            <span class="font-bold text-green-700">{{ $barragem->empresa_projetista }}</span>
                                        </div>
                                        @endif

                                        @if($barragem->empresa_construtora)
                                        <div class="border-b border-green-200 pb-2">
                                            <span class="text-gray-600 text-xs block mb-1">Empresa Construtora:</span>
                                            <span class="font-bold text-green-700">{{ $barragem->empresa_construtora }}</span>
                                        </div>
                                        @endif

                                        @if($barragem->ano_projeto)
                                        <div class="flex justify-between">
                                            <span class="text-gray-600">Ano do Projeto:</span>
                                            <span class="font-bold text-green-700">{{ $barragem->ano_projeto }}</span>
                                        </div>
                                        @endif
                                    </div>
                                </div>
                                @endif

                                <!-- Gestão e Responsabilidade -->
                                <div class="border border-green-300 rounded-xl p-4 hover:bg-green-50 hover:shadow-lg transition-all">
                                    <div class="flex items-center mb-4">
                                        <div class="w-10 h-10 bg-cyan-600 rounded-lg flex items-center justify-center mr-3">
                                            <i class="fas fa-user-tie text-white text-sm"></i>
                                        </div>
                                        <h4 class="font-bold text-gray-900"><i class="fas fa-user-tie text-cyan-600 mr-2"></i>Gestão e Responsabilidade</h4>
                                    </div>

                                    <div class="space-y-3 text-sm">
                                        @if($barragem->proprietario)
                                        <div class="border-b border-green-200 pb-2">
                                            <span class="text-gray-600 text-xs block mb-1">Proprietário:</span>
                                            <span class="font-bold text-green-700">{{ $barragem->proprietario }}</span>
                                        </div>
                                        @endif

                                        @if($barragem->operador)
                                        <div class="border-b border-green-200 pb-2">
                                            <span class="text-gray-600 text-xs block mb-1">Operador:</span>
                                            <span class="font-bold text-green-700">{{ $barragem->operador }}</span>
                                        </div>
                                        @endif

                                        @php
                                            $statusConfig = [
                                                'activa' => ['class' => 'bg-green-50 border-green-200 text-green-900', 'icon' => 'fas fa-check-circle', 'label' => 'Operacional'],
                                                'inactiva' => ['class' => 'bg-red-50 border-red-200 text-red-900', 'icon' => 'fas fa-times-circle', 'label' => 'Fora de Operação'],
                                                'manutencao' => ['class' => 'bg-yellow-50 border-yellow-200 text-yellow-900', 'icon' => 'fas fa-wrench', 'label' => 'Em Manutenção']
                                            ];
                                            $config = $statusConfig[$barragem->estado] ?? ['class' => 'bg-gray-50 border-gray-200 text-gray-900', 'icon' => 'fas fa-question-circle', 'label' => ucfirst($barragem->estado)];
                                        @endphp
                                        <div class="{{ $config['class'] }} border rounded-lg p-2 text-center">
                                            <span class="text-xs font-bold block mb-1 text-black">Status Operacional</span>
                                            <div class="font-bold flex items-center justify-center text-black">
                                                <i class="{{ $config['icon'] }} mr-1"></i>
                                                {{ $config['label'] }}
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Right Sidebar -->
            <div class="col-span-12 lg:col-span-4 space-y-6">

                @if($barragem->ultima_leitura)
                <!-- Current Status -->
                <div class="bg-white rounded-2xl shadow-lg overflow-hidden">
                    <div class="bg-gradient-to-r from-green-600 to-emerald-600 px-6 py-4 text-white">
                        <h2 class="text-xl font-bold text-gray-900">
                            <i class="fas fa-tachometer-alt mr-2"></i>Status Operacional
                        </h2>
                    </div>
                    <div class="p-6">
                        @if($barragem->ultima_leitura->caudal_afluente || $barragem->ultima_leitura->caudal_efluente)
                        <div class="mb-6">
                            <h4 class="font-bold text-gray-900 mb-3 flex items-center">
                                <i class="fas fa-water mr-2 text-blue-500"></i>Caudais (m³/s)
                            </h4>
                            <div class="space-y-2">
                                @if($barragem->ultima_leitura->caudal_afluente)
                                <div class="flex justify-between items-center py-2 border-b border-gray-100">
                                    <span class="text-gray-600">Afluente:</span>
                                    <span class="font-bold text-lg text-blue-600">{{ number_format($barragem->ultima_leitura->caudal_afluente, 1) }}</span>
                                </div>
                                @endif
                                @if($barragem->ultima_leitura->caudal_efluente)
                                <div class="flex justify-between items-center py-2 border-b border-gray-100">
                                    <span class="text-gray-600">Efluente:</span>
                                    <span class="font-bold text-lg text-green-600">{{ number_format($barragem->ultima_leitura->caudal_efluente, 1) }}</span>
                                </div>
                                @endif
                            </div>
                        </div>
                        @endif

                        <div class="mb-6">
                            <h4 class="font-bold text-gray-900 mb-3 flex items-center">
                                <i class="fas fa-certificate mr-2 text-purple-500"></i>Qualidade dos Dados
                            </h4>
                            <div class="space-y-3">
                                <div class="flex justify-between items-center">
                                    <span class="text-gray-600">Método:</span>
                                    <span class="px-2 py-1 bg-blue-100 text-blue-800 rounded-lg text-sm font-medium">
                                        {{ ucfirst($barragem->ultima_leitura->metodo_leitura ?? 'Manual') }}
                                    </span>
                                </div>
                                <div class="flex justify-between items-center">
                                    <span class="text-gray-600">Atualização:</span>
                                    <span class="text-sm font-medium text-gray-900">
                                        {{ $barragem->ultima_leitura->data_leitura->format('d/m/Y H:i') }}
                                    </span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                @endif

                <!-- Quick Tools -->
                <div class="bg-white rounded-2xl shadow-lg overflow-hidden relative z-10">
                    <div class="bg-gradient-to-r from-blue-600 to-indigo-600 px-6 py-4 text-white">
                        <h3 class="text-lg font-bold flex items-center">
                            <i class="fas fa-tools mr-2"></i>Ferramentas Rápidas
                        </h3>
                    </div>
                    <div class="p-6 bg-white">
                        <div class="space-y-3">
                            <button onclick="addReading()"
                                    class="w-full bg-green-600 hover:bg-green-700 text-white rounded-xl py-4 px-4 font-bold transition-all shadow-md hover:shadow-lg flex items-center justify-center border border-green-600">
                                <i class="fas fa-plus-circle mr-3 text-white"></i>Adicionar Leitura
                            </button>
                            <a href="{{ route('barragens.leituras', $barragem) }}"
                               class="w-full block bg-blue-600 hover:bg-blue-700 text-white rounded-xl py-4 px-4 font-bold transition-all shadow-md hover:shadow-lg text-center border border-blue-600">
                                <i class="fas fa-list mr-3 text-white"></i>Ver Histórico
                            </a>
                            <button onclick="generateReport()"
                                    class="w-full bg-orange-600 hover:bg-orange-700 text-white rounded-xl py-4 px-4 font-bold transition-all shadow-md hover:shadow-lg flex items-center justify-center border border-orange-600"
                                    style="background-color: #ea580c !important; color: white !important;">
                                <i class="fas fa-file-pdf mr-3" style="color: white !important;"></i>
                                <span style="color: white !important;">Gerar Relatório</span>
                            </button>
                            <a href="{{ route('barragens.curvas-volume.index', $barragem) }}"
                               class="w-full block bg-purple-600 hover:bg-purple-700 text-white rounded-xl py-4 px-4 font-bold transition-all shadow-md hover:shadow-lg text-center border border-purple-600"
                               style="background-color: #9333ea !important; color: white !important; text-decoration: none;">
                                <i class="fas fa-chart-area mr-3" style="color: white !important;"></i>
                                <span style="color: white !important;">Curvas Volume</span>
                            </a>
                        </div>
                    </div>
                </div>

                <!-- System Information -->
                <div class="bg-white rounded-2xl shadow-lg overflow-hidden">
                    <div class="bg-gradient-to-r from-gray-600 to-slate-600 px-6 py-4 text-white">
                        <h2 class="text-xl font-bold text-gray-900">
                            <i class="fas fa-info-circle mr-2"></i>Informações do Sistema
                        </h2>
                    </div>
                    <div class="p-6">
                        @if($barragem->observacoes)
                        <div class="mb-4 p-4 bg-yellow-50 border-l-4 border-yellow-400 rounded-r-lg">
                            <h4 class="font-bold text-yellow-800 mb-2">Observações:</h4>
                            <p class="text-sm text-yellow-700">{{ $barragem->observacoes }}</p>
                        </div>
                        @endif

                        <div class="space-y-3 text-sm">
                            <div class="flex justify-between py-2 border-b border-gray-100">
                                <span class="text-gray-600">Criado:</span>
                                <span class="font-medium">{{ $barragem->created_at->format('d/m/Y') }}</span>
                            </div>
                            <div class="flex justify-between py-2">
                                <span class="text-gray-600">Atualizado:</span>
                                <span class="font-medium">{{ $barragem->updated_at->format('d/m/Y') }}</span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
let damChart = null;
let currentPeriod = '7d';

function generateReport() {
    alert('Funcionalidade de relatório será implementada em breve.');
}

function addReading() {
    window.location.href = '{{ route("barragens.leituras", $barragem) }}';
}

// Chart functions
function initializeDamChart() {
    const ctx = document.getElementById('damChart').getContext('2d');

    damChart = new Chart(ctx, {
        type: 'line',
        data: { labels: [], datasets: [] },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    position: 'bottom',
                    labels: { usePointStyle: true, padding: 20, font: { weight: 'bold' } }
                },
                tooltip: {
                    backgroundColor: 'rgba(0, 0, 0, 0.8)',
                    titleFont: { weight: 'bold' },
                    bodyFont: { weight: 'bold' },
                    cornerRadius: 8
                }
            },
            scales: {
                y: {
                    beginAtZero: false,
                    grid: { color: 'rgba(0, 0, 0, 0.1)' },
                    ticks: { font: { weight: 'bold' } },
                    title: { display: true, text: 'Cota (m)', color: 'rgb(59, 130, 246)', font: { weight: 'bold' } }
                },
                y1: {
                    type: 'linear', display: true, position: 'right',
                    grid: { drawOnChartArea: false },
                    ticks: { font: { weight: 'bold' } },
                    title: { display: true, text: 'Volume (Mm³)', color: 'rgb(34, 197, 94)', font: { weight: 'bold' } }
                },
                y2: { type: 'linear', display: false, min: 0, max: 100 },
                x: { grid: { color: 'rgba(0, 0, 0, 0.1)' }, ticks: { font: { weight: 'bold' } } }
            },
            elements: { line: { tension: 0.4 }, point: { radius: 4, hoverRadius: 8 } }
        }
    });

    updateChartData(currentPeriod);
}

function updateChartData(period) {
    const url = `/api/barragens/{{ $barragem->id }}/chart-data?period=${period}`;
    const chartContainer = document.getElementById('damChart').parentElement;

    // Skeleton loader melhorado
    chartContainer.innerHTML = `
        <div class="h-80 flex flex-col items-center justify-center space-y-4 bg-gray-50 rounded-lg">
            <div class="flex items-center space-x-2">
                <div class="animate-spin rounded-full h-8 w-8 border-3 border-blue-500 border-t-transparent"></div>
                <div class="text-gray-600 font-medium">Carregando dados do gráfico...</div>
            </div>
            <div class="w-full max-w-md">
                <div class="animate-pulse space-y-2">
                    <div class="h-2 bg-gray-300 rounded-full w-3/4"></div>
                    <div class="h-2 bg-gray-300 rounded-full w-1/2"></div>
                    <div class="h-2 bg-gray-300 rounded-full w-5/6"></div>
                </div>
            </div>
        </div>
    `;

    fetch(url)
        .then(response => response.json())
        .then(data => {
            chartContainer.innerHTML = '<canvas id="damChart" class="w-full h-full"></canvas>';
            const ctx = document.getElementById('damChart').getContext('2d');

            if (damChart) damChart.destroy();

            damChart = new Chart(ctx, {
                type: 'line',
                data: { labels: data.labels || [], datasets: data.datasets || [] },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        title: {
                            display: true,
                            text: `BARRAGEM ${data.barragem_nome ? data.barragem_nome.toUpperCase() : '{{ strtoupper($barragem->nome) }}'}`,
                            font: { size: 18, weight: 'bold' },
                            color: '#1f2937',
                            padding: { bottom: 30 }
                        },
                        legend: {
                            position: 'bottom',
                            labels: {
                                usePointStyle: false,
                                padding: 20,
                                font: { weight: 'bold', size: 12 },
                                generateLabels: function(chart) {
                                    const labels = chart.data.datasets.map((dataset, i) => ({
                                        text: dataset.label,
                                        fillStyle: dataset.borderColor,
                                        strokeStyle: dataset.borderColor,
                                        lineWidth: dataset.type === 'bar' ? 0 : 3,
                                        hidden: !chart.isDatasetVisible(i),
                                        datasetIndex: i
                                    }));
                                    return labels;
                                }
                            }
                        },
                        tooltip: {
                            backgroundColor: 'rgba(0, 0, 0, 0.8)',
                            titleFont: { weight: 'bold' },
                            bodyFont: { weight: 'bold' },
                            cornerRadius: 8,
                            callbacks: {
                                label: function(context) {
                                    let label = context.dataset.label || '';
                                    if (label) {
                                        label += ': ';
                                    }
                                    if (label.includes('Precipitação')) {
                                        label += context.parsed.y.toFixed(1) + ' mm';
                                    } else if (label.includes('Volume')) {
                                        label += context.parsed.y.toFixed(2) + ' Mm³';
                                    } else {
                                        label += context.parsed.y.toFixed(2) + ' m';
                                    }
                                    return label;
                                }
                            }
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: false,
                            grid: {
                                color: 'rgba(0, 0, 0, 0.3)',
                                lineWidth: 1,
                                drawBorder: true
                            },
                            ticks: {
                                font: { weight: 'bold', size: 11 },
                                color: '#374151',
                                stepSize: period === '1y' ? 0.5 : undefined,
                                callback: function(value) {
                                    return value.toFixed(1);
                                }
                            },
                            title: {
                                display: true,
                                text: 'Cota (m)',
                                color: '#1f2937',
                                font: { weight: 'bold', size: 14 },
                                padding: { bottom: 10 }
                            },
                            min: period === '1y' ? 996 : undefined,
                            max: period === '1y' ? 1002 : undefined
                        },
                        y1: {
                            type: 'linear',
                            display: period !== '1y',
                            position: 'right',
                            grid: { drawOnChartArea: false },
                            ticks: {
                                font: { weight: 'bold', size: 11 },
                                color: '#059669'
                            },
                            title: {
                                display: true,
                                text: 'Volume (Mm³)',
                                color: '#059669',
                                font: { weight: 'bold', size: 14 }
                            }
                        },
                        y2: {
                            type: 'linear',
                            display: period === '1y',
                            position: 'right',
                            min: 0,
                            max: 60,
                            reverse: true, // Inverte a escala como no gráfico original
                            grid: { drawOnChartArea: false },
                            ticks: {
                                font: { weight: 'bold', size: 11 },
                                color: '#7c3aed',
                                stepSize: 10,
                                callback: function(value) {
                                    return value.toFixed(1);
                                }
                            },
                            title: {
                                display: true,
                                text: 'Precipitação (mm)',
                                color: '#7c3aed',
                                font: { weight: 'bold', size: 14 },
                                padding: { bottom: 10 }
                            }
                        },
                        x: {
                            grid: {
                                color: 'rgba(0, 0, 0, 0.3)',
                                lineWidth: 1,
                                drawBorder: true
                            },
                            ticks: {
                                font: { weight: 'bold', size: 11 },
                                color: '#374151'
                            }
                        }
                    },
                    elements: {
                        line: { tension: 0.1 },
                        point: { radius: 4, hoverRadius: 6 },
                        bar: { borderWidth: 1 }
                    }
                }
            });
        })
        .catch(error => {
            console.error('Erro ao carregar dados do gráfico:', error);

            // Estado de erro melhorado
            chartContainer.innerHTML = `
                <div class="h-80 flex flex-col items-center justify-center space-y-4 bg-red-50 border border-red-200 rounded-lg">
                    <div class="text-center space-y-3">
                        <div class="w-16 h-16 bg-red-100 rounded-full flex items-center justify-center mx-auto">
                            <i class="fas fa-chart-line-down text-red-500 text-2xl"></i>
                        </div>
                        <h3 class="text-lg font-semibold text-red-800">Erro ao Carregar Gráfico</h3>
                        <p class="text-red-600 text-sm max-w-md">
                            Não foi possível carregar os dados do gráfico. Verifique sua conexão com a internet ou tente novamente.
                        </p>
                        <div class="flex items-center justify-center space-x-3 pt-2">
                            <button onclick="updateChartData('${period}')"
                                    class="inline-flex items-center px-4 py-2 bg-red-600 hover:bg-red-700 text-white rounded-lg font-medium transition-colors focus:ring-2 focus:ring-red-500 focus:ring-offset-2">
                                <i class="fas fa-redo mr-2"></i>Tentar Novamente
                            </button>
                            <button onclick="showErrorDetails('${error.message || 'Erro desconhecido'}')"
                                    class="inline-flex items-center px-4 py-2 bg-gray-100 hover:bg-gray-200 text-gray-700 rounded-lg font-medium transition-colors">
                                <i class="fas fa-info-circle mr-2"></i>Detalhes
                            </button>
                        </div>
                    </div>
                </div>
            `;
        });
}

function updateChartPeriod(period) {
    currentPeriod = period;

    document.querySelectorAll('.period-btn').forEach(btn => {
        btn.classList.remove('bg-blue-600', 'text-white');
        btn.classList.add('bg-gray-200', 'text-gray-700');
    });

    const selectedBtn = document.querySelector(`[data-period="${period}"]`);
    if (selectedBtn) {
        selectedBtn.classList.remove('bg-gray-200', 'text-gray-700');
        selectedBtn.classList.add('bg-blue-600', 'text-white');
    }

    updateChartData(period);
}

document.addEventListener('DOMContentLoaded', function() {
    setTimeout(() => {
        const chartCanvas = document.getElementById('damChart');
        if (chartCanvas) {
            initializeDamChart();
        }
    }, 100);
});

// Mobile Menu Functions
function toggleMobileMenu() {
    const button = document.getElementById('mobile-menu-button');
    const panel = document.getElementById('mobile-menu-panel');
    const isExpanded = button.getAttribute('aria-expanded') === 'true';

    button.setAttribute('aria-expanded', !isExpanded);
    panel.classList.toggle('hidden');

    // Add backdrop blur and outside click handler when open
    if (!isExpanded) {
        // Close menu when clicking outside
        document.addEventListener('click', closeMobileMenuOnOutsideClick);

        // Handle escape key
        document.addEventListener('keydown', closeMobileMenuOnEscape);

        // Prevent button click from closing immediately
        event.stopPropagation();
    } else {
        document.removeEventListener('click', closeMobileMenuOnOutsideClick);
        document.removeEventListener('keydown', closeMobileMenuOnEscape);
    }
}

function closeMobileMenuOnOutsideClick(event) {
    const button = document.getElementById('mobile-menu-button');
    const panel = document.getElementById('mobile-menu-panel');

    if (!button.contains(event.target) && !panel.contains(event.target)) {
        closeMobileMenu();
    }
}

function closeMobileMenuOnEscape(event) {
    if (event.key === 'Escape') {
        closeMobileMenu();
    }
}

function closeMobileMenu() {
    const button = document.getElementById('mobile-menu-button');
    const panel = document.getElementById('mobile-menu-panel');

    button.setAttribute('aria-expanded', 'false');
    panel.classList.add('hidden');

    document.removeEventListener('click', closeMobileMenuOnOutsideClick);
    document.removeEventListener('keydown', closeMobileMenuOnEscape);
}

// Funções para melhor feedback visual e tratamento de erros
function showErrorDetails(errorMessage) {
    const timestamp = new Date().toLocaleString('pt-MZ');
    const userAgent = navigator.userAgent;
    const url = window.location.href;

    const details = `
Detalhes do Erro:
━━━━━━━━━━━━━━━━━━━━━━
📅 Hora: ${timestamp}
🌐 URL: ${url}
💻 Navegador: ${userAgent}
⚠️ Erro: ${errorMessage}
━━━━━━━━━━━━━━━━━━━━━━

Para reportar este erro, envie estas informações para o suporte técnico.
    `;

    if (confirm('Detalhes do erro copiados para a área de transferência.\n\nDeseja reportar este erro?')) {
        // Copiar para clipboard se possível
        if (navigator.clipboard) {
            navigator.clipboard.writeText(details).then(() => {
                console.log('Detalhes do erro copiados para clipboard');
            });
        }

        // Abrir email para suporte (opcional)
        const mailto = `mailto:suporte@exemplo.com?subject=Erro%20no%20Gráfico%20da%20Barragem&body=${encodeURIComponent(details)}`;
        window.open(mailto);
    }
}

function addButtonLoadingState(button, originalText = null) {
    if (!originalText) {
        originalText = button.innerHTML;
    }

    button.disabled = true;
    button.dataset.originalContent = originalText;
    button.innerHTML = `
        <svg class="animate-spin -ml-1 mr-2 h-4 w-4 text-current" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
            <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle>
            <path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
        </svg>
        Carregando...
    `;

    return originalText;
}

function removeButtonLoadingState(button) {
    if (button.dataset.originalContent) {
        button.innerHTML = button.dataset.originalContent;
        button.disabled = false;
        delete button.dataset.originalContent;
    }
}
</script>
@endsection