@extends('layouts.app')

@section('title', 'Gerar Boletim')

@section('content')
<div class="space-y-6">
    <!-- Header -->
    <div class="bg-white rounded-lg shadow p-6">
        <div class="flex items-center justify-between">
            <div>
                <h1 class="text-2xl font-bold text-gray-800">
                    <i class="fas fa-file-plus mr-3 text-blue-600"></i>
                    Gerar Novo Boletim
                </h1>
                <p class="text-gray-600 mt-2">Selecione o tipo de boletim e período para geração</p>
            </div>
            <a href="{{ route('boletins.index') }}" class="bg-gray-600 hover:bg-gray-700 text-white px-4 py-2 rounded-lg">
                <i class="fas fa-arrow-left mr-2"></i>
                Voltar
            </a>
        </div>
    </div>

    <!-- Messages -->
    @if(session('success'))
        <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-6">
            {{ session('success') }}
        </div>
    @endif

    @if(session('error'))
        <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-6">
            {{ session('error') }}
        </div>
    @endif

    @if($errors->any())
        <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-6">
            <ul class="list-disc list-inside">
                @foreach($errors->all() as $error)
                    <li>{{ $error }}</li>
                @endforeach
            </ul>
        </div>
    @endif

    <!-- Form -->
    <div class="bg-white rounded-lg shadow p-6">
        <form method="POST" action="{{ route('boletins.store') }}" class="space-y-6">
            @csrf
            
            <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                <!-- Tipo de Boletim -->
                <div>
                    <label for="tipo" class="block text-sm font-medium text-gray-700 mb-2">
                        Tipo de Boletim <span class="text-red-500">*</span>
                    </label>
                    <select id="tipo" name="tipo" required 
                            class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 @error('tipo') border-red-500 @enderror">
                        <option value="">Selecione o tipo</option>
                        <option value="hidrologico_regional" {{ old('tipo') == 'hidrologico_regional' ? 'selected' : '' }}>
                            Boletim Hidrológico Regional
                        </option>
                        <option value="hidrologico_provincial" {{ old('tipo') == 'hidrologico_provincial' ? 'selected' : '' }}>
                            Boletim Hidrológico da Província
                        </option>
                        <option value="informativo_barragens" {{ old('tipo') == 'informativo_barragens' ? 'selected' : '' }}>
                            Boletim Informativo de Barragens
                        </option>
                        <option value="diario" {{ old('tipo') == 'diario' ? 'selected' : '' }}>
                            Boletim Diário
                        </option>
                        <option value="semanal" {{ old('tipo') == 'semanal' ? 'selected' : '' }}>
                            Boletim Semanal
                        </option>
                        <option value="mensal" {{ old('tipo') == 'mensal' ? 'selected' : '' }}>
                            Boletim Mensal
                        </option>
                        <option value="emergencia" {{ old('tipo') == 'emergencia' ? 'selected' : '' }}>
                            Boletim de Emergência
                        </option>
                    </select>
                    @error('tipo')
                        <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                    @enderror
                </div>

                <!-- Período -->
                <div id="periodo-container" class="hidden">
                    <label class="block text-sm font-medium text-gray-700 mb-2">
                        Período de Dados
                    </label>
                    <div class="grid grid-cols-2 gap-2">
                        <div>
                            <label for="data_inicio" class="block text-xs text-gray-500 mb-1">Data Início</label>
                            <input type="date" id="data_inicio" name="data_inicio" value="{{ old('data_inicio') }}"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 @error('data_inicio') border-red-500 @enderror">
                            @error('data_inicio')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>
                        <div>
                            <label for="data_fim" class="block text-xs text-gray-500 mb-1">Data Fim</label>
                            <input type="date" id="data_fim" name="data_fim" value="{{ old('data_fim') }}"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 @error('data_fim') border-red-500 @enderror">
                            @error('data_fim')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>
                    </div>
                </div>
            </div>

            <!-- Descrição dos Tipos -->
            <div class="bg-gray-50 rounded-lg p-4">
                <h3 class="text-sm font-medium text-gray-800 mb-3">Tipos de Boletim:</h3>
                <div class="space-y-2 text-sm text-gray-600">
                    <div class="boletim-desc" data-tipo="hidrologico_regional">
                        <strong>Hidrológico Regional:</strong> Relatório completo com dados de todas as barragens e estações da região, análise de tendências e previsões hidrológicas.
                    </div>
                    <div class="boletim-desc" data-tipo="hidrologico_provincial">
                        <strong>Hidrológico da Província:</strong> Boletim específico da Província de Nampula com dados das principais barragens e situação hidrológica local.
                    </div>
                    <div class="boletim-desc" data-tipo="informativo_barragens">
                        <strong>Informativo de Barragens:</strong> Situação atual das barragens com níveis, volumes e alertas ativos.
                    </div>
                    <div class="boletim-desc" data-tipo="diario">
                        <strong>Diário:</strong> Resumo diário das condições hidrológicas e meteorológicas.
                    </div>
                    <div class="boletim-desc" data-tipo="semanal">
                        <strong>Semanal:</strong> Análise semanal com tendências e comparações.
                    </div>
                    <div class="boletim-desc" data-tipo="mensal">
                        <strong>Mensal:</strong> Relatório mensal completo com estatísticas e análises detalhadas.
                    </div>
                    <div class="boletim-desc" data-tipo="emergencia">
                        <strong>Emergência:</strong> Boletim especial para situações de emergência hidrológica.
                    </div>
                </div>
            </div>

            <!-- Informações de Geração -->
            <div class="bg-blue-50 border border-blue-200 rounded-lg p-4">
                <div class="flex">
                    <i class="fas fa-info-circle text-blue-600 mr-3 mt-1"></i>
                    <div class="text-sm text-blue-800">
                        <h4 class="font-medium mb-2">Informações sobre a Geração:</h4>
                        <ul class="space-y-1 list-disc list-inside">
                            <li>Os dados serão coletados automaticamente das barragens e estações</li>
                            <li>O boletim será gerado em formato PDF, Word e Excel</li>
                            <li>Após a geração, você poderá revisar e editar antes da aprovação</li>
                            <li>Boletins aprovados podem ser publicados e distribuídos</li>
                        </ul>
                    </div>
                </div>
            </div>

            <!-- Botões -->
            <div class="flex justify-end space-x-3">
                <a href="{{ route('boletins.index') }}" 
                   class="bg-gray-300 hover:bg-gray-400 text-gray-800 px-6 py-2 rounded-lg">
                    Cancelar
                </a>
                <button type="submit" 
                        class="bg-blue-600 hover:bg-blue-700 text-white px-6 py-2 rounded-lg flex items-center">
                    <i class="fas fa-file-plus mr-2"></i>
                    Gerar Boletim
                </button>
            </div>
        </form>
    </div>
</div>

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', function() {
    const tipoSelect = document.getElementById('tipo');
    const periodoContainer = document.getElementById('periodo-container');
    const dataInicio = document.getElementById('data_inicio');
    const dataFim = document.getElementById('data_fim');
    const descriptions = document.querySelectorAll('.boletim-desc');

    function updatePeriodo() {
        const tipo = tipoSelect.value;
        
        // Show/hide period fields based on type
        if (tipo === 'hidrologico_regional' || tipo === 'hidrologico_provincial' || tipo === 'semanal' || tipo === 'mensal') {
            periodoContainer.classList.remove('hidden');
            
            // Set default dates based on type
            const today = new Date();
            let startDate, endDate;
            
            switch(tipo) {
                case 'semanal':
                    startDate = new Date(today);
                    startDate.setDate(today.getDate() - 7);
                    endDate = today;
                    break;
                case 'mensal':
                    startDate = new Date(today.getFullYear(), today.getMonth() - 1, 1);
                    endDate = new Date(today.getFullYear(), today.getMonth(), 0);
                    break;
                default: // hidrologico_regional
                    startDate = new Date(today);
                    startDate.setDate(today.getDate() - 7);
                    endDate = today;
            }
            
            dataInicio.value = startDate.toISOString().split('T')[0];
            dataFim.value = endDate.toISOString().split('T')[0];
        } else {
            periodoContainer.classList.add('hidden');
            dataInicio.value = '';
            dataFim.value = '';
        }
        
        // Highlight relevant description
        descriptions.forEach(desc => {
            if (desc.dataset.tipo === tipo) {
                desc.style.backgroundColor = '#EBF8FF';
                desc.style.border = '1px solid #3182CE';
                desc.style.borderRadius = '0.375rem';
                desc.style.padding = '0.5rem';
            } else {
                desc.style.backgroundColor = '';
                desc.style.border = '';
                desc.style.borderRadius = '';
                desc.style.padding = '';
            }
        });
    }

    tipoSelect.addEventListener('change', updatePeriodo);
    
    // Initial setup
    if (tipoSelect.value) {
        updatePeriodo();
    }
});
</script>
@endpush
@endsection