import openpyxl
from openpyxl.utils import get_column_letter

def analyze_ui_interface():
    """Analisar a interface do usuário e formas de entrada de dados"""
    wb = openpyxl.load_workbook('ficheiros/Cópia de BARRAGENS ACT 8.8.xlsm', data_only=False)

    print('🖥️  ANÁLISE DA INTERFACE DO USUÁRIO E ENTRADA DE DADOS')
    print('=' * 70)

    # Verificar todas as folhas para identificar áreas de entrada
    all_sheets = wb.sheetnames
    
    print('🔍 Procurando por áreas de entrada de dados e controles...')
    print()
    
    # Analisar folhas em busca de padrões de entrada
    for sheet_name in all_sheets:
        ws = wb[sheet_name]
        print(f'📊 FOLHA: {sheet_name}')
        print('-' * 40)
        
        # Procurar por células que parecem ser entrada manual (não fórmulas)
        entry_points = []
        data_ranges = []
        
        # Analisar as primeiras linhas para padrões de cabeçalho e entrada
        for row in range(1, min(50, ws.max_row + 1)):
            for col in range(1, min(50, ws.max_column + 1)):
                cell = ws.cell(row=row, column=col)
                
                if cell.value:
                    cell_str = str(cell.value)
                    
                    # Identificar possíveis pontos de entrada baseados em texto
                    entry_indicators = ['data:', 'inserir', 'entrada', 'input', 'novo', 'adicionar']
                    if any(indicator in cell_str.lower() for indicator in entry_indicators):
                        entry_points.append(f'   {get_column_letter(col)}{row}: {cell.value}')
                    
                    # Identificar células com datas recentes (última entrada)
                    if ('2023' in cell_str or '2024' in cell_str) and row < 20:
                        data_ranges.append(f'   {get_column_letter(col)}{row}: {cell.value}')
        
        # Analisar padrão de dados (onde provavelmente inserem novos dados)
        print('🎯 Possíveis pontos de entrada:')
        if entry_points:
            for point in entry_points[:3]:
                print(point)
        else:
            # Se não encontrou pontos óbvios, analisar padrão dos dados
            # Verificar se há uma área onde os dados param (fim dos dados atuais)
            last_data_row = 0
            for row in range(ws.max_row, 0, -1):
                has_data = False
                for col in range(1, min(10, ws.max_column + 1)):
                    if ws.cell(row=row, column=col).value:
                        has_data = True
                        break
                if has_data:
                    last_data_row = row
                    break
            
            if last_data_row > 5:  # Se há dados substanciais
                print(f'   Última linha com dados: {last_data_row}')
                print(f'   Próxima entrada seria provavelmente na linha: {last_data_row + 1}')
                
                # Mostrar como seria uma nova entrada
                sample_row = []
                for col in range(1, min(8, ws.max_column + 1)):
                    cell = ws.cell(row=last_data_row, column=col)
                    if cell.value:
                        if str(cell.value).startswith('='):
                            sample_row.append('FÓRMULA')
                        else:
                            sample_row.append(str(cell.value)[:15])
                    else:
                        sample_row.append('vazio')
                
                print(f'   Padrão da última entrada: {" | ".join(sample_row)}')
            else:
                print('   Nenhum padrão claro de entrada identificado')
        
        print()
        
        # Limitar saída para não sobrecarregar
        if sheet_name in ['NAMPULA', 'NACALA', 'MUGICA', 'CHIPEMBE', 'LOCUMUE', 'MITUCUE', 'GRAFICOS DE BARRAGEM']:
            continue
        else:
            # Para outras folhas, dar uma olhada mais rápida
            continue

def analyze_workflow_pattern():
    """Analisar o padrão de workflow"""
    wb = openpyxl.load_workbook('ficheiros/Cópia de BARRAGENS ACT 8.8.xlsm', data_only=False)

    print('🔄 ANÁLISE DO FLUXO DE TRABALHO')
    print('=' * 50)
    
    # Analisar como os dados fluem entre folhas
    print('📈 Padrão identificado baseado na análise:')
    print()
    print('1. ESTRUTURA DOS DADOS:')
    print('   • Cada barragem tem sua própria folha (NAMPULA, NACALA, etc.)')
    print('   • Dados organizados cronologicamente (coluna A = datas)')
    print('   • Métricas principais: NPA, NME, V.Morto, Nível, Cota, Precipitação')
    print()
    
    print('2. ENTRADA DE DADOS:')
    print('   • Dados inseridos manualmente nas colunas específicas')
    print('   • Data na coluna A (gerada automaticamente com fórmulas =A+1)')
    print('   • Valores de leitura nas colunas C, D, E, F (NPA, NME, Volume, etc.)')
    print('   • Fórmulas automáticas para calcular datas sequenciais')
    print()
    
    print('3. GERAÇÃO DE GRÁFICOS:')
    print('   • Folha "GRAFICOS DE BARRAGEM" contém 7 gráficos')
    print('   • 5 gráficos de barras + 2 gráficos de linha')
    print('   • Gráficos referenciam dados das folhas das barragens')
    print('   • Atualização automática quando novos dados são inseridos')
    print()
    
    print('4. FLUXO DE TRABALHO TÍPICO:')
    print('   ➤ Usuário abre folha da barragem específica')
    print('   ➤ Vai para a última linha com dados')
    print('   ➤ Insere novos valores nas colunas C, D, E, F, etc.')
    print('   ➤ Data é gerada automaticamente (fórmula)')
    print('   ➤ Gráficos na folha "GRAFICOS DE BARRAGEM" atualizam automaticamente')

if __name__ == "__main__":
    try:
        analyze_ui_interface()
        print()
        analyze_workflow_pattern()
    except Exception as e:
        print(f'❌ Erro: {e}')
        import traceback
        traceback.print_exc()
