<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\DivisaoAdministrativa;
use Illuminate\Http\Request;

/**
 * Controller para gestao de Divisoes Administrativas
 */
class DivisaoController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'permission:configurar-sistema']);
    }

    /**
     * Listar todas as divisoes
     */
    public function index(Request $request)
    {
        $query = DivisaoAdministrativa::with('parent')
            ->withCount(['children', 'users', 'barragens', 'estacoes']);

        // Filtro por tipo
        if ($request->filled('tipo')) {
            $query->where('tipo', $request->tipo);
        }

        // Filtro por provincia (parent)
        if ($request->filled('provincia_id')) {
            $query->where('parent_id', $request->provincia_id);
        }

        // Pesquisa
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('nome', 'like', "%{$search}%")
                  ->orWhere('codigo', 'like', "%{$search}%");
            });
        }

        // Ordenacao
        $query->orderBy('tipo')->orderBy('nome');

        $divisoes = $query->paginate(20);
        $provincias = DivisaoAdministrativa::provincias()->orderBy('nome')->get();

        return view('admin.divisoes.index', compact('divisoes', 'provincias'));
    }

    /**
     * Formulario de criacao
     */
    public function create()
    {
        $provincias = DivisaoAdministrativa::provincias()->orderBy('nome')->get();

        return view('admin.divisoes.create', compact('provincias'));
    }

    /**
     * Criar nova divisao
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'nome' => 'required|string|max:100',
            'codigo' => 'required|string|max:20|unique:divisoes_administrativas,codigo',
            'tipo' => 'required|in:provincia,distrito,municipio,setor,zona',
            'parent_id' => 'nullable|exists:divisoes_administrativas,id',
            'area_km2' => 'nullable|numeric|min:0',
            'populacao' => 'nullable|integer|min:0',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180',
            'descricao' => 'nullable|string|max:500',
            'ativo' => 'boolean',
        ]);

        $divisao = DivisaoAdministrativa::create([
            'nome' => $validated['nome'],
            'codigo' => strtoupper($validated['codigo']),
            'tipo' => $validated['tipo'],
            'parent_id' => $validated['parent_id'] ?? null,
            'area_km2' => $validated['area_km2'] ?? null,
            'populacao' => $validated['populacao'] ?? null,
            'latitude' => $validated['latitude'] ?? null,
            'longitude' => $validated['longitude'] ?? null,
            'descricao' => $validated['descricao'] ?? null,
            'ativo' => $request->boolean('ativo', true),
        ]);

        return redirect()
            ->route('admin.admin.divisoes-admin.show', $divisao)
            ->with('success', "Divisao '{$divisao->nome}' criada com sucesso!");
    }

    /**
     * Ver detalhes
     */
    public function show(DivisaoAdministrativa $divisao)
    {
        $divisao->load(['parent', 'children', 'users', 'barragens', 'estacoes']);

        return view('admin.divisoes.show', compact('divisao'));
    }

    /**
     * Formulario de edicao
     */
    public function edit(DivisaoAdministrativa $divisao)
    {
        $provincias = DivisaoAdministrativa::provincias()
            ->where('id', '!=', $divisao->id)
            ->orderBy('nome')
            ->get();

        return view('admin.divisoes.edit', compact('divisao', 'provincias'));
    }

    /**
     * Actualizar divisao
     */
    public function update(Request $request, DivisaoAdministrativa $divisao)
    {
        $validated = $request->validate([
            'nome' => 'required|string|max:100',
            'codigo' => 'required|string|max:20|unique:divisoes_administrativas,codigo,' . $divisao->id,
            'tipo' => 'required|in:provincia,distrito,municipio,setor,zona',
            'parent_id' => 'nullable|exists:divisoes_administrativas,id',
            'area_km2' => 'nullable|numeric|min:0',
            'populacao' => 'nullable|integer|min:0',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180',
            'descricao' => 'nullable|string|max:500',
            'ativo' => 'boolean',
        ]);

        // Nao permitir que seja pai de si mesmo
        if ($validated['parent_id'] == $divisao->id) {
            return back()->with('error', 'Uma divisao nao pode ser pai de si mesma.');
        }

        $divisao->update([
            'nome' => $validated['nome'],
            'codigo' => strtoupper($validated['codigo']),
            'tipo' => $validated['tipo'],
            'parent_id' => $validated['parent_id'] ?? null,
            'area_km2' => $validated['area_km2'] ?? null,
            'populacao' => $validated['populacao'] ?? null,
            'latitude' => $validated['latitude'] ?? null,
            'longitude' => $validated['longitude'] ?? null,
            'descricao' => $validated['descricao'] ?? null,
            'ativo' => $request->boolean('ativo', true),
        ]);

        return redirect()
            ->route('admin.admin.divisoes-admin.show', $divisao)
            ->with('success', "Divisao '{$divisao->nome}' actualizada com sucesso!");
    }

    /**
     * Eliminar divisao
     */
    public function destroy(DivisaoAdministrativa $divisao)
    {
        // Verificar se tem subdivisoes
        if ($divisao->children()->count() > 0) {
            return back()->with('error', 'Esta divisao tem subdivisoes. Elimine-as primeiro.');
        }

        // Verificar se tem utilizadores
        if ($divisao->users()->count() > 0) {
            return back()->with('error', 'Esta divisao tem utilizadores atribuidos. Remova-os primeiro.');
        }

        // Verificar se tem barragens
        if ($divisao->barragens()->count() > 0) {
            return back()->with('error', 'Esta divisao tem barragens associadas.');
        }

        $nome = $divisao->nome;
        $divisao->delete();

        return redirect()
            ->route('admin.admin.divisoes-admin.index')
            ->with('success', "Divisao '{$nome}' eliminada com sucesso!");
    }

    /**
     * API para obter subdivisoes (AJAX)
     */
    public function getSubdivisoes(DivisaoAdministrativa $divisao)
    {
        $subdivisoes = $divisao->children()
            ->ativo()
            ->orderBy('nome')
            ->get(['id', 'nome', 'codigo', 'tipo']);

        return response()->json($subdivisoes);
    }
}
