<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\UserDivisao;
use App\Models\DivisaoAdministrativa;
use App\Services\RegionalAccessService;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Permission;

/**
 * Controller para gestao de atribuicoes de divisoes a utilizadores
 */
class UserDivisaoController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'permission:gerir-usuarios']);
    }

    /**
     * Listar todas as atribuicoes de divisoes
     */
    public function index()
    {
        $users = User::with(['roles', 'divisoesAtivas.divisao'])
            ->has('divisoesAtivas')
            ->orderBy('name')
            ->paginate(20);

        return view('admin.usuarios.divisoes.index', compact('users'));
    }

    /**
     * Ver atribuicoes de divisoes de um utilizador
     */
    public function show(User $user)
    {
        $user->load(['roles', 'todasDivisoes.divisao']);

        $divisoes = DivisaoAdministrativa::ativo()
            ->orderBy('tipo')
            ->orderBy('nome')
            ->get()
            ->groupBy('tipo');

        $permissions = Permission::orderBy('name')->get();

        return view('admin.usuarios.divisoes.show', compact('user', 'divisoes', 'permissions'));
    }

    /**
     * Atribuir divisao a utilizador
     */
    public function store(Request $request, User $user)
    {
        $validated = $request->validate([
            'divisao_administrativa_id' => 'required|exists:divisoes_administrativas,id',
            'nivel_acesso' => 'required|in:full,direct',
            'is_primary' => 'boolean',
            'permissoes_grant' => 'nullable|array',
            'permissoes_grant.*' => 'exists:permissions,name',
            'permissoes_revoke' => 'nullable|array',
            'permissoes_revoke.*' => 'exists:permissions,name',
            'valido_de' => 'nullable|date',
            'valido_ate' => 'nullable|date|after_or_equal:valido_de',
        ]);

        // Verificar se ja existe
        if ($user->divisoes()->where('divisao_administrativa_id', $validated['divisao_administrativa_id'])->exists()) {
            return back()->with('error', 'Esta divisao ja esta atribuida a este utilizador.');
        }

        // Se marcar como primaria, desmarcar outras
        if ($request->boolean('is_primary')) {
            UserDivisao::where('user_id', $user->id)->update(['is_primary' => false]);
        }

        // Montar permissoes customizadas
        $permissoesCustomizadas = null;
        if (!empty($validated['permissoes_grant']) || !empty($validated['permissoes_revoke'])) {
            $permissoesCustomizadas = [
                'grant' => $validated['permissoes_grant'] ?? [],
                'revoke' => $validated['permissoes_revoke'] ?? [],
            ];
        }

        $user->divisoes()->attach($validated['divisao_administrativa_id'], [
            'nivel_acesso' => $validated['nivel_acesso'],
            'is_primary' => $request->boolean('is_primary'),
            'permissoes_customizadas' => $permissoesCustomizadas,
            'valido_de' => $validated['valido_de'] ?? null,
            'valido_ate' => $validated['valido_ate'] ?? null,
            'ativo' => true,
        ]);

        // Limpar cache
        RegionalAccessService::clearUserCache($user->id);

        return redirect()
            ->route('admin.usuarios.divisoes.show', $user)
            ->with('success', 'Divisao atribuida com sucesso!');
    }

    /**
     * Actualizar atribuicao de divisao
     */
    public function update(Request $request, User $user, UserDivisao $divisao)
    {
        $validated = $request->validate([
            'nivel_acesso' => 'required|in:full,direct',
            'is_primary' => 'boolean',
            'permissoes_grant' => 'nullable|array',
            'permissoes_grant.*' => 'exists:permissions,name',
            'permissoes_revoke' => 'nullable|array',
            'permissoes_revoke.*' => 'exists:permissions,name',
            'valido_de' => 'nullable|date',
            'valido_ate' => 'nullable|date|after_or_equal:valido_de',
            'ativo' => 'boolean',
        ]);

        // Se marcar como primaria, desmarcar outras
        if ($request->boolean('is_primary')) {
            UserDivisao::where('user_id', $user->id)
                ->where('id', '!=', $divisao->id)
                ->update(['is_primary' => false]);
        }

        // Montar permissoes customizadas
        $permissoesCustomizadas = null;
        if (!empty($validated['permissoes_grant']) || !empty($validated['permissoes_revoke'])) {
            $permissoesCustomizadas = [
                'grant' => $validated['permissoes_grant'] ?? [],
                'revoke' => $validated['permissoes_revoke'] ?? [],
            ];
        }

        $divisao->update([
            'nivel_acesso' => $validated['nivel_acesso'],
            'is_primary' => $request->boolean('is_primary'),
            'permissoes_customizadas' => $permissoesCustomizadas,
            'valido_de' => $validated['valido_de'] ?? null,
            'valido_ate' => $validated['valido_ate'] ?? null,
            'ativo' => $request->boolean('ativo', true),
        ]);

        // Limpar cache
        RegionalAccessService::clearUserCache($user->id);

        return redirect()
            ->route('admin.usuarios.divisoes.show', $user)
            ->with('success', 'Atribuicao actualizada com sucesso!');
    }

    /**
     * Remover atribuicao de divisao
     */
    public function destroy(User $user, UserDivisao $divisao)
    {
        $divisao->delete();

        // Limpar cache
        RegionalAccessService::clearUserCache($user->id);

        return redirect()
            ->route('admin.usuarios.divisoes.show', $user)
            ->with('success', 'Atribuicao removida com sucesso!');
    }

    /**
     * Atribuicao em lote
     */
    public function bulkAssign(Request $request)
    {
        $validated = $request->validate([
            'user_ids' => 'required|array|min:1',
            'user_ids.*' => 'exists:users,id',
            'divisao_administrativa_id' => 'required|exists:divisoes_administrativas,id',
            'nivel_acesso' => 'required|in:full,direct',
        ]);

        $attached = 0;
        foreach ($validated['user_ids'] as $userId) {
            $user = User::find($userId);

            if (!$user->divisoes()->where('divisao_administrativa_id', $validated['divisao_administrativa_id'])->exists()) {
                $user->divisoes()->attach($validated['divisao_administrativa_id'], [
                    'nivel_acesso' => $validated['nivel_acesso'],
                    'ativo' => true,
                ]);
                $attached++;

                RegionalAccessService::clearUserCache($userId);
            }
        }

        return redirect()
            ->back()
            ->with('success', "Divisao atribuida a {$attached} utilizador(es) com sucesso!");
    }
}
