<?php

namespace App\Http\Controllers;

use App\Models\DivisaoAdministrativa;
use App\Models\AreaJurisdicao;
use App\Models\BaciaHidrografica;
use App\Models\Estacao;
use App\Models\Barragem;
use App\Models\LeituraBarragem;
use App\Models\LeituraEstacao;
use Illuminate\Http\Request;
use Carbon\Carbon;

class ContextoHidrologicoController extends Controller
{
    /**
     * 4. CRIAR FILTROS PARA CHAMAR INFORMAÇÕES SOLICITADAS
     * E CRIAR UM CONTEXTO PARA O SISTEMA HIDROLÓGICO
     */
    public function index(Request $request)
    {
        // Filtros disponíveis
        $divisoes = DivisaoAdministrativa::ativo()->orderBy('tipo')->orderBy('nome')->get()->groupBy('tipo');
        $areas = AreaJurisdicao::ativo()->orderBy('nome')->get();
        $bacias = BaciaHidrografica::orderBy('nome')->get();

        // Aplicar filtros
        $filtros = [
            'divisao_id' => $request->get('divisao_id'),
            'area_id' => $request->get('area_id'),
            'bacia_id' => $request->get('bacia_id'),
            'tipo_divisao' => $request->get('tipo_divisao'),
            'data_inicio' => $request->get('data_inicio', Carbon::now()->subMonth()->toDateString()),
            'data_fim' => $request->get('data_fim', Carbon::now()->toDateString()),
        ];

        // Construir contexto baseado nos filtros
        $contexto = $this->construirContexto($filtros);

        return view('contexto.index', compact('divisoes', 'areas', 'bacias', 'filtros', 'contexto'));
    }

    /**
     * Construir contexto hidrológico baseado nos filtros
     */
    private function construirContexto($filtros)
    {
        $query_bacias = BaciaHidrografica::query();
        $query_estacoes = Estacao::query();
        $query_barragens = Barragem::query();

        // Aplicar filtro de divisão administrativa
        if ($filtros['divisao_id']) {
            $query_bacias->where('divisao_administrativa_id', $filtros['divisao_id']);
            $query_estacoes->where('divisao_administrativa_id', $filtros['divisao_id']);
            $query_barragens->where('divisao_administrativa_id', $filtros['divisao_id']);
        }

        // Aplicar filtro de área de jurisdição
        if ($filtros['area_id']) {
            $query_bacias->where('area_jurisdicao_id', $filtros['area_id']);
            $query_estacoes->whereHas('bacia_hidrografica', function($q) use ($filtros) {
                $q->where('area_jurisdicao_id', $filtros['area_id']);
            });
            $query_barragens->whereHas('bacia_hidrografica', function($q) use ($filtros) {
                $q->where('area_jurisdicao_id', $filtros['area_id']);
            });
        }

        // Aplicar filtro de bacia
        if ($filtros['bacia_id']) {
            $query_estacoes->where('bacia_hidrografica_id', $filtros['bacia_id']);
            $query_barragens->where('bacia_hidrografica_id', $filtros['bacia_id']);
            $query_bacias->where('id', $filtros['bacia_id']);
        }

        // Aplicar filtro de tipo de divisão
        if ($filtros['tipo_divisao']) {
            $divisoes_ids = DivisaoAdministrativa::where('tipo', $filtros['tipo_divisao'])->pluck('id');
            $query_bacias->whereIn('divisao_administrativa_id', $divisoes_ids);
            $query_estacoes->whereIn('divisao_administrativa_id', $divisoes_ids);
            $query_barragens->whereIn('divisao_administrativa_id', $divisoes_ids);
        }

        // Obter resultados
        $bacias = $query_bacias->with(['divisaoAdministrativa', 'areaJurisdicao'])->get();
        $estacoes = $query_estacoes->with(['bacia_hidrografica', 'divisaoAdministrativa'])->get();
        $barragens = $query_barragens->with(['bacia_hidrografica', 'divisaoAdministrativa'])->get();

        // Obter leituras do período
        $estacoes_ids = $estacoes->pluck('id');
        $barragens_ids = $barragens->pluck('id');

        $leituras_estacoes = LeituraEstacao::whereIn('estacao_id', $estacoes_ids)
                                          ->whereBetween('data_leitura', [$filtros['data_inicio'], $filtros['data_fim']])
                                          ->count();

        $leituras_barragens = LeituraBarragem::whereIn('barragem_id', $barragens_ids)
                                            ->whereBetween('data_leitura', [$filtros['data_inicio'], $filtros['data_fim']])
                                            ->count();

        // Estatísticas do contexto
        $stats = [
            'total_bacias' => $bacias->count(),
            'total_estacoes' => $estacoes->count(),
            'total_barragens' => $barragens->count(),
            'area_total_km2' => $bacias->sum('area_km2'),
            'leituras_periodo' => $leituras_estacoes + $leituras_barragens,
            'estacoes_por_tipo' => $estacoes->groupBy('tipo')->map->count(),
            'barragens_ativas' => $barragens->where('estado', 'activa')->count(),
        ];

        return [
            'bacias' => $bacias,
            'estacoes' => $estacoes,
            'barragens' => $barragens,
            'stats' => $stats,
        ];
    }

    /**
     * Exportar dados do contexto
     */
    public function exportar(Request $request)
    {
        $filtros = [
            'divisao_id' => $request->get('divisao_id'),
            'area_id' => $request->get('area_id'),
            'bacia_id' => $request->get('bacia_id'),
            'tipo_divisao' => $request->get('tipo_divisao'),
            'data_inicio' => $request->get('data_inicio', Carbon::now()->subMonth()->toDateString()),
            'data_fim' => $request->get('data_fim', Carbon::now()->toDateString()),
        ];

        $contexto = $this->construirContexto($filtros);

        // Retornar JSON para download
        return response()->json($contexto, 200, [
            'Content-Type' => 'application/json',
            'Content-Disposition' => 'attachment; filename="contexto_hidrologico_' . date('Y-m-d') . '.json"',
        ]);
    }

    /**
     * API para obter dados filtrados dinamicamente
     */
    public function api(Request $request)
    {
        $filtros = [
            'divisao_id' => $request->get('divisao_id'),
            'area_id' => $request->get('area_id'),
            'bacia_id' => $request->get('bacia_id'),
            'tipo_divisao' => $request->get('tipo_divisao'),
            'data_inicio' => $request->get('data_inicio', Carbon::now()->subMonth()->toDateString()),
            'data_fim' => $request->get('data_fim', Carbon::now()->toDateString()),
        ];

        $contexto = $this->construirContexto($filtros);

        return response()->json($contexto);
    }

    /**
     * Obter divisões filhas de uma divisão pai
     */
    public function getSubdivisoes($parentId)
    {
        $subdivisoes = DivisaoAdministrativa::where('parent_id', $parentId)
                                           ->ativo()
                                           ->orderBy('nome')
                                           ->get();

        return response()->json($subdivisoes);
    }

    /**
     * Obter bacias de uma divisão administrativa
     */
    public function getBaciasPorDivisao($divisaoId)
    {
        $bacias = BaciaHidrografica::where('divisao_administrativa_id', $divisaoId)
                                  ->orderBy('nome')
                                  ->get();

        return response()->json($bacias);
    }
}
