<?php

namespace App\Http\Controllers;

use App\Models\Barragem;
use App\Models\CurvaCotaVolume;
use App\Models\CurvaDescarga;
use Illuminate\Http\Request;

class CorrespondenciaCurvasController extends Controller
{
    /**
     * Exibir interface de gestão de correspondências de curvas
     */
    public function index()
    {
        $barragens = Barragem::with(['curvas_cota_volume'])->get();

        return view('curvas.correspondencias', compact('barragens'));
    }

    /**
     * Obter dados de correspondência para uma barragem específica
     */
    public function getDadosBarragem($barragemId)
    {
        $barragem = Barragem::with(['curvas_cota_volume'])->findOrFail($barragemId);

        $curvasCotaVolume = $barragem->curvas_cota_volume()
            ->orderBy('cota', 'asc')
            ->get()
            ->map(function ($curva) {
                return [
                    'id' => $curva->id,
                    'cota' => $curva->cota,
                    'volume' => $curva->volume,
                    'area_espelho_agua' => $curva->area_espelho_agua,
                ];
            });

        return response()->json([
            'barragem' => [
                'id' => $barragem->id,
                'nome' => $barragem->nome,
                'cota_minima' => $barragem->cota_minima,
                'cota_maxima' => $barragem->cota_maxima,
                'volume_maximo' => $barragem->volume_maximo,
            ],
            'curvas_cota_volume' => $curvasCotaVolume,
        ]);
    }

    /**
     * Calcular correspondências entre cota e volume
     */
    public function calcularCorrespondencias(Request $request)
    {
        $request->validate([
            'barragem_id' => 'required|exists:barragens,id',
            'tipo_calculo' => 'required|in:cota_para_volume,volume_para_cota',
            'valor' => 'required|numeric',
        ]);

        $barragem = Barragem::findOrFail($request->barragem_id);
        $tipo = $request->tipo_calculo;
        $valor = $request->valor;

        if ($tipo === 'cota_para_volume') {
            $volume = $this->calcularVolumePorCota($barragem, $valor);
            $percentagem = $barragem->volume_maximo > 0
                ? ($volume / $barragem->volume_maximo) * 100
                : 0;

            return response()->json([
                'sucesso' => true,
                'tipo' => 'cota_para_volume',
                'cota_entrada' => $valor,
                'volume_calculado' => round($volume, 2),
                'percentagem_enchimento' => round($percentagem, 2),
                'unidade_volume' => 'Mm³',
            ]);
        } else {
            $cota = $this->calcularCotaPorVolume($barragem, $valor);
            $percentagem = $barragem->volume_maximo > 0
                ? ($valor / $barragem->volume_maximo) * 100
                : 0;

            return response()->json([
                'sucesso' => true,
                'tipo' => 'volume_para_cota',
                'volume_entrada' => $valor,
                'cota_calculada' => round($cota, 2),
                'percentagem_enchimento' => round($percentagem, 2),
                'unidade_cota' => 'm',
            ]);
        }
    }

    /**
     * Calcular volume baseado na cota usando interpolação linear
     */
    private function calcularVolumePorCota(Barragem $barragem, $cota)
    {
        $curva_inferior = $barragem->curvas_cota_volume()
            ->where('cota', '<=', $cota)
            ->orderBy('cota', 'desc')
            ->first();

        $curva_superior = $barragem->curvas_cota_volume()
            ->where('cota', '>', $cota)
            ->orderBy('cota', 'asc')
            ->first();

        // Se não houver curva inferior, retornar volume mínimo
        if (!$curva_inferior) {
            return $barragem->volume_morto ?? 0;
        }

        // Se não houver curva superior, retornar volume da última curva
        if (!$curva_superior) {
            return $curva_inferior->volume;
        }

        // Interpolação linear entre dois pontos
        $delta_cota = $curva_superior->cota - $curva_inferior->cota;
        $delta_volume = $curva_superior->volume - $curva_inferior->volume;
        $proporcao = ($cota - $curva_inferior->cota) / $delta_cota;

        return $curva_inferior->volume + ($delta_volume * $proporcao);
    }

    /**
     * Calcular cota baseada no volume usando interpolação linear inversa
     */
    private function calcularCotaPorVolume(Barragem $barragem, $volume)
    {
        $curva_inferior = $barragem->curvas_cota_volume()
            ->where('volume', '<=', $volume)
            ->orderBy('volume', 'desc')
            ->first();

        $curva_superior = $barragem->curvas_cota_volume()
            ->where('volume', '>', $volume)
            ->orderBy('volume', 'asc')
            ->first();

        // Se não houver curva inferior, retornar cota mínima
        if (!$curva_inferior) {
            return $barragem->cota_minima ?? 0;
        }

        // Se não houver curva superior, retornar cota da última curva
        if (!$curva_superior) {
            return $curva_inferior->cota;
        }

        // Interpolação linear inversa
        $delta_volume = $curva_superior->volume - $curva_inferior->volume;
        $delta_cota = $curva_superior->cota - $curva_inferior->cota;
        $proporcao = ($volume - $curva_inferior->volume) / $delta_volume;

        return $curva_inferior->cota + ($delta_cota * $proporcao);
    }

    /**
     * Gerar tabela de correspondências completa
     */
    public function gerarTabelaCorrespondencias($barragemId)
    {
        $barragem = Barragem::findOrFail($barragemId);

        $curvas = $barragem->curvas_cota_volume()
            ->orderBy('cota', 'asc')
            ->get();

        if ($curvas->isEmpty()) {
            return response()->json([
                'sucesso' => false,
                'mensagem' => 'Nenhuma curva cota-volume cadastrada para esta barragem',
            ]);
        }

        $tabela = $curvas->map(function ($curva) use ($barragem) {
            $percentagem = $barragem->volume_maximo > 0
                ? ($curva->volume / $barragem->volume_maximo) * 100
                : 0;

            return [
                'cota' => round($curva->cota, 2),
                'volume' => round($curva->volume, 2),
                'area_espelho_agua' => round($curva->area_espelho_agua ?? 0, 2),
                'percentagem_enchimento' => round($percentagem, 2),
            ];
        });

        return response()->json([
            'sucesso' => true,
            'barragem' => [
                'id' => $barragem->id,
                'nome' => $barragem->nome,
                'volume_maximo' => $barragem->volume_maximo,
            ],
            'tabela_correspondencias' => $tabela,
            'total_pontos' => $curvas->count(),
        ]);
    }

    /**
     * Validar consistência das curvas
     */
    public function validarCurvas($barragemId)
    {
        $barragem = Barragem::findOrFail($barragemId);

        $curvas = $barragem->curvas_cota_volume()
            ->orderBy('cota', 'asc')
            ->get();

        $erros = [];
        $avisos = [];

        // Verificar se há curvas cadastradas
        if ($curvas->isEmpty()) {
            $erros[] = 'Nenhuma curva cota-volume cadastrada';
            return response()->json([
                'sucesso' => false,
                'erros' => $erros,
                'avisos' => $avisos,
            ]);
        }

        // Verificar se cotas são crescentes
        for ($i = 1; $i < $curvas->count(); $i++) {
            if ($curvas[$i]->cota <= $curvas[$i-1]->cota) {
                $erros[] = "Cota não crescente entre pontos {$i} e ".($i+1);
            }
        }

        // Verificar se volumes são crescentes
        for ($i = 1; $i < $curvas->count(); $i++) {
            if ($curvas[$i]->volume < $curvas[$i-1]->volume) {
                $erros[] = "Volume decrescente entre pontos {$i} e ".($i+1);
            }
        }

        // Verificar cobertura de cotas
        $primeira_cota = $curvas->first()->cota;
        $ultima_cota = $curvas->last()->cota;

        if ($barragem->cota_minima && $primeira_cota > $barragem->cota_minima) {
            $avisos[] = "Primeira cota da curva ({$primeira_cota}m) é maior que cota mínima da barragem ({$barragem->cota_minima}m)";
        }

        if ($barragem->cota_maxima && $ultima_cota < $barragem->cota_maxima) {
            $avisos[] = "Última cota da curva ({$ultima_cota}m) é menor que cota máxima da barragem ({$barragem->cota_maxima}m)";
        }

        // Verificar volume máximo
        $ultimo_volume = $curvas->last()->volume;
        if ($barragem->volume_maximo && abs($ultimo_volume - $barragem->volume_maximo) > 0.1) {
            $avisos[] = "Volume da última curva ({$ultimo_volume} Mm³) difere do volume máximo da barragem ({$barragem->volume_maximo} Mm³)";
        }

        return response()->json([
            'sucesso' => count($erros) === 0,
            'erros' => $erros,
            'avisos' => $avisos,
            'total_pontos' => $curvas->count(),
            'intervalo_cotas' => [
                'minima' => $primeira_cota,
                'maxima' => $ultima_cota,
            ],
            'intervalo_volumes' => [
                'minimo' => $curvas->first()->volume,
                'maximo' => $ultimo_volume,
            ],
        ]);
    }
}
