<?php

namespace App\Http\Controllers;

use App\Models\DivisaoAdministrativa;
use App\Models\AreaJurisdicao;
use App\Models\BaciaHidrografica;
use App\Models\Estacao;
use App\Models\Barragem;
use Illuminate\Http\Request;

class JurisdicaoController extends Controller
{
    /**
     * 1. DELIMITAR ÁREA DE JURISDIÇÃO POR DIVISÕES
     */
    public function divisoes()
    {
        $divisoes = DivisaoAdministrativa::with('parent')
                                        ->ativo()
                                        ->orderBy('tipo')
                                        ->orderBy('nome')
                                        ->get()
                                        ->groupBy('tipo');

        $stats = [
            'provincias' => DivisaoAdministrativa::tipo('provincia')->ativo()->count(),
            'distritos' => DivisaoAdministrativa::tipo('distrito')->ativo()->count(),
            'municipios' => DivisaoAdministrativa::tipo('municipio')->ativo()->count(),
            'setores' => DivisaoAdministrativa::tipo('setor')->ativo()->count(),
            'zonas' => DivisaoAdministrativa::tipo('zona')->ativo()->count(),
        ];

        return view('jurisdicao.divisoes.index', compact('divisoes', 'stats'));
    }

    public function divisoesCreate()
    {
        $pais = DivisaoAdministrativa::orderBy('nome')->get();
        return view('jurisdicao.divisoes.create', compact('pais'));
    }

    public function divisoesStore(Request $request)
    {
        $validated = $request->validate([
            'nome' => 'required|string|max:255',
            'codigo' => 'required|string|unique:divisoes_administrativas',
            'tipo' => 'required|in:provincia,distrito,municipio,setor,zona',
            'parent_id' => 'nullable|exists:divisoes_administrativas,id',
            'area_km2' => 'nullable|numeric',
            'populacao' => 'nullable|integer',
            'descricao' => 'nullable|string',
            'latitude' => 'nullable|numeric',
            'longitude' => 'nullable|numeric',
        ]);

        DivisaoAdministrativa::create($validated);

        return redirect()->route('jurisdicao.divisoes')
                        ->with('success', 'Divisão administrativa criada com sucesso!');
    }

    /**
     * 2. CRIAR MAPA DA ÁREA DE JURISDIÇÃO
     */
    public function mapa()
    {
        $areas = AreaJurisdicao::ativo()->with('bacias')->get();

        // Carregar todas as bacias com suas coordenadas
        $bacias = BaciaHidrografica::whereNotNull('coordenadas_geograficas')
                                  ->with('divisaoAdministrativa')
                                  ->get();

        $estacoes = Estacao::whereNotNull('latitude')
                          ->whereNotNull('longitude')
                          ->with('divisaoAdministrativa')
                          ->get();

        $barragens = Barragem::whereNotNull('latitude')
                            ->whereNotNull('longitude')
                            ->with('divisaoAdministrativa')
                            ->get();

        return view('jurisdicao.mapa', compact('areas', 'bacias', 'estacoes', 'barragens'));
    }

    /**
     * 3. ALISTAR/MAPEAR TODAS AS BACIAS, SUB-BACIAS E RIOS
     */
    public function inventario()
    {
        $bacias = BaciaHidrografica::with(['divisaoAdministrativa', 'areaJurisdicao'])
                                  ->orderBy('nome')
                                  ->get();

        $stats = [
            'total_bacias' => $bacias->count(),
            'bacias_com_coordenadas' => $bacias->where('coordenadas_geograficas', '!=', null)->count(),
            'bacias_com_divisao' => $bacias->where('divisao_administrativa_id', '!=', null)->count(),
            'area_total_km2' => $bacias->sum('area_km2'),
        ];

        // Agrupar por divisão administrativa
        $baciasPorDivisao = $bacias->groupBy('divisao_administrativa_id');

        return view('jurisdicao.inventario', compact('bacias', 'stats', 'baciasPorDivisao'));
    }

    /**
     * Atualizar coordenadas geográficas de uma bacia
     */
    public function atualizarCoordenadas(Request $request, $id)
    {
        $validated = $request->validate([
            'coordenadas_geograficas' => 'required|json',
        ]);

        $bacia = BaciaHidrografica::findOrFail($id);
        $bacia->coordenadas_geograficas = json_decode($validated['coordenadas_geograficas'], true);
        $bacia->save();

        return response()->json(['success' => true, 'message' => 'Coordenadas atualizadas com sucesso!']);
    }

    /**
     * Gestão de Áreas de Jurisdição
     */
    public function areas()
    {
        $areas = AreaJurisdicao::withCount('bacias')->get();
        return view('jurisdicao.areas.index', compact('areas'));
    }

    public function areasCreate()
    {
        return view('jurisdicao.areas.create');
    }

    public function areasStore(Request $request)
    {
        $validated = $request->validate([
            'nome' => 'required|string|max:255',
            'codigo' => 'required|string|unique:areas_jurisdicao',
            'descricao' => 'nullable|string',
            'limites_geograficos' => 'required|json',
            'area_total_km2' => 'required|numeric',
            'responsavel' => 'nullable|string',
            'contato' => 'nullable|string',
            'email' => 'nullable|email',
        ]);

        $validated['limites_geograficos'] = json_decode($validated['limites_geograficos'], true);

        AreaJurisdicao::create($validated);

        return redirect()->route('jurisdicao.areas')
                        ->with('success', 'Área de jurisdição criada com sucesso!');
    }

    /**
     * Vincular bacia a divisão administrativa
     */
    public function vincularBacia(Request $request)
    {
        $validated = $request->validate([
            'bacia_id' => 'required|exists:bacias_hidrograficas,id',
            'divisao_administrativa_id' => 'nullable|exists:divisoes_administrativas,id',
            'area_jurisdicao_id' => 'nullable|exists:areas_jurisdicao,id',
        ]);

        $bacia = BaciaHidrografica::findOrFail($validated['bacia_id']);
        $bacia->update([
            'divisao_administrativa_id' => $validated['divisao_administrativa_id'] ?? null,
            'area_jurisdicao_id' => $validated['area_jurisdicao_id'] ?? null,
        ]);

        return response()->json(['success' => true, 'message' => 'Bacia vinculada com sucesso!']);
    }
}
