<?php

namespace App\Http\Controllers;

use App\Models\Estacao;
use App\Models\LeituraEstacao;
use App\Models\Barragem;
use App\Models\LeituraBarragem;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class MediasController extends Controller
{
    /**
     * Página principal de médias
     */
    public function index()
    {
        return view('medias.index');
    }

    /**
     * MÉDIA DE PRECIPITAÇÃO
     */
    public function precipitacao(Request $request)
    {
        $periodo = $request->get('periodo', 'mensal'); // diario, semanal, mensal, anual, personalizado
        $estacaoId = $request->get('estacao_id');
        $dataInicio = $request->get('data_inicio', Carbon::now()->startOfMonth());
        $dataFim = $request->get('data_fim', Carbon::now());

        // Buscar estações disponíveis
        $estacoes = Estacao::whereIn('tipo', ['pluviometrica', 'meteorologica', 'climatologica'])
                           ->orderBy('nome')
                           ->get();

        $medias = [];
        $graficoLabels = [];
        $graficoDados = [];

        if ($estacaoId) {
            $estacao = Estacao::findOrFail($estacaoId);

            // Buscar leituras com precipitação
            $query = LeituraEstacao::where('estacao_id', $estacaoId)
                                   ->whereBetween('data_leitura', [$dataInicio, $dataFim])
                                   ->whereNotNull('precipitacao_mm')
                                   ->where('precipitacao_mm', '>', 0);

            switch ($periodo) {
                case 'diario':
                    $medias = $query->select(
                        DB::raw('DATE(data_leitura) as data'),
                        DB::raw('AVG(precipitacao_mm) as media'),
                        DB::raw('SUM(precipitacao_mm) as total'),
                        DB::raw('MIN(precipitacao_mm) as minimo'),
                        DB::raw('MAX(precipitacao_mm) as maximo'),
                        DB::raw('COUNT(*) as leituras')
                    )
                    ->groupBy('data')
                    ->orderBy('data')
                    ->get();
                    break;

                case 'semanal':
                    $medias = $query->select(
                        DB::raw('YEAR(data_leitura) as ano'),
                        DB::raw('WEEK(data_leitura) as semana'),
                        DB::raw('AVG(precipitacao_mm) as media'),
                        DB::raw('SUM(precipitacao_mm) as total'),
                        DB::raw('MIN(precipitacao_mm) as minimo'),
                        DB::raw('MAX(precipitacao_mm) as maximo'),
                        DB::raw('COUNT(*) as leituras')
                    )
                    ->groupBy('ano', 'semana')
                    ->orderBy('ano')
                    ->orderBy('semana')
                    ->get();
                    break;

                case 'mensal':
                    $medias = $query->select(
                        DB::raw('YEAR(data_leitura) as ano'),
                        DB::raw('MONTH(data_leitura) as mes'),
                        DB::raw('AVG(precipitacao_mm) as media'),
                        DB::raw('SUM(precipitacao_mm) as total'),
                        DB::raw('MIN(precipitacao_mm) as minimo'),
                        DB::raw('MAX(precipitacao_mm) as maximo'),
                        DB::raw('COUNT(*) as leituras')
                    )
                    ->groupBy('ano', 'mes')
                    ->orderBy('ano')
                    ->orderBy('mes')
                    ->get();
                    break;

                case 'anual':
                    $medias = $query->select(
                        DB::raw('YEAR(data_leitura) as ano'),
                        DB::raw('AVG(precipitacao_mm) as media'),
                        DB::raw('SUM(precipitacao_mm) as total'),
                        DB::raw('MIN(precipitacao_mm) as minimo'),
                        DB::raw('MAX(precipitacao_mm) as maximo'),
                        DB::raw('COUNT(*) as leituras')
                    )
                    ->groupBy('ano')
                    ->orderBy('ano')
                    ->get();
                    break;
            }

            // Preparar dados para gráfico
            foreach ($medias as $media) {
                switch ($periodo) {
                    case 'diario':
                        $graficoLabels[] = Carbon::parse($media->data)->format('d/m/Y');
                        break;
                    case 'semanal':
                        $graficoLabels[] = "Sem {$media->semana}/{$media->ano}";
                        break;
                    case 'mensal':
                        $graficoLabels[] = Carbon::create($media->ano, $media->mes)->format('M/Y');
                        break;
                    case 'anual':
                        $graficoLabels[] = $media->ano;
                        break;
                }
                $graficoDados[] = round($media->media, 2);
            }
        }

        return view('medias.precipitacao', compact(
            'estacoes',
            'medias',
            'periodo',
            'estacaoId',
            'dataInicio',
            'dataFim',
            'graficoLabels',
            'graficoDados'
        ));
    }

    /**
     * MÉDIA DE COTAS NAS BARRAGENS
     */
    public function cotasBarragens(Request $request)
    {
        $periodo = $request->get('periodo', 'mensal');
        $barragemId = $request->get('barragem_id');
        $dataInicio = $request->get('data_inicio', Carbon::now()->startOfMonth());
        $dataFim = $request->get('data_fim', Carbon::now());

        $barragens = Barragem::orderBy('nome')->get();

        $medias = [];
        $graficoLabels = [];
        $graficoDados = [];

        if ($barragemId) {
            $barragem = Barragem::findOrFail($barragemId);

            // Buscar leituras com cota
            $query = LeituraBarragem::where('barragem_id', $barragemId)
                                    ->whereBetween('data_leitura', [$dataInicio, $dataFim])
                                    ->whereNotNull('cota_actual');

            switch ($periodo) {
                case 'diario':
                    $medias = $query->select(
                        DB::raw('DATE(data_leitura) as data'),
                        DB::raw('AVG(cota_actual) as media'),
                        DB::raw('MIN(cota_actual) as minimo'),
                        DB::raw('MAX(cota_actual) as maximo'),
                        DB::raw('COUNT(*) as leituras')
                    )
                    ->groupBy('data')
                    ->orderBy('data')
                    ->get();
                    break;

                case 'semanal':
                    $medias = $query->select(
                        DB::raw('YEAR(data_leitura) as ano'),
                        DB::raw('WEEK(data_leitura) as semana'),
                        DB::raw('AVG(cota_actual) as media'),
                        DB::raw('MIN(cota_actual) as minimo'),
                        DB::raw('MAX(cota_actual) as maximo'),
                        DB::raw('COUNT(*) as leituras')
                    )
                    ->groupBy('ano', 'semana')
                    ->orderBy('ano')
                    ->orderBy('semana')
                    ->get();
                    break;

                case 'mensal':
                    $medias = $query->select(
                        DB::raw('YEAR(data_leitura) as ano'),
                        DB::raw('MONTH(data_leitura) as mes'),
                        DB::raw('AVG(cota_actual) as media'),
                        DB::raw('MIN(cota_actual) as minimo'),
                        DB::raw('MAX(cota_actual) as maximo'),
                        DB::raw('COUNT(*) as leituras')
                    )
                    ->groupBy('ano', 'mes')
                    ->orderBy('ano')
                    ->orderBy('mes')
                    ->get();
                    break;

                case 'anual':
                    $medias = $query->select(
                        DB::raw('YEAR(data_leitura) as ano'),
                        DB::raw('AVG(cota_actual) as media'),
                        DB::raw('MIN(cota_actual) as minimo'),
                        DB::raw('MAX(cota_actual) as maximo'),
                        DB::raw('COUNT(*) as leituras')
                    )
                    ->groupBy('ano')
                    ->orderBy('ano')
                    ->get();
                    break;
            }

            // Preparar dados para gráfico
            foreach ($medias as $media) {
                switch ($periodo) {
                    case 'diario':
                        $graficoLabels[] = Carbon::parse($media->data)->format('d/m/Y');
                        break;
                    case 'semanal':
                        $graficoLabels[] = "Sem {$media->semana}/{$media->ano}";
                        break;
                    case 'mensal':
                        $graficoLabels[] = Carbon::create($media->ano, $media->mes)->format('M/Y');
                        break;
                    case 'anual':
                        $graficoLabels[] = $media->ano;
                        break;
                }
                $graficoDados[] = round($media->media, 3);
            }
        }

        return view('medias.cotas-barragens', compact(
            'barragens',
            'medias',
            'periodo',
            'barragemId',
            'dataInicio',
            'dataFim',
            'graficoLabels',
            'graficoDados'
        ));
    }

    /**
     * MÉDIA DE NÍVEIS
     */
    public function niveis(Request $request)
    {
        $periodo = $request->get('periodo', 'mensal');
        $estacaoId = $request->get('estacao_id');
        $dataInicio = $request->get('data_inicio', Carbon::now()->startOfMonth());
        $dataFim = $request->get('data_fim', Carbon::now());

        $estacoes = Estacao::where('tipo', 'hidrometrica')
                           ->orderBy('nome')
                           ->get();

        $medias = [];
        $graficoLabels = [];
        $graficoDados = [];

        if ($estacaoId) {
            $estacao = Estacao::findOrFail($estacaoId);

            // Definir a fórmula de cálculo da média de níveis
            $nivelMedioFormula = '(COALESCE(nivel_6h, 0) + COALESCE(nivel_9h, 0) + COALESCE(nivel_12h, 0) + COALESCE(nivel_15h, 0) + COALESCE(nivel_18h, 0)) / ((CASE WHEN nivel_6h IS NOT NULL THEN 1 ELSE 0 END) + (CASE WHEN nivel_9h IS NOT NULL THEN 1 ELSE 0 END) + (CASE WHEN nivel_12h IS NOT NULL THEN 1 ELSE 0 END) + (CASE WHEN nivel_15h IS NOT NULL THEN 1 ELSE 0 END) + (CASE WHEN nivel_18h IS NOT NULL THEN 1 ELSE 0 END))';

            // Buscar leituras
            $query = LeituraEstacao::where('estacao_id', $estacaoId)
                                   ->whereBetween('data_leitura', [$dataInicio, $dataFim])
                                   ->where(function($q) {
                                       $q->whereNotNull('nivel_6h')
                                         ->orWhereNotNull('nivel_9h')
                                         ->orWhereNotNull('nivel_12h')
                                         ->orWhereNotNull('nivel_15h')
                                         ->orWhereNotNull('nivel_18h');
                                   });

            switch ($periodo) {
                case 'diario':
                    $medias = $query->select(
                        DB::raw('DATE(data_leitura) as data'),
                        DB::raw("AVG($nivelMedioFormula) as media"),
                        DB::raw("MIN($nivelMedioFormula) as minimo"),
                        DB::raw("MAX($nivelMedioFormula) as maximo"),
                        DB::raw('COUNT(*) as leituras')
                    )
                    ->groupBy('data')
                    ->orderBy('data')
                    ->get();
                    break;

                case 'semanal':
                    $medias = $query->select(
                        DB::raw('YEAR(data_leitura) as ano'),
                        DB::raw('WEEK(data_leitura) as semana'),
                        DB::raw("AVG($nivelMedioFormula) as media"),
                        DB::raw("MIN($nivelMedioFormula) as minimo"),
                        DB::raw("MAX($nivelMedioFormula) as maximo"),
                        DB::raw('COUNT(*) as leituras')
                    )
                    ->groupBy('ano', 'semana')
                    ->orderBy('ano')
                    ->orderBy('semana')
                    ->get();
                    break;

                case 'mensal':
                    $medias = $query->select(
                        DB::raw('YEAR(data_leitura) as ano'),
                        DB::raw('MONTH(data_leitura) as mes'),
                        DB::raw("AVG($nivelMedioFormula) as media"),
                        DB::raw("MIN($nivelMedioFormula) as minimo"),
                        DB::raw("MAX($nivelMedioFormula) as maximo"),
                        DB::raw('COUNT(*) as leituras')
                    )
                    ->groupBy('ano', 'mes')
                    ->orderBy('ano')
                    ->orderBy('mes')
                    ->get();
                    break;

                case 'anual':
                    $medias = $query->select(
                        DB::raw('YEAR(data_leitura) as ano'),
                        DB::raw("AVG($nivelMedioFormula) as media"),
                        DB::raw("MIN($nivelMedioFormula) as minimo"),
                        DB::raw("MAX($nivelMedioFormula) as maximo"),
                        DB::raw('COUNT(*) as leituras')
                    )
                    ->groupBy('ano')
                    ->orderBy('ano')
                    ->get();
                    break;
            }

            // Preparar dados para gráfico
            foreach ($medias as $media) {
                switch ($periodo) {
                    case 'diario':
                        $graficoLabels[] = Carbon::parse($media->data)->format('d/m/Y');
                        break;
                    case 'semanal':
                        $graficoLabels[] = "Sem {$media->semana}/{$media->ano}";
                        break;
                    case 'mensal':
                        $graficoLabels[] = Carbon::create($media->ano, $media->mes)->format('M/Y');
                        break;
                    case 'anual':
                        $graficoLabels[] = $media->ano;
                        break;
                }
                $graficoDados[] = round($media->media, 3);
            }
        }

        return view('medias.niveis', compact(
            'estacoes',
            'medias',
            'periodo',
            'estacaoId',
            'dataInicio',
            'dataFim',
            'graficoLabels',
            'graficoDados'
        ));
    }
}
