<?php

namespace App\Http\Controllers;

use App\Models\ColetaQA;
use App\Models\Estacao;
use App\Models\LimiteParametro;
use Illuminate\Http\Request;
use Carbon\Carbon;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Chart\Chart;
use PhpOffice\PhpSpreadsheet\Chart\DataSeries;
use PhpOffice\PhpSpreadsheet\Chart\DataSeriesValues;
use PhpOffice\PhpSpreadsheet\Chart\Legend;
use PhpOffice\PhpSpreadsheet\Chart\PlotArea;
use PhpOffice\PhpSpreadsheet\Chart\Title as ChartTitle;
use Barryvdh\DomPDF\Facade\Pdf;

class QAGlobalExportController extends Controller
{
    /**
     * Exportar para Excel com múltiplas abas (similar ao ficheiro original)
     */
    public function exportarExcel(Request $request)
    {
        $dataInicio = $request->get('data_inicio');
        $dataFim = $request->get('data_fim');

        $query = ColetaQA::with(['estacao.bacia_hidrografica', 'parametrosFisicoQuimicos', 'metaisPesados', 'iqaCalculado']);

        if ($dataInicio && $dataFim) {
            $query->whereBetween('data_coleta', [$dataInicio, $dataFim]);
        }

        $coletas = $query->orderBy('data_coleta')->get();

        if ($coletas->isEmpty()) {
            return back()->with('error', 'Nenhum dado encontrado para exportação.');
        }

        $spreadsheet = new Spreadsheet();

        // Remover sheet padrão
        $spreadsheet->removeSheetByIndex(0);

        // Criar abas para cada parâmetro
        $this->criarAbaTemperatura($spreadsheet, $coletas);
        $this->criarAbaPH($spreadsheet, $coletas);
        $this->criarAbaCE($spreadsheet, $coletas);
        $this->criarAbaTDS($spreadsheet, $coletas);
        $this->criarAbaNitrato($spreadsheet, $coletas);
        $this->criarAbaCloretos($spreadsheet, $coletas);
        $this->criarAbaFosfato($spreadsheet, $coletas);
        $this->criarAbaFerro($spreadsheet, $coletas);
        $this->criarAbaMetalPesado($spreadsheet, $coletas, 'Pb', 'chumbo');
        $this->criarAbaMetalPesado($spreadsheet, $coletas, 'Cd', 'cadmio');
        $this->criarAbaMetalPesado($spreadsheet, $coletas, 'Zn', 'zinco');
        $this->criarAbaMetalPesado($spreadsheet, $coletas, 'Cu', 'cobre');
        $this->criarAbaMetalPesado($spreadsheet, $coletas, 'As', 'arsenio');
        $this->criarAbaMetalPesado($spreadsheet, $coletas, 'Hg', 'mercurio');
        $this->criarAbaIQAGeral($spreadsheet, $coletas);

        // Salvar arquivo
        $filename = 'Proc_Dados_QA_Global_' . date('Y-m-d_His') . '.xlsx';
        $tempFile = storage_path('app/temp/' . $filename);

        if (!file_exists(storage_path('app/temp'))) {
            mkdir(storage_path('app/temp'), 0755, true);
        }

        $writer = new Xlsx($spreadsheet);
        $writer->setIncludeCharts(true);
        $writer->save($tempFile);

        return response()->download($tempFile, $filename)->deleteFileAfterSend(true);
    }

    private function criarAbaTemperatura($spreadsheet, $coletas)
    {
        $sheet = $spreadsheet->createSheet();
        $sheet->setTitle('Temperatura');

        // Cabeçalhos
        $sheet->setCellValue('B2', 'ID');
        $sheet->setCellValue('C2', 'ESTAÇÕES');
        $sheet->setCellValue('D2', 'TEMPERATURA ºC');
        $sheet->setCellValue('E2', 'LIMITE MÍNIMO');
        $sheet->setCellValue('F2', 'LIMITE MÁXIMO');

        // Limites
        $limite = LimiteParametro::where('parametro', 'temperatura')->first();

        // Dados
        $row = 3;
        $id = 1;
        foreach ($coletas as $coleta) {
            $temp = $coleta->parametrosFisicoQuimicos->temperatura ?? '';
            $sheet->setCellValue("B{$row}", $id);
            $sheet->setCellValue("C{$row}", $coleta->estacao->codigo);
            $sheet->setCellValue("D{$row}", $temp);
            $sheet->setCellValue("E{$row}", $limite->limite_minimo ?? '');
            $sheet->setCellValue("F{$row}", $limite->limite_maximo ?? '');
            $row++;
            $id++;
        }

        // Formatar
        $sheet->getStyle('B2:F2')->getFont()->setBold(true);
        $sheet->getColumnDimension('C')->setWidth(15);
        $sheet->getColumnDimension('D')->setWidth(18);
    }

    private function criarAbaPH($spreadsheet, $coletas)
    {
        $sheet = $spreadsheet->createSheet();
        $sheet->setTitle('PH');

        $sheet->setCellValue('B2', 'ID');
        $sheet->setCellValue('C2', 'ESTAÇÕES');
        $sheet->setCellValue('D2', 'PH');
        $sheet->setCellValue('E2', 'LIMITE MÍNIMO');
        $sheet->setCellValue('F2', 'LIMITE MÁXIMO');

        $limite = LimiteParametro::where('parametro', 'ph')->first();

        $row = 3;
        $id = 1;
        foreach ($coletas as $coleta) {
            $ph = $coleta->parametrosFisicoQuimicos->ph ?? '';
            $sheet->setCellValue("B{$row}", $id);
            $sheet->setCellValue("C{$row}", $coleta->estacao->codigo);
            $sheet->setCellValue("D{$row}", $ph);
            $sheet->setCellValue("E{$row}", $limite->limite_minimo ?? '');
            $sheet->setCellValue("F{$row}", $limite->limite_maximo ?? '');
            $row++;
            $id++;
        }

        $sheet->getStyle('B2:F2')->getFont()->setBold(true);
    }

    private function criarAbaCE($spreadsheet, $coletas)
    {
        $sheet = $spreadsheet->createSheet();
        $sheet->setTitle('CE');

        $sheet->setCellValue('B2', 'ID');
        $sheet->setCellValue('C2', 'ESTAÇÕES');
        $sheet->setCellValue('D2', 'CE µS/cm');
        $sheet->setCellValue('E2', 'LIMITE MÍNIMO');
        $sheet->setCellValue('F2', 'LIMITE MÁXIMO');

        $limite = LimiteParametro::where('parametro', 'condutividade_eletrica')->first();

        $row = 3;
        $id = 1;
        foreach ($coletas as $coleta) {
            $ce = $coleta->parametrosFisicoQuimicos->condutividade_eletrica ?? '';
            $sheet->setCellValue("B{$row}", $id);
            $sheet->setCellValue("C{$row}", $coleta->estacao->codigo);
            $sheet->setCellValue("D{$row}", $ce);
            $sheet->setCellValue("E{$row}", $limite->limite_minimo ?? '');
            $sheet->setCellValue("F{$row}", $limite->limite_maximo ?? '');
            $row++;
            $id++;
        }

        $sheet->getStyle('B2:F2')->getFont()->setBold(true);
    }

    private function criarAbaTDS($spreadsheet, $coletas)
    {
        $sheet = $spreadsheet->createSheet();
        $sheet->setTitle('TDS');

        $sheet->setCellValue('B2', 'ID');
        $sheet->setCellValue('C2', 'ESTAÇÕES');
        $sheet->setCellValue('D2', 'TDS mg/L');
        $sheet->setCellValue('E2', 'LIMITE MÁXIMO');

        $limite = LimiteParametro::where('parametro', 'tds')->first();

        $row = 3;
        $id = 1;
        foreach ($coletas as $coleta) {
            $tds = $coleta->parametrosFisicoQuimicos->tds ?? '';
            $sheet->setCellValue("B{$row}", $id);
            $sheet->setCellValue("C{$row}", $coleta->estacao->codigo);
            $sheet->setCellValue("D{$row}", $tds);
            $sheet->setCellValue("E{$row}", $limite->limite_maximo ?? '');
            $row++;
            $id++;
        }

        $sheet->getStyle('B2:E2')->getFont()->setBold(true);
    }

    private function criarAbaNitrato($spreadsheet, $coletas)
    {
        $sheet = $spreadsheet->createSheet();
        $sheet->setTitle('Nitrato');

        $sheet->setCellValue('B2', 'ID');
        $sheet->setCellValue('C2', 'ESTAÇÕES');
        $sheet->setCellValue('D2', 'NITRATO mg/L');
        $sheet->setCellValue('E2', 'LIMITE MÁXIMO');

        $limite = LimiteParametro::where('parametro', 'nitrato')->first();

        $row = 3;
        $id = 1;
        foreach ($coletas as $coleta) {
            $nitrato = $coleta->parametrosFisicoQuimicos->nitrato ?? '';
            $sheet->setCellValue("B{$row}", $id);
            $sheet->setCellValue("C{$row}", $coleta->estacao->codigo);
            $sheet->setCellValue("D{$row}", $nitrato);
            $sheet->setCellValue("E{$row}", $limite->limite_maximo ?? '');
            $row++;
            $id++;
        }

        $sheet->getStyle('B2:E2')->getFont()->setBold(true);
    }

    private function criarAbaCloretos($spreadsheet, $coletas)
    {
        $sheet = $spreadsheet->createSheet();
        $sheet->setTitle('Cloretos');

        $sheet->setCellValue('B2', 'ID');
        $sheet->setCellValue('C2', 'ESTAÇÕES');
        $sheet->setCellValue('D2', 'CLORETO mg/L');
        $sheet->setCellValue('E2', 'LIMITE MÁXIMO');

        $limite = LimiteParametro::where('parametro', 'cloretos')->first();

        $row = 3;
        $id = 1;
        foreach ($coletas as $coleta) {
            $cloreto = $coleta->parametrosFisicoQuimicos->cloretos ?? '';
            $sheet->setCellValue("B{$row}", $id);
            $sheet->setCellValue("C{$row}", $coleta->estacao->codigo);
            $sheet->setCellValue("D{$row}", $cloreto);
            $sheet->setCellValue("E{$row}", $limite->limite_maximo ?? '');
            $row++;
            $id++;
        }

        $sheet->getStyle('B2:E2')->getFont()->setBold(true);
    }

    private function criarAbaFosfato($spreadsheet, $coletas)
    {
        $sheet = $spreadsheet->createSheet();
        $sheet->setTitle('Fosfato');

        $sheet->setCellValue('B2', 'ID');
        $sheet->setCellValue('C2', 'ESTAÇÕES');
        $sheet->setCellValue('D2', 'FOSFATO mg/l');
        $sheet->setCellValue('E2', 'LIMITE MÁXIMO');

        $limite = LimiteParametro::where('parametro', 'fosfato')->first();

        $row = 3;
        $id = 1;
        foreach ($coletas as $coleta) {
            $fosfato = $coleta->parametrosFisicoQuimicos->fosfato ?? '';
            $sheet->setCellValue("B{$row}", $id);
            $sheet->setCellValue("C{$row}", $coleta->estacao->codigo);
            $sheet->setCellValue("D{$row}", $fosfato);
            $sheet->setCellValue("E{$row}", $limite->limite_maximo ?? '');
            $row++;
            $id++;
        }

        $sheet->getStyle('B2:E2')->getFont()->setBold(true);
    }

    private function criarAbaFerro($spreadsheet, $coletas)
    {
        $sheet = $spreadsheet->createSheet();
        $sheet->setTitle('Ferro');

        $sheet->setCellValue('B2', 'ID');
        $sheet->setCellValue('C2', 'ESTAÇÕES');
        $sheet->setCellValue('D2', 'FERRO mg/l');
        $sheet->setCellValue('E2', 'LIMITE MÁXIMO');

        $limite = LimiteParametro::where('parametro', 'ferro')->first();

        $row = 3;
        $id = 1;
        foreach ($coletas as $coleta) {
            $ferro = $coleta->metaisPesados->ferro ?? '';
            $sheet->setCellValue("B{$row}", $id);
            $sheet->setCellValue("C{$row}", $coleta->estacao->codigo);
            $sheet->setCellValue("D{$row}", $ferro);
            $sheet->setCellValue("E{$row}", $limite->limite_maximo ?? '');
            $row++;
            $id++;
        }

        $sheet->getStyle('B2:E2')->getFont()->setBold(true);
    }

    private function criarAbaMetalPesado($spreadsheet, $coletas, $simbolo, $campo)
    {
        $sheet = $spreadsheet->createSheet();
        $sheet->setTitle($simbolo);

        $sheet->setCellValue('B2', 'ID');
        $sheet->setCellValue('C2', 'ESTAÇÕES');
        $sheet->setCellValue('D2', "{$simbolo} mg/l");
        $sheet->setCellValue('E2', 'LIMITE MÁXIMO');

        $limite = LimiteParametro::where('parametro', $campo)->first();

        $row = 3;
        $id = 1;
        foreach ($coletas as $coleta) {
            $valor = $coleta->metaisPesados->$campo ?? '';
            $sheet->setCellValue("B{$row}", $id);
            $sheet->setCellValue("C{$row}", $coleta->estacao->codigo);
            $sheet->setCellValue("D{$row}", $valor);
            $sheet->setCellValue("E{$row}", $limite->limite_maximo ?? '');
            $row++;
            $id++;
        }

        $sheet->getStyle('B2:E2')->getFont()->setBold(true);
    }

    private function criarAbaIQAGeral($spreadsheet, $coletas)
    {
        $sheet = $spreadsheet->createSheet();
        $sheet->setTitle('IQA Geral');

        // Cabeçalhos mesclados
        $sheet->mergeCells('B2:B3');
        $sheet->mergeCells('C2:C3');
        $sheet->mergeCells('D2:L2');
        $sheet->mergeCells('M2:M3');
        $sheet->mergeCells('N2:N3');

        $sheet->setCellValue('B2', 'B.Hidrográfica');
        $sheet->setCellValue('C2', 'Estação');
        $sheet->setCellValue('D2', 'PARÂMETROS');
        $sheet->setCellValue('M2', 'IQA');
        $sheet->setCellValue('N2', 'Estado do IQA');

        // Sub-cabeçalhos
        $sheet->setCellValue('D3', 'OD (%)');
        $sheet->setCellValue('E3', 'C.F.(NMP/100mL)');
        $sheet->setCellValue('F3', 'PH');
        $sheet->setCellValue('G3', 'DBO (mg/L)');
        $sheet->setCellValue('H3', 'Nit. Total. (mg/L)');
        $sheet->setCellValue('I3', 'Fosf.to t. (mg/L)');
        $sheet->setCellValue('J3', 'Turb (NTU)');
        $sheet->setCellValue('K3', 'Sol.to.t (mg/L)');
        $sheet->setCellValue('L3', 'Temp. (ºC)');

        // Dados
        $row = 4;
        foreach ($coletas as $coleta) {
            $params = $coleta->parametrosFisicoQuimicos;
            $iqa = $coleta->iqaCalculado;

            $sheet->setCellValue("B{$row}", $coleta->estacao->bacia_hidrografica->nome ?? '');
            $sheet->setCellValue("C{$row}", $coleta->estacao->codigo);
            $sheet->setCellValue("D{$row}", $params->oxigenio_dissolvido ?? '');
            $sheet->setCellValue("E{$row}", $params->coliformes_fecais ?? '');
            $sheet->setCellValue("F{$row}", $params->ph ?? '');
            $sheet->setCellValue("G{$row}", $params->dbo ?? '');
            $sheet->setCellValue("H{$row}", $params->nitrogenio_total ?? '');
            $sheet->setCellValue("I{$row}", $params->fosfato_total ?? '');
            $sheet->setCellValue("J{$row}", $params->turbidez ?? '');
            $sheet->setCellValue("K{$row}", $params->solidos_totais ?? '');
            $sheet->setCellValue("L{$row}", $params->temperatura ?? '');
            $sheet->setCellValue("M{$row}", $iqa->iqa_valor ?? '');
            $sheet->setCellValue("N{$row}", $iqa->iqa_classificacao ?? '');

            $row++;
        }

        // Formatar
        $sheet->getStyle('B2:N3')->getFont()->setBold(true);
        $sheet->getStyle('B2:N3')->getAlignment()->setHorizontal('center');
    }

    /**
     * Exportar para CSV
     */
    public function exportarCSV(Request $request)
    {
        $dataInicio = $request->get('data_inicio');
        $dataFim = $request->get('data_fim');

        $query = ColetaQA::with(['estacao.bacia_hidrografica', 'parametrosFisicoQuimicos', 'metaisPesados', 'iqaCalculado']);

        if ($dataInicio && $dataFim) {
            $query->whereBetween('data_coleta', [$dataInicio, $dataFim]);
        }

        $dados = $query->orderBy('data_coleta')->get();

        $filename = 'qa_global_' . date('Y-m-d_His') . '.csv';

        $headers = [
            'Content-Type' => 'text/csv; charset=UTF-8',
            'Content-Disposition' => "attachment; filename=\"$filename\"",
        ];

        $callback = function() use ($dados) {
            $file = fopen('php://output', 'w');

            // BOM para UTF-8
            fprintf($file, chr(0xEF).chr(0xBB).chr(0xBF));

            // Cabeçalhos
            fputcsv($file, [
                'ID', 'Bacia Hidrográfica', 'Estação', 'Data Coleta', 'Hora', 'Responsável',
                'Temperatura(°C)', 'pH', 'CE(µS/cm)', 'TDS(mg/L)', 'OD(%)', 'Turbidez(NTU)',
                'Nitrato(mg/L)', 'N Total(mg/L)', 'Fosfato(mg/L)', 'P Total(mg/L)',
                'DBO(mg/L)', 'Coliformes', 'Sólidos Totais(mg/L)', 'Cloretos(mg/L)',
                'Fe(mg/L)', 'Pb(mg/L)', 'Cd(mg/L)', 'Zn(mg/L)', 'Cu(mg/L)', 'As(mg/L)', 'Hg(mg/L)',
                'IQA', 'Classificação', 'Validado'
            ]);

            // Dados
            foreach ($dados as $dado) {
                fputcsv($file, [
                    $dado->id,
                    $dado->estacao->bacia_hidrografica->nome ?? '',
                    $dado->estacao->codigo,
                    $dado->data_coleta->format('Y-m-d'),
                    $dado->hora_coleta ? $dado->hora_coleta->format('H:i') : '',
                    $dado->responsavel,
                    $dado->parametrosFisicoQuimicos->temperatura,
                    $dado->parametrosFisicoQuimicos->ph,
                    $dado->parametrosFisicoQuimicos->condutividade_eletrica,
                    $dado->parametrosFisicoQuimicos->tds,
                    $dado->parametrosFisicoQuimicos->oxigenio_dissolvido,
                    $dado->parametrosFisicoQuimicos->turbidez,
                    $dado->parametrosFisicoQuimicos->nitrato,
                    $dado->parametrosFisicoQuimicos->nitrogenio_total,
                    $dado->parametrosFisicoQuimicos->fosfato,
                    $dado->parametrosFisicoQuimicos->fosfato_total,
                    $dado->parametrosFisicoQuimicos->dbo,
                    $dado->parametrosFisicoQuimicos->coliformes_fecais,
                    $dado->parametrosFisicoQuimicos->solidos_totais,
                    $dado->parametrosFisicoQuimicos->cloretos,
                    $dado->metaisPesados->ferro,
                    $dado->metaisPesados->chumbo,
                    $dado->metaisPesados->cadmio,
                    $dado->metaisPesados->zinco,
                    $dado->metaisPesados->cobre,
                    $dado->metaisPesados->arsenio,
                    $dado->metaisPesados->mercurio,
                    $dado->iqaCalculado->iqa_valor ?? '',
                    $dado->iqaCalculado->iqa_classificacao ?? '',
                    $dado->validado ? 'Sim' : 'Não',
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    /**
     * Exportar para PDF
     */
    public function exportarPDF(Request $request, ColetaQA $coletaQa = null)
    {
        if ($coletaQa) {
            // Exportar uma coleta específica
            $coletaQa->load([
                'estacao.bacia_hidrografica',
                'parametrosFisicoQuimicos',
                'metaisPesados',
                'iqaCalculado',
                'validadoPor'
            ]);

            $alertas = $coletaQa->verificarConformidade();
            $limites = LimiteParametro::all()->keyBy('parametro');

            $pdf = Pdf::loadView('qa-global.pdf.coleta', compact('coletaQa', 'alertas', 'limites'));

            $filename = "Relatorio_QA_{$coletaQa->estacao->codigo}_{$coletaQa->data_coleta->format('Y-m-d')}.pdf";

            return $pdf->download($filename);
        } else {
            // Exportar relatório geral
            $dataInicio = $request->get('data_inicio', Carbon::now()->subMonths(3));
            $dataFim = $request->get('data_fim', Carbon::now());

            $coletas = ColetaQA::with(['estacao.bacia_hidrografica', 'iqaCalculado', 'parametrosFisicoQuimicos'])
                              ->whereBetween('data_coleta', [$dataInicio, $dataFim])
                              ->orderBy('data_coleta')
                              ->get();

            $pdf = Pdf::loadView('qa-global.pdf.relatorio-geral', compact('coletas', 'dataInicio', 'dataFim'));

            return $pdf->download('Relatorio_QA_Geral_' . date('Y-m-d') . '.pdf');
        }
    }
}
