<?php

namespace App\Http\Controllers;

use App\Models\QualidadeAgua;
use App\Models\Barragem;
use App\Models\Estacao;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class QualidadeAguaController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = QualidadeAgua::with(['barragem', 'estacao', 'validadoPor']);

        // Filtros
        if ($request->filled('barragem_id')) {
            $query->where('barragem_id', $request->barragem_id);
        }

        if ($request->filled('estacao_id')) {
            $query->where('estacao_id', $request->estacao_id);
        }

        if ($request->filled('data_inicio') && $request->filled('data_fim')) {
            $query->whereBetween('data_coleta', [$request->data_inicio, $request->data_fim]);
        }

        if ($request->filled('classificacao')) {
            $query->where('classificacao_iqa', $request->classificacao);
        }

        if ($request->filled('validado')) {
            $query->where('validado', $request->validado);
        }

        $qualidades = $query->orderBy('data_coleta', 'desc')
                           ->orderBy('hora_coleta', 'desc')
                           ->paginate(20);

        $barragens = Barragem::orderBy('nome')->get();
        $estacoes = Estacao::orderBy('nome')->get();

        return view('qualidade-agua.index', compact('qualidades', 'barragens', 'estacoes'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $barragens = Barragem::orderBy('nome')->get();
        $estacoes = Estacao::orderBy('nome')->get();

        return view('qualidade-agua.create', compact('barragens', 'estacoes'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'barragem_id' => 'nullable|exists:barragens,id',
            'estacao_id' => 'nullable|exists:estacoes,id',
            'data_coleta' => 'required|date',
            'hora_coleta' => 'required',
            'oxigenio_dissolvido' => 'nullable|numeric|min:0|max:20',
            'coliformes_fecais' => 'nullable|numeric|min:0',
            'ph' => 'nullable|numeric|min:0|max:14',
            'dbo5' => 'nullable|numeric|min:0',
            'nitrogenio_total' => 'nullable|numeric|min:0',
            'fosforo_total' => 'nullable|numeric|min:0',
            'temperatura' => 'nullable|numeric|min:-10|max:60',
            'turbidez' => 'nullable|numeric|min:0',
            'solidos_totais' => 'nullable|numeric|min:0',
            'condutividade' => 'nullable|numeric|min:0',
            'salinidade' => 'nullable|numeric|min:0',
            'clorofila_a' => 'nullable|numeric|min:0',
            'transparencia' => 'nullable|numeric|min:0',
            'local_coleta' => 'nullable|string|max:255',
            'profundidade_coleta' => 'nullable|numeric|min:0',
            'responsavel_coleta' => 'nullable|string|max:255',
            'laboratorio' => 'nullable|string|max:255',
            'data_analise' => 'nullable|date',
            'observacoes' => 'nullable|string',
        ]);

        // Validar que pelo menos um local (barragem ou estação) foi selecionado
        if (empty($validated['barragem_id']) && empty($validated['estacao_id'])) {
            return back()->withErrors(['barragem_id' => 'Selecione uma barragem ou estação.'])->withInput();
        }

        $qualidade = QualidadeAgua::create($validated);

        // Calcular IQA automaticamente
        $qualidade->calcularIQA();
        $qualidade->save();

        return redirect()->route('qualidade-agua.show', $qualidade)
                        ->with('success', 'Dados de qualidade de água registrados com sucesso. IQA calculado: ' . $qualidade->iqa);
    }

    /**
     * Display the specified resource.
     */
    public function show(QualidadeAgua $qualidadeAgua)
    {
        $qualidadeAgua->load(['barragem', 'estacao', 'validadoPor']);

        return view('qualidade-agua.show', compact('qualidadeAgua'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(QualidadeAgua $qualidadeAgua)
    {
        $barragens = Barragem::orderBy('nome')->get();
        $estacoes = Estacao::orderBy('nome')->get();

        return view('qualidade-agua.edit', compact('qualidadeAgua', 'barragens', 'estacoes'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, QualidadeAgua $qualidadeAgua)
    {
        $validated = $request->validate([
            'barragem_id' => 'nullable|exists:barragens,id',
            'estacao_id' => 'nullable|exists:estacoes,id',
            'data_coleta' => 'required|date',
            'hora_coleta' => 'required',
            'oxigenio_dissolvido' => 'nullable|numeric|min:0|max:20',
            'coliformes_fecais' => 'nullable|numeric|min:0',
            'ph' => 'nullable|numeric|min:0|max:14',
            'dbo5' => 'nullable|numeric|min:0',
            'nitrogenio_total' => 'nullable|numeric|min:0',
            'fosforo_total' => 'nullable|numeric|min:0',
            'temperatura' => 'nullable|numeric|min:-10|max:60',
            'turbidez' => 'nullable|numeric|min:0',
            'solidos_totais' => 'nullable|numeric|min:0',
            'condutividade' => 'nullable|numeric|min:0',
            'salinidade' => 'nullable|numeric|min:0',
            'clorofila_a' => 'nullable|numeric|min:0',
            'transparencia' => 'nullable|numeric|min:0',
            'local_coleta' => 'nullable|string|max:255',
            'profundidade_coleta' => 'nullable|numeric|min:0',
            'responsavel_coleta' => 'nullable|string|max:255',
            'laboratorio' => 'nullable|string|max:255',
            'data_analise' => 'nullable|date',
            'observacoes' => 'nullable|string',
        ]);

        if (empty($validated['barragem_id']) && empty($validated['estacao_id'])) {
            return back()->withErrors(['barragem_id' => 'Selecione uma barragem ou estação.'])->withInput();
        }

        $qualidadeAgua->update($validated);

        // Recalcular IQA
        $qualidadeAgua->calcularIQA();
        $qualidadeAgua->save();

        return redirect()->route('qualidade-agua.show', $qualidadeAgua)
                        ->with('success', 'Dados de qualidade de água atualizados com sucesso. IQA recalculado: ' . $qualidadeAgua->iqa);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(QualidadeAgua $qualidadeAgua)
    {
        $qualidadeAgua->delete();

        return redirect()->route('qualidade-agua.index')
                        ->with('success', 'Registro de qualidade de água excluído com sucesso.');
    }

    /**
     * Validar um registro de qualidade de água
     */
    public function validar(QualidadeAgua $qualidadeAgua)
    {
        $qualidadeAgua->update([
            'validado' => true,
            'validado_por' => Auth::id(),
            'data_validacao' => now(),
        ]);

        return back()->with('success', 'Registro validado com sucesso.');
    }

    /**
     * Recalcular IQA manualmente
     */
    public function recalcularIQA(QualidadeAgua $qualidadeAgua)
    {
        $iqa = $qualidadeAgua->calcularIQA();
        $qualidadeAgua->save();

        return back()->with('success', 'IQA recalculado: ' . $iqa . ' - Classificação: ' . $qualidadeAgua->classificacao_iqa);
    }

    /**
     * Estatísticas de qualidade de água
     */
    public function estatisticas(Request $request)
    {
        $barragemId = $request->get('barragem_id');
        $estacaoId = $request->get('estacao_id');
        $dataInicio = $request->get('data_inicio', Carbon::now()->subYear());
        $dataFim = $request->get('data_fim', Carbon::now());

        $query = QualidadeAgua::query()
            ->whereBetween('data_coleta', [$dataInicio, $dataFim]);

        if ($barragemId) {
            $query->where('barragem_id', $barragemId);
        }

        if ($estacaoId) {
            $query->where('estacao_id', $estacaoId);
        }

        // Estatísticas gerais
        $estatisticas = [
            'total_amostras' => $query->count(),
            'iqa_medio' => $query->avg('iqa'),
            'iqa_minimo' => $query->min('iqa'),
            'iqa_maximo' => $query->max('iqa'),
            'validados' => $query->where('validado', true)->count(),
            'por_classificacao' => $query->select('classificacao_iqa', DB::raw('count(*) as total'))
                                        ->groupBy('classificacao_iqa')
                                        ->pluck('total', 'classificacao_iqa')
                                        ->toArray(),
        ];

        // Tendência mensal do IQA
        $tendencia = $query->select(
                                DB::raw('DATE_FORMAT(data_coleta, "%Y-%m") as mes'),
                                DB::raw('AVG(iqa) as iqa_medio'),
                                DB::raw('COUNT(*) as total_amostras')
                            )
                            ->groupBy('mes')
                            ->orderBy('mes')
                            ->get();

        // Médias de parâmetros
        $parametros = [
            'oxigenio_dissolvido' => $query->avg('oxigenio_dissolvido'),
            'coliformes_fecais' => $query->avg('coliformes_fecais'),
            'ph' => $query->avg('ph'),
            'dbo5' => $query->avg('dbo5'),
            'nitrogenio_total' => $query->avg('nitrogenio_total'),
            'fosforo_total' => $query->avg('fosforo_total'),
            'temperatura' => $query->avg('temperatura'),
            'turbidez' => $query->avg('turbidez'),
            'solidos_totais' => $query->avg('solidos_totais'),
        ];

        $barragens = Barragem::orderBy('nome')->get();
        $estacoes = Estacao::orderBy('nome')->get();

        return view('qualidade-agua.estatisticas', compact(
            'estatisticas',
            'tendencia',
            'parametros',
            'barragens',
            'estacoes',
            'barragemId',
            'estacaoId',
            'dataInicio',
            'dataFim'
        ));
    }

    /**
     * Exportar dados para Excel/CSV
     */
    public function exportar(Request $request)
    {
        $query = QualidadeAgua::with(['barragem', 'estacao']);

        if ($request->filled('barragem_id')) {
            $query->where('barragem_id', $request->barragem_id);
        }

        if ($request->filled('estacao_id')) {
            $query->where('estacao_id', $request->estacao_id);
        }

        if ($request->filled('data_inicio') && $request->filled('data_fim')) {
            $query->whereBetween('data_coleta', [$request->data_inicio, $request->data_fim]);
        }

        $dados = $query->orderBy('data_coleta')->get();

        $filename = 'qualidade_agua_' . date('Y-m-d_His') . '.csv';

        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"$filename\"",
        ];

        $callback = function() use ($dados) {
            $file = fopen('php://output', 'w');

            // Cabeçalhos
            fputcsv($file, [
                'ID',
                'Barragem',
                'Estação',
                'Data Coleta',
                'Hora Coleta',
                'OD (mg/L)',
                'Coliformes Fecais',
                'pH',
                'DBO5 (mg/L)',
                'Nitrogênio Total (mg/L)',
                'Fósforo Total (mg/L)',
                'Temperatura (°C)',
                'Turbidez (UNT)',
                'Sólidos Totais (mg/L)',
                'IQA',
                'Classificação',
                'Validado',
                'Observações'
            ]);

            // Dados
            foreach ($dados as $dado) {
                fputcsv($file, [
                    $dado->id,
                    $dado->barragem->nome ?? '',
                    $dado->estacao->nome ?? '',
                    $dado->data_coleta,
                    $dado->hora_coleta,
                    $dado->oxigenio_dissolvido,
                    $dado->coliformes_fecais,
                    $dado->ph,
                    $dado->dbo5,
                    $dado->nitrogenio_total,
                    $dado->fosforo_total,
                    $dado->temperatura,
                    $dado->turbidez,
                    $dado->solidos_totais,
                    $dado->iqa,
                    $dado->classificacao_iqa,
                    $dado->validado ? 'Sim' : 'Não',
                    $dado->observacoes
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}
