<?php

namespace App\Http\Controllers;

use App\Models\ColetaQA;
use App\Models\Estacao;
use App\Models\BaciaHidrografica;
use App\Models\LimiteParametro;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class QualidadeAguaGlobalController extends Controller
{
    /**
     * Listagem de coletas
     */
    public function index(Request $request)
    {
        $query = ColetaQA::with(['estacao.bacia_hidrografica', 'iqaCalculado']);

        // Filtros
        if ($request->filled('estacao_id')) {
            $query->where('estacao_id', $request->estacao_id);
        }

        if ($request->filled('data_inicio') && $request->filled('data_fim')) {
            $query->whereBetween('data_coleta', [$request->data_inicio, $request->data_fim]);
        }

        if ($request->filled('classificacao')) {
            $query->whereHas('iqaCalculado', function ($q) use ($request) {
                $q->where('iqa_classificacao', $request->classificacao);
            });
        }

        $coletas = $query->orderBy('data_coleta', 'desc')
                        ->orderBy('hora_coleta', 'desc')
                        ->paginate(20);

        $estacoes = Estacao::orderBy('codigo')->get();

        return view('qa-global.index', compact('coletas', 'estacoes'));
    }

    /**
     * Formulário de nova coleta
     */
    public function create()
    {
        $estacoes = Estacao::with('bacia_hidrografica')->orderBy('codigo')->get();
        $limites = LimiteParametro::all()->keyBy('parametro');

        return view('qa-global.create', compact('estacoes', 'limites'));
    }

    /**
     * Salvar nova coleta
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'estacao_id' => 'required|exists:estacoes,id',
            'data_coleta' => 'required|date',
            'hora_coleta' => 'nullable',
            'responsavel' => 'nullable|string|max:100',
            'observacoes' => 'nullable|string',

            // Parâmetros físico-químicos
            'temperatura' => 'nullable|numeric|min:0|max:50',
            'ph' => 'nullable|numeric|min:0|max:14',
            'condutividade_eletrica' => 'nullable|integer|min:0',
            'tds' => 'nullable|integer|min:0',
            'oxigenio_dissolvido' => 'nullable|numeric|min:0',
            'turbidez' => 'nullable|numeric|min:0',
            'nitrato' => 'nullable|numeric|min:0',
            'nitrogenio_total' => 'nullable|numeric|min:0',
            'fosfato' => 'nullable|numeric|min:0',
            'fosfato_total' => 'nullable|numeric|min:0',
            'dbo' => 'nullable|numeric|min:0',
            'coliformes_fecais' => 'nullable|integer|min:0',
            'solidos_totais' => 'nullable|integer|min:0',
            'cloretos' => 'nullable|numeric|min:0',

            // Metais pesados
            'ferro' => 'nullable|numeric|min:0',
            'chumbo' => 'nullable|numeric|min:0',
            'cadmio' => 'nullable|numeric|min:0',
            'zinco' => 'nullable|numeric|min:0',
            'cobre' => 'nullable|numeric|min:0',
            'arsenio' => 'nullable|numeric|min:0',
            'mercurio' => 'nullable|numeric|min:0',
        ]);

        DB::beginTransaction();
        try {
            // Criar coleta
            $coleta = ColetaQA::create([
                'estacao_id' => $validated['estacao_id'],
                'data_coleta' => $validated['data_coleta'],
                'hora_coleta' => $validated['hora_coleta'],
                'responsavel' => $validated['responsavel'],
                'observacoes' => $validated['observacoes'],
            ]);

            // Atualizar parâmetros físico-químicos
            $coleta->parametrosFisicoQuimicos()->update([
                'temperatura' => $validated['temperatura'] ?? null,
                'ph' => $validated['ph'] ?? null,
                'condutividade_eletrica' => $validated['condutividade_eletrica'] ?? null,
                'tds' => $validated['tds'] ?? null,
                'oxigenio_dissolvido' => $validated['oxigenio_dissolvido'] ?? null,
                'turbidez' => $validated['turbidez'] ?? null,
                'nitrato' => $validated['nitrato'] ?? null,
                'nitrogenio_total' => $validated['nitrogenio_total'] ?? null,
                'fosfato' => $validated['fosfato'] ?? null,
                'fosfato_total' => $validated['fosfato_total'] ?? null,
                'dbo' => $validated['dbo'] ?? null,
                'coliformes_fecais' => $validated['coliformes_fecais'] ?? null,
                'solidos_totais' => $validated['solidos_totais'] ?? null,
                'cloretos' => $validated['cloretos'] ?? null,
            ]);

            // Atualizar metais pesados
            $coleta->metaisPesados()->update([
                'ferro' => $validated['ferro'] ?? null,
                'chumbo' => $validated['chumbo'] ?? null,
                'cadmio' => $validated['cadmio'] ?? null,
                'zinco' => $validated['zinco'] ?? null,
                'cobre' => $validated['cobre'] ?? null,
                'arsenio' => $validated['arsenio'] ?? null,
                'mercurio' => $validated['mercurio'] ?? null,
            ]);

            // Calcular IQA
            $iqa = $coleta->calcularIQA();

            DB::commit();

            return redirect()->route('qa-global.show', $coleta)
                           ->with('success', "Coleta registrada com sucesso! IQA calculado: {$iqa}");

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withErrors(['error' => 'Erro ao salvar coleta: ' . $e->getMessage()])->withInput();
        }
    }

    /**
     * Visualizar coleta
     */
    public function show(ColetaQA $coletaQa)
    {
        $coletaQa->load([
            'estacao.bacia_hidrografica',
            'parametrosFisicoQuimicos',
            'metaisPesados',
            'iqaCalculado',
            'validadoPor'
        ]);

        $alertas = $coletaQa->verificarConformidade();
        $limites = LimiteParametro::all()->keyBy('parametro');

        return view('qa-global.show', compact('coletaQa', 'alertas', 'limites'));
    }

    /**
     * Editar coleta
     */
    public function edit(ColetaQA $coletaQa)
    {
        $coletaQa->load(['parametrosFisicoQuimicos', 'metaisPesados']);
        $estacoes = Estacao::with('bacia_hidrografica')->orderBy('codigo')->get();
        $limites = LimiteParametro::all()->keyBy('parametro');

        return view('qa-global.edit', compact('coletaQa', 'estacoes', 'limites'));
    }

    /**
     * Atualizar coleta
     */
    public function update(Request $request, ColetaQA $coletaQa)
    {
        $validated = $request->validate([
            'estacao_id' => 'required|exists:estacoes,id',
            'data_coleta' => 'required|date',
            'hora_coleta' => 'nullable',
            'responsavel' => 'nullable|string|max:100',
            'observacoes' => 'nullable|string',

            // Parâmetros físico-químicos
            'temperatura' => 'nullable|numeric|min:0|max:50',
            'ph' => 'nullable|numeric|min:0|max:14',
            'condutividade_eletrica' => 'nullable|integer|min:0',
            'tds' => 'nullable|integer|min:0',
            'oxigenio_dissolvido' => 'nullable|numeric|min:0',
            'turbidez' => 'nullable|numeric|min:0',
            'nitrato' => 'nullable|numeric|min:0',
            'nitrogenio_total' => 'nullable|numeric|min:0',
            'fosfato' => 'nullable|numeric|min:0',
            'fosfato_total' => 'nullable|numeric|min:0',
            'dbo' => 'nullable|numeric|min:0',
            'coliformes_fecais' => 'nullable|integer|min:0',
            'solidos_totais' => 'nullable|integer|min:0',
            'cloretos' => 'nullable|numeric|min:0',

            // Metais pesados
            'ferro' => 'nullable|numeric|min:0',
            'chumbo' => 'nullable|numeric|min:0',
            'cadmio' => 'nullable|numeric|min:0',
            'zinco' => 'nullable|numeric|min:0',
            'cobre' => 'nullable|numeric|min:0',
            'arsenio' => 'nullable|numeric|min:0',
            'mercurio' => 'nullable|numeric|min:0',
        ]);

        DB::beginTransaction();
        try {
            // Atualizar coleta
            $coletaQa->update([
                'estacao_id' => $validated['estacao_id'],
                'data_coleta' => $validated['data_coleta'],
                'hora_coleta' => $validated['hora_coleta'],
                'responsavel' => $validated['responsavel'],
                'observacoes' => $validated['observacoes'],
            ]);

            // Atualizar parâmetros físico-químicos
            $coletaQa->parametrosFisicoQuimicos()->update([
                'temperatura' => $validated['temperatura'] ?? null,
                'ph' => $validated['ph'] ?? null,
                'condutividade_eletrica' => $validated['condutividade_eletrica'] ?? null,
                'tds' => $validated['tds'] ?? null,
                'oxigenio_dissolvido' => $validated['oxigenio_dissolvido'] ?? null,
                'turbidez' => $validated['turbidez'] ?? null,
                'nitrato' => $validated['nitrato'] ?? null,
                'nitrogenio_total' => $validated['nitrogenio_total'] ?? null,
                'fosfato' => $validated['fosfato'] ?? null,
                'fosfato_total' => $validated['fosfato_total'] ?? null,
                'dbo' => $validated['dbo'] ?? null,
                'coliformes_fecais' => $validated['coliformes_fecais'] ?? null,
                'solidos_totais' => $validated['solidos_totais'] ?? null,
                'cloretos' => $validated['cloretos'] ?? null,
            ]);

            // Atualizar metais pesados
            $coletaQa->metaisPesados()->update([
                'ferro' => $validated['ferro'] ?? null,
                'chumbo' => $validated['chumbo'] ?? null,
                'cadmio' => $validated['cadmio'] ?? null,
                'zinco' => $validated['zinco'] ?? null,
                'cobre' => $validated['cobre'] ?? null,
                'arsenio' => $validated['arsenio'] ?? null,
                'mercurio' => $validated['mercurio'] ?? null,
            ]);

            // Recalcular IQA
            $iqa = $coletaQa->calcularIQA();

            DB::commit();

            return redirect()->route('qa-global.show', $coletaQa)
                           ->with('success', "Coleta atualizada com sucesso! IQA recalculado: {$iqa}");

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withErrors(['error' => 'Erro ao atualizar coleta: ' . $e->getMessage()])->withInput();
        }
    }

    /**
     * Excluir coleta
     */
    public function destroy(ColetaQA $coletaQa)
    {
        $coletaQa->delete();

        return redirect()->route('qa-global.index')
                       ->with('success', 'Coleta excluída com sucesso.');
    }

    /**
     * Validar coleta
     */
    public function validar(ColetaQA $coletaQa)
    {
        $coletaQa->update([
            'validado' => true,
            'validado_por' => Auth::id(),
            'data_validacao' => now(),
        ]);

        return back()->with('success', 'Coleta validada com sucesso.');
    }

    /**
     * Gráficos por parâmetro
     */
    public function graficos(Request $request)
    {
        $parametro = $request->get('parametro', 'temperatura');
        $dataInicio = $request->get('data_inicio', Carbon::now()->subMonths(6));
        $dataFim = $request->get('data_fim', Carbon::now());

        $coletas = ColetaQA::with(['estacao', 'parametrosFisicoQuimicos', 'metaisPesados', 'iqaCalculado'])
                          ->whereBetween('data_coleta', [$dataInicio, $dataFim])
                          ->orderBy('data_coleta')
                          ->get();

        $limites = LimiteParametro::where('parametro', $parametro)->first();

        return view('qa-global.graficos', compact('coletas', 'parametro', 'limites'));
    }

    /**
     * Dashboard IQA Geral
     */
    public function dashboard(Request $request)
    {
        $dataInicio = $request->get('data_inicio', Carbon::now()->subMonths(3));
        $dataFim = $request->get('data_fim', Carbon::now());

        $coletas = ColetaQA::with(['estacao.bacia_hidrografica', 'iqaCalculado', 'parametrosFisicoQuimicos'])
                          ->whereBetween('data_coleta', [$dataInicio, $dataFim])
                          ->get();

        // Estatísticas gerais
        $estatisticas = [
            'total_coletas' => $coletas->count(),
            'iqa_medio' => $coletas->avg(function ($coleta) {
                return $coleta->iqaCalculado->iqa_valor ?? 0;
            }),
            'validadas' => $coletas->where('validado', true)->count(),
        ];

        // Distribuição por classificação
        $distribuicao = $coletas->groupBy(function ($coleta) {
            return $coleta->iqaCalculado->iqa_classificacao ?? 'Sem IQA';
        })->map->count();

        // IQA por estação
        $iqaPorEstacao = $coletas->groupBy('estacao_id')->map(function ($grupo) {
            $estacao = $grupo->first()->estacao;
            $iqaMedio = $grupo->avg(function ($coleta) {
                return $coleta->iqaCalculado->iqa_valor ?? 0;
            });
            $classificacao = $grupo->last()->iqaCalculado->iqa_classificacao ?? 'Sem IQA';

            return [
                'estacao' => $estacao->codigo,
                'bacia' => $estacao->bacia_hidrografica->nome ?? '',
                'iqa' => round($iqaMedio, 2),
                'classificacao' => $classificacao,
            ];
        });

        return view('qa-global.dashboard', compact('estatisticas', 'distribuicao', 'iqaPorEstacao', 'dataInicio', 'dataFim'));
    }
}
