<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;

class TemplateController extends Controller
{
    /**
     * Gerar template de importação para estação
     */
    public function gerarTemplate($tipo, $formato = 'novo')
    {
        // Se for formato existente, retornar template já criado
        if ($formato === 'existente') {
            return $this->downloadTemplateExistente($tipo);
        }
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Configurar cabeçalho baseado no tipo de estação
        $cabecalho = $this->getCabecalhoPorTipo($tipo);
        $exemplos = $this->getExemplosPorTipo($tipo);

        // Definir título
        $sheet->setTitle('Template - ' . ucfirst($tipo));

        // Escrever cabeçalho
        $sheet->fromArray($cabecalho, null, 'A1');

        // Formatar cabeçalho
        $headerRange = 'A1:' . $sheet->getHighestColumn() . '1';
        $sheet->getStyle($headerRange)->getFont()->setBold(true);
        $sheet->getStyle($headerRange)->getFill()
            ->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)
            ->getStartColor()->setRGB('4472C4');
        $sheet->getStyle($headerRange)->getFont()->getColor()->setRGB('FFFFFF');

        // Escrever exemplos
        $linha = 2;
        foreach ($exemplos as $exemplo) {
            $sheet->fromArray($exemplo, null, 'A' . $linha);
            $linha++;
        }

        // Ajustar largura das colunas
        foreach (range('A', $sheet->getHighestColumn()) as $coluna) {
            $sheet->getColumnDimension($coluna)->setAutoSize(true);
        }

        // Adicionar validações e comentários
        $this->adicionarValidacoes($sheet, $tipo);

        // Configurar response para download
        $writer = new Xlsx($spreadsheet);
        $filename = "template-leituras-{$tipo}.xlsx";

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment;filename="' . $filename . '"');
        header('Cache-Control: max-age=0');

        $writer->save('php://output');
        exit;
    }

    /**
     * Obter cabeçalho baseado no tipo de estação
     */
    private function getCabecalhoPorTipo($tipo)
    {
        $basico = ['Data (dd/mm/aaaa)', 'Observações'];

        switch ($tipo) {
            case 'hidrometrica':
                return array_merge([
                    'Data (dd/mm/aaaa)',
                    'Hora 6h (HH:MM)',
                    'Nível 6h (m)',
                    'Hora 12h (HH:MM)',
                    'Nível 12h (m)',
                    'Hora 18h (HH:MM)',
                    'Nível 18h (m)',
                    'Observações'
                ]);

            case 'pluviometrica':
                return array_merge([
                    'Data (dd/mm/aaaa)',
                    'Precipitação (mm)',
                    'Intensidade Máxima (mm/h)',
                    'Duração (horas)',
                    'Tipo de Precipitação',
                    'Observações'
                ]);

            case 'meteorologica':
                return array_merge([
                    'Data (dd/mm/aaaa)',
                    'Precipitação (mm)',
                    'Temperatura Máxima (°C)',
                    'Temperatura Mínima (°C)',
                    'Temperatura Atual (°C)',
                    'Humidade Relativa (%)',
                    'Pressão Atmosférica (hPa)',
                    'Velocidade Vento (km/h)',
                    'Direção Vento',
                    'Evaporação (mm)',
                    'Insolação (horas)',
                    'Observações'
                ]);

            case 'climatologica':
                return array_merge([
                    'Data (dd/mm/aaaa)',
                    'Precipitação (mm)',
                    'Temperatura Máxima (°C)',
                    'Temperatura Mínima (°C)',
                    'Humidade Relativa (%)',
                    'Evaporação (mm)',
                    'Insolação (horas)',
                    'Velocidade Vento (km/h)',
                    'Direção Vento',
                    'Pressão Atmosférica (hPa)',
                    'Observações'
                ]);

            default:
                return $basico;
        }
    }

    /**
     * Obter exemplos baseados no tipo de estação
     */
    private function getExemplosPorTipo($tipo)
    {
        $dataExemplo = date('d/m/Y');
        $dataExemplo2 = date('d/m/Y', strtotime('+1 day'));

        switch ($tipo) {
            case 'hidrometrica':
                return [
                    [$dataExemplo, '06:00', '2.45', '12:00', '2.47', '18:00', '2.43', 'Condições normais'],
                    [$dataExemplo2, '06:00', '2.50', '12:00', '2.52', '18:00', '2.48', 'Leve elevação do nível']
                ];

            case 'pluviometrica':
                return [
                    [$dataExemplo, '15.2', '5.2', '3.5', 'chuva', 'Chuva moderada'],
                    [$dataExemplo2, '0.0', '', '', '', 'Sem precipitação']
                ];

            case 'meteorologica':
                return [
                    [$dataExemplo, '15.2', '28.5', '18.2', '24.1', '65.5', '1013.2', '12.5', 'NE', '4.2', '8.5', 'Condições normais'],
                    [$dataExemplo2, '0.0', '30.1', '19.8', '25.3', '58.3', '1015.1', '8.2', 'E', '5.1', '9.2', 'Céu limpo']
                ];

            case 'climatologica':
                return [
                    [$dataExemplo, '15.2', '28.5', '18.2', '65.5', '4.2', '8.5', '12.5', 'NE', '1013.2', 'Condições normais'],
                    [$dataExemplo2, '0.0', '30.1', '19.8', '58.3', '5.1', '9.2', '8.2', 'E', '1015.1', 'Céu limpo']
                ];

            default:
                return [
                    [$dataExemplo, 'Exemplo de observação'],
                    [$dataExemplo2, 'Outro exemplo']
                ];
        }
    }

    /**
     * Adicionar validações às células
     */
    private function adicionarValidacoes($sheet, $tipo)
    {
        // Validação de data na coluna A (a partir da linha 2)
        $validation = $sheet->getCell('A2')->getDataValidation();
        $validation->setType(\PhpOffice\PhpSpreadsheet\Cell\DataValidation::TYPE_DATE);
        $validation->setErrorStyle(\PhpOffice\PhpSpreadsheet\Cell\DataValidation::STYLE_STOP);
        $validation->setAllowBlank(false);
        $validation->setShowInputMessage(true);
        $validation->setShowErrorMessage(true);
        $validation->setErrorTitle('Data inválida');
        $validation->setError('Por favor, insira uma data válida no formato dd/mm/aaaa');
        $validation->setPromptTitle('Data');
        $validation->setPrompt('Insira a data da leitura no formato dd/mm/aaaa');

        // Aplicar validação para múltiplas linhas (até linha 1000)
        for ($i = 2; $i <= 1000; $i++) {
            $sheet->getCell('A' . $i)->setDataValidation(clone $validation);
        }

        // Validações específicas por tipo
        if ($tipo === 'hidrometrica') {
            // Validação para níveis (colunas C, E, G - devem ser números positivos)
            $niveisColumns = ['C', 'E', 'G'];
            foreach ($niveisColumns as $col) {
                $validationNivel = $sheet->getCell($col . '2')->getDataValidation();
                $validationNivel->setType(\PhpOffice\PhpSpreadsheet\Cell\DataValidation::TYPE_DECIMAL);
                $validationNivel->setOperator(\PhpOffice\PhpSpreadsheet\Cell\DataValidation::OPERATOR_GREATERTHANOREQUALTO);
                $validationNivel->setFormula1(0);
                $validationNivel->setAllowBlank(true);
                $validationNivel->setShowErrorMessage(true);
                $validationNivel->setErrorTitle('Valor inválido');
                $validationNivel->setError('O nível deve ser um número maior ou igual a zero');

                for ($i = 2; $i <= 1000; $i++) {
                    $sheet->getCell($col . $i)->setDataValidation(clone $validationNivel);
                }
            }
        }

        if (in_array($tipo, ['meteorologica', 'climatologica'])) {
            // Validação para humidade (deve estar entre 0 e 100)
            $humidadeCol = $tipo === 'meteorologica' ? 'F' : 'E';
            $validationHumidade = $sheet->getCell($humidadeCol . '2')->getDataValidation();
            $validationHumidade->setType(\PhpOffice\PhpSpreadsheet\Cell\DataValidation::TYPE_DECIMAL);
            $validationHumidade->setOperator(\PhpOffice\PhpSpreadsheet\Cell\DataValidation::OPERATOR_BETWEEN);
            $validationHumidade->setFormula1(0);
            $validationHumidade->setFormula2(100);
            $validationHumidade->setAllowBlank(true);
            $validationHumidade->setShowErrorMessage(true);
            $validationHumidade->setErrorTitle('Humidade inválida');
            $validationHumidade->setError('A humidade deve estar entre 0 e 100%');

            for ($i = 2; $i <= 1000; $i++) {
                $sheet->getCell($humidadeCol . $i)->setDataValidation(clone $validationHumidade);
            }
        }

        // Adicionar comentários explicativos
        $this->adicionarComentarios($sheet, $tipo);
    }

    /**
     * Adicionar comentários explicativos
     */
    private function adicionarComentarios($sheet, $tipo)
    {
        // Comentário na célula A1
        $comentario = $sheet->getComment('A1');
        $comentario->setAuthor('Sistema');
        $comentario->getText()->createTextRun("Template para importação de leituras\n\n");
        $comentario->getText()->createTextRun("Instruções:\n");
        $comentario->getText()->createTextRun("1. Preencha os dados nas linhas abaixo\n");
        $comentario->getText()->createTextRun("2. Use o formato de data dd/mm/aaaa\n");
        $comentario->getText()->createTextRun("3. Use vírgula ou ponto para decimais\n");
        $comentario->getText()->createTextRun("4. Mantenha os nomes das colunas inalterados\n");
        $comentario->getText()->createTextRun("5. Remova as linhas de exemplo antes de importar");

        $comentario->setWidth('300px');
        $comentario->setHeight('200px');
    }

    /**
     * Download template existente
     */
    private function downloadTemplateExistente($tipo)
    {
        $templates = [
            'hidrometrica' => 'template-existente-estacao.xls',
            'pluviometrica' => 'template-existente-pluviometrica.xls',
            'meteorologica' => 'template-existente-completo.xls',
            'climatologica' => 'template-existente-completo.xls'
        ];

        $filename = $templates[$tipo] ?? 'template-existente-estacao.xls';
        $filepath = public_path('templates/' . $filename);

        if (!file_exists($filepath)) {
            abort(404, 'Template não encontrado');
        }

        return response()->download($filepath, "template-{$tipo}-existente.xls");
    }

    /**
     * Obter lista de templates disponíveis
     */
    public function listarTemplates($tipo)
    {
        return response()->json([
            'novo' => [
                'nome' => 'Template Moderno',
                'descricao' => 'Formato otimizado com validações automáticas',
                'url' => route('templates.leituras', ['tipo' => $tipo, 'formato' => 'novo'])
            ],
            'existente' => [
                'nome' => 'Template Atual do Sistema',
                'descricao' => 'Formato já utilizado pelos operadores',
                'url' => route('templates.leituras', ['tipo' => $tipo, 'formato' => 'existente'])
            ]
        ]);
    }
}