<?php
// app/Models/Barragem.php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Barragem extends Model
{
    use HasFactory;

    protected $table = 'barragens';

    protected $fillable = [
        // Identificação
        'nome',
        'codigo',
        'bacia_hidrografica_id',
        'divisao_administrativa_id',
        'provincia',
        'distrito',
        'latitude',
        'longitude',
        
        // Características de Volume
        'volume_maximo',
        'volume_morto',
        'volume_util',
        'capacidade_total',
        
        // Cotas e Níveis
        'cota_maxima',
        'cota_minima',
        'cota_npa',
        'cota_nme',
        'cota_nmc',
        'escala_npa', // Valor na escala/régua correspondente ao NPA (para cálculo: Cota = NPA - (escala_npa - nivel))
        'nivel_descargador',
        
        // Características Físicas
        'area_bacia_hidraulica',
        'area_inundacao',
        'altura_barragem',
        'comprimento_crista',
        
        // Características Hidrológicas
        'rio_principal',
        'afluentes',
        'estacoes_pluviometricas',
        
        // Classificação Técnica
        'tipo_barragem',
        'finalidade_principal',
        'tipo_descargador',
        'classe_risco',
        'categoria_dano',
        
        // Informações de Projeto
        'ano_construcao',
        'empresa_projetista',
        'empresa_construtora',
        'ano_projeto',
        'ano_inicio_construcao',
        'ano_conclusao',
        
        // Características Operacionais
        'sistema_automatizado',
        'telemetria',
        'frequencia_monitoramento',
        'estado',
        'observacoes',
    ];

    protected $casts = [
        // Coordenadas
        'latitude' => 'decimal:8',
        'longitude' => 'decimal:8',
        
        // Volumes
        'volume_maximo' => 'decimal:2',
        'volume_morto' => 'decimal:2',
        'volume_util' => 'decimal:2',
        'capacidade_total' => 'decimal:2',
        
        // Cotas e Níveis
        'cota_maxima' => 'decimal:2',
        'cota_minima' => 'decimal:2',
        'cota_npa' => 'decimal:2',
        'cota_nme' => 'decimal:2',
        'cota_nmc' => 'decimal:2',
        'escala_npa' => 'decimal:2',
        'nivel_descargador' => 'decimal:2',
        
        // Características Físicas
        'area_bacia_hidraulica' => 'decimal:2',
        'area_inundacao' => 'decimal:2',
        'altura_barragem' => 'decimal:2',
        'comprimento_crista' => 'decimal:2',
        
        // Arrays JSON
        'afluentes' => 'array',
        'estacoes_pluviometricas' => 'array',
        
        // Booleanos
        'sistema_automatizado' => 'boolean',
        'telemetria' => 'boolean',
    ];

    public function bacia_hidrografica(): BelongsTo
    {
        return $this->belongsTo(BaciaHidrografica::class);
    }

    public function divisaoAdministrativa(): BelongsTo
    {
        return $this->belongsTo(DivisaoAdministrativa::class);
    }

    public function leituras(): HasMany
    {
        return $this->hasMany(LeituraBarragem::class);
    }

    public function alertas(): HasMany
    {
        return $this->hasMany(Alerta::class);
    }

    public function curvas_volume(): HasMany
    {
        return $this->hasMany(CurvaVolume::class);
    }

    public function curvas_cota_volume(): HasMany
    {
        return $this->hasMany(CurvaVolume::class);
    }

    public function ultima_leitura()
    {
        return $this->hasOne(LeituraBarragem::class)->latestOfMany();
    }

    public function leitura_hoje()
    {
        return $this->hasOne(LeituraBarragem::class)
            ->whereDate('data_leitura', today());
    }

    public function calcularVolumeAtual($cota)
    {
        // Primeiro, tentar usar curva cota-volume se existir
        if ($this->curvas_cota_volume()->count() > 0) {
            $curva_inferior = $this->curvas_cota_volume()
                ->where('cota', '<=', $cota)
                ->orderBy('cota', 'desc')
                ->first();

            $curva_superior = $this->curvas_cota_volume()
                ->where('cota', '>', $cota)
                ->orderBy('cota', 'asc')
                ->first();

            if ($curva_inferior && $curva_superior) {
                // Interpolação linear
                $delta_cota = $curva_superior->cota - $curva_inferior->cota;
                $delta_volume = $curva_superior->volume - $curva_inferior->volume;
                $proporcao = ($cota - $curva_inferior->cota) / $delta_cota;
                return $curva_inferior->volume + ($delta_volume * $proporcao);
            }

            if ($curva_inferior) {
                return $curva_inferior->volume;
            }
        }

        // Fallback: Interpolação linear baseada em NME, NPA, volume_morto e volume_maximo
        // Assumindo relação linear entre cota e volume
        if ($this->cota_nme && $this->cota_npa && $this->volume_maximo) {
            $cota_nme = (float) $this->cota_nme;
            $cota_npa = (float) $this->cota_npa;
            $vol_morto = (float) ($this->volume_morto ?? 0);
            $vol_max = (float) $this->volume_maximo;

            // Se cota abaixo do NME
            if ($cota <= $cota_nme) {
                return $vol_morto;
            }

            // Se cota acima do NPA
            if ($cota >= $cota_npa) {
                return $vol_max;
            }

            // Interpolação linear entre NME e NPA
            $delta_cota = $cota_npa - $cota_nme;
            if ($delta_cota > 0) {
                $delta_volume = $vol_max - $vol_morto;
                $proporcao = ($cota - $cota_nme) / $delta_cota;
                return $vol_morto + ($delta_volume * $proporcao);
            }
        }

        return $this->volume_morto ?? 0;
    }

    public function calcularPercentagemEnchimento($cota)
    {
        // Método baseado nos ficheiros Excel: (volume_atual / volume_total_maximo) * 100
        if ($this->volume_maximo > 0) {
            $ultima_leitura = $this->ultima_leitura;
            if ($ultima_leitura && $ultima_leitura->volume_actual > 0) {
                return min(100, max(0, ($ultima_leitura->volume_actual / $this->volume_maximo) * 100));
            }
        }

        // Se não tem volume atual, usar curvas cota-volume se existirem
        if ($this->curvas_cota_volume()->count() > 0) {
            $volume_atual = $this->calcularVolumeAtual($cota);
            if ($volume_atual > 0 && $this->volume_maximo > 0) {
                return min(100, max(0, ($volume_atual / $this->volume_maximo) * 100));
            }
        }

        return 0;
    }
    
    /**
     * Calcula o volume útil da barragem
     */
    public function getVolumeUtil()
    {
        // Se volume_util estiver preenchido, usar esse valor
        if ($this->volume_util) {
            return $this->volume_util;
        }
        
        // Senão, calcular como volume_maximo - volume_morto
        return ($this->volume_maximo ?? 0) - ($this->volume_morto ?? 0);
    }
    
    /**
     * Obtém a capacidade total da barragem
     */
    public function getCapacidadeTotal()
    {
        return $this->capacidade_total ?? $this->volume_maximo ?? 0;
    }


    /**
     * Accessor para nível de alerta
     * Calcula automaticamente baseado nos níveis de segurança configurados
     * Nível de alerta = 95% do NPA (Nível de Pleno Armazenamento)
     */
    public function getNivelAlertaAttribute()
    {
        // Se cota_npa está definido, usar 95% dele como nível de alerta
        if ($this->cota_npa) {
            return $this->cota_npa * 0.95;
        }

        // Fallback: usar 90% da cota máxima
        if ($this->cota_maxima) {
            return $this->cota_maxima * 0.90;
        }

        return null;
    }

    /**
     * Accessor para nível de emergência
     * Calcula automaticamente baseado nos níveis de segurança configurados
     * Nível de emergência = entre NPA e NMC
     */
    public function getNivelEmergenciaAttribute()
    {
        // Se NMC está definido, usar como nível de emergência
        if ($this->cota_nmc) {
            return $this->cota_nmc;
        }

        // Fallback: usar 95% da cota máxima
        if ($this->cota_maxima) {
            return $this->cota_maxima * 0.95;
        }

        return null;
    }

    public function getStatusNivel($cota)
    {
        // Verificação baseada nos níveis oficiais de segurança de barragens

        // NMC - Nível Máximo de Cheia (mais crítico)
        if ($this->cota_nmc && $cota >= $this->cota_nmc) {
            return 'nmc_excedido'; // Nível Máximo de Cheia excedido
        }

        // Próximo ao NMC (90% do NMC)
        if ($this->cota_nmc && $cota >= ($this->cota_nmc * 0.9)) {
            return 'proximo_nmc'; // Aproximando-se do Nível Máximo de Cheia
        }

        // NPA - Nível de Pleno Armazenamento
        if ($this->cota_npa && $cota >= $this->cota_npa) {
            return 'npa_atingido'; // Nível de Pleno Armazenamento atingido
        }

        // Entre NME e NPA (zona operacional normal)
        if ($this->cota_nme && $this->cota_npa && $cota >= $this->cota_nme && $cota < $this->cota_npa) {
            return 'operacional'; // Faixa operacional normal
        }

        // NME - Nível Mínimo de Exploração
        if ($this->cota_nme && $cota >= $this->cota_nme) {
            return 'nme_atingido'; // Acima do Nível Mínimo de Exploração
        }

        // Abaixo do NME
        if ($this->cota_nme && $cota < $this->cota_nme) {
            return 'abaixo_nme'; // Abaixo do Nível Mínimo de Exploração
        }

        return 'normal'; // Status padrão
    }
    
    /**
     * Obtém informações detalhadas sobre o status atual
     */
    public function getStatusDetalhado($cota = null)
    {
        if (!$cota && $this->ultima_leitura) {
            $cota = $this->ultima_leitura->cota_actual;
        }
        
        if (!$cota) {
            return ['status' => 'desconhecido', 'descricao' => 'Sem dados de cota'];
        }
        
        $status = $this->getStatusNivel($cota);
        
        $descricoes = [
            'nmc_excedido' => 'CRÍTICO: Nível Máximo de Cheia (NMC) excedido',
            'proximo_nmc' => 'ALERTA: Aproximando-se do Nível Máximo de Cheia (NMC)',
            'npa_atingido' => 'ATENÇÃO: Nível Pleno de Armazenamento (NPA) atingido',
            'operacional' => 'NORMAL: Faixa operacional adequada',
            'nme_atingido' => 'OK: Acima do Nível Mínimo de Exploração (NME)',
            'abaixo_nme' => 'BAIXO: Abaixo do Nível Mínimo de Exploração (NME)',
            'normal' => 'Normal',
            'desconhecido' => 'Status não determinado'
        ];
        
        return [
            'status' => $status,
            'descricao' => $descricoes[$status] ?? 'Status indefinido',
            'cota_atual' => $cota
        ];
    }
    
    /**
     * Verifica se a barragem tem sistema automatizado
     */
    public function temSistemaAutomatizado()
    {
        return $this->sistema_automatizado || $this->telemetria;
    }
    
    /**
     * Obtém lista de afluentes formatada
     */
    public function getAfluentesFormatados()
    {
        if (is_array($this->afluentes)) {
            return implode(', ', $this->afluentes);
        }
        
        return $this->afluentes ?? 'Não informado';
    }
    
    /**
     * Obtém explicação dos níveis de segurança em linguagem simples
     */
    public function getNiveisSegurancaExplicacao()
    {
        return [
            'nme' => [
                'nome' => 'Nível Mínimo de Exploração (NME)',
                'descricao' => 'O menor nível para operação normal da barragem',
                'valor' => $this->cota_nme,
                'unidade' => 'm'
            ],
            'npa' => [
                'nome' => 'Nível Pleno de Armazenamento (NPA)',
                'descricao' => 'Nível ideal de armazenamento da barragem',
                'valor' => $this->cota_npa,
                'unidade' => 'm'
            ],
            'nmc' => [
                'nome' => 'Nível Máximo de Cheia (NMC)',
                'descricao' => 'Nível máximo seguro antes de riscos críticos',
                'valor' => $this->cota_nmc,
                'unidade' => 'm'
            ]
        ];
    }

    /**
     * Verifica se todos os níveis de segurança estão configurados
     */
    public function temNiveisSegurancaCompletos()
    {
        return !is_null($this->cota_nme) && !is_null($this->cota_npa) && !is_null($this->cota_nmc);
    }

    /**
     * Obtém a cor do status para interface de usuário
     */
    public function getCorStatus($cota)
    {
        $status = $this->getStatusNivel($cota);

        $cores = [
            'nmc_excedido' => '#ef4444', // Vermelho crítico
            'proximo_nmc' => '#f97316',  // Laranja alerta
            'npa_atingido' => '#eab308', // Amarelo atenção
            'operacional' => '#22c55e',  // Verde normal
            'nme_atingido' => '#3b82f6', // Azul ok
            'abaixo_nme' => '#6b7280',   // Cinza baixo
            'normal' => '#9ca3af'        // Cinza neutro
        ];

        return $cores[$status] ?? '#9ca3af';
    }

    /**
     * Obtém classificação de risco completa
     */
    public function getClassificacaoRisco()
    {
        return [
            'classe_risco' => $this->classe_risco ?? 'medio',
            'categoria_dano' => $this->categoria_dano ?? 'medio',
            'descricao' => $this->getDescricaoRisco()
        ];
    }
    
    private function getDescricaoRisco()
    {
        $classe = $this->classe_risco ?? 'medio';
        $categoria = $this->categoria_dano ?? 'medio';
        
        if ($classe === 'alto' || $categoria === 'alto') {
            return 'Barragem de alto risco - Monitoramento intensivo';
        }
        
        if ($classe === 'medio' || $categoria === 'medio') {
            return 'Barragem de risco moderado - Monitoramento regular';
        }
        
        return 'Barragem de baixo risco - Monitoramento padrão';
    }

    public function getComparacaoAnoAnterior($data_atual)
    {
        $data_ano_anterior = $data_atual->copy()->subYear();
        
        $leitura_ano_anterior = $this->leituras()
            ->whereDate('data_leitura', $data_ano_anterior)
            ->first();

        if (!$leitura_ano_anterior) {
            return null;
        }

        $ultima_leitura = $this->ultima_leitura;
        if (!$ultima_leitura) {
            return null;
        }

        $percentagem_atual = $ultima_leitura->percentagem_enchimento;
        $percentagem_anterior = $leitura_ano_anterior->percentagem_enchimento;

        return $percentagem_atual - $percentagem_anterior;
    }
    
    /**
     * Scope para filtrar por tipo de barragem
     */
    public function scopeByTipo($query, $tipo)
    {
        return $query->where('tipo_barragem', $tipo);
    }
    
    /**
     * Scope para filtrar por finalidade
     */
    public function scopeByFinalidade($query, $finalidade)
    {
        return $query->where('finalidade_principal', $finalidade);
    }
    
    /**
     * Scope para barragens com risco alto
     */
    public function scopeAltoRisco($query)
    {
        return $query->where('classe_risco', 'alto')
                    ->orWhere('categoria_dano', 'alto');
    }
    
    /**
     * Scope para barragens automatizadas
     */
    public function scopeAutomatizadas($query)
    {
        return $query->where('sistema_automatizado', true)
                    ->orWhere('telemetria', true);
    }

    /**
     * Formatar volumes em linguagem simples para usuários
     */
    public function getVolumesFormatados()
    {
        return [
            'volume_maximo' => [
                'valor' => $this->volume_maximo,
                'texto' => $this->volume_maximo ? number_format($this->volume_maximo, 1) . ' Mm³' : 'Não informado',
                'descricao' => 'Volume máximo que a barragem pode armazenar'
            ],
            'volume_util' => [
                'valor' => $this->getVolumeUtil(),
                'texto' => $this->getVolumeUtil() ? number_format($this->getVolumeUtil(), 1) . ' Mm³' : 'Não informado',
                'descricao' => 'Volume disponível para uso (excluindo volume morto)'
            ],
            'volume_morto' => [
                'valor' => $this->volume_morto,
                'texto' => $this->volume_morto ? number_format($this->volume_morto, 1) . ' Mm³' : 'Não informado',
                'descricao' => 'Volume que não pode ser utilizado'
            ],
            'capacidade_total' => [
                'valor' => $this->getCapacidadeTotal(),
                'texto' => $this->getCapacidadeTotal() ? number_format($this->getCapacidadeTotal(), 1) . ' Mm³' : 'Não informado',
                'descricao' => 'Capacidade total da barragem'
            ]
        ];
    }

    /**
     * Converter volume para diferentes unidades se necessário
     */
    public function converterVolume($volume, $unidadeDestino = 'Mm3')
    {
        if (!$volume) return null;

        switch ($unidadeDestino) {
            case 'hm3': // Hectômetros cúbicos (antigo)
                return $volume; // 1 Mm³ = 1 hm³
            case 'm3': // Metros cúbicos
                return $volume * 1000000; // 1 Mm³ = 1.000.000 m³
            case 'litros':
                return $volume * 1000000000; // 1 Mm³ = 1 bilhão de litros
            default:
                return $volume; // Mm³ (padrão)
        }
    }

    /**
     * Explicação simples das unidades para usuários
     */
    public function getExplicacaoUnidades()
    {
        return [
            'simbolo' => 'Mm³',
            'nome_completo' => 'Milhões de metros cúbicos',
            'explicacao' => 'Um milhão de metros cúbicos equivale a 1 bilhão de litros de água',
            'exemplo' => '1 Mm³ = volume de 400 piscinas olímpicas aproximadamente'
        ];
    }
}