<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasOne;

class ColetaQA extends Model
{
    use HasFactory;

    protected $table = 'coletas_qa';

    protected $fillable = [
        'estacao_id',
        'data_coleta',
        'hora_coleta',
        'responsavel',
        'observacoes',
        'validado',
        'validado_por',
        'data_validacao',
    ];

    protected $casts = [
        'data_coleta' => 'date',
        'hora_coleta' => 'datetime',
        'validado' => 'boolean',
        'data_validacao' => 'datetime',
    ];

    protected static function boot()
    {
        parent::boot();

        static::created(function ($coleta) {
            // Criar registros relacionados vazios
            $coleta->parametrosFisicoQuimicos()->create([]);
            $coleta->metaisPesados()->create([]);
        });
    }

    public function estacao(): BelongsTo
    {
        return $this->belongsTo(Estacao::class);
    }

    public function parametrosFisicoQuimicos(): HasOne
    {
        return $this->hasOne(ParametrosFisicoQuimicos::class, 'coleta_id');
    }

    public function metaisPesados(): HasOne
    {
        return $this->hasOne(MetaisPesados::class, 'coleta_id');
    }

    public function iqaCalculado(): HasOne
    {
        return $this->hasOne(IQACalculado::class, 'coleta_id');
    }

    public function validadoPor(): BelongsTo
    {
        return $this->belongsTo(User::class, 'validado_por');
    }

    /**
     * Calcular e salvar IQA
     */
    public function calcularIQA()
    {
        $params = $this->parametrosFisicoQuimicos;

        if (!$params) {
            return null;
        }

        $servico = new \App\Services\IQACalculatorService();
        $resultado = $servico->calcular($params);

        // Salvar ou atualizar IQA
        $this->iqaCalculado()->updateOrCreate(
            ['coleta_id' => $this->id],
            [
                'iqa_valor' => $resultado['iqa'],
                'iqa_classificacao' => $resultado['classificacao'],
                'metodo_calculo' => 'NSF-WQI',
                'detalhes_calculo' => json_encode($resultado['detalhes']),
            ]
        );

        return $resultado['iqa'];
    }

    /**
     * Verificar conformidade com limites
     */
    public function verificarConformidade()
    {
        $alertas = [];
        $params = $this->parametrosFisicoQuimicos;
        $metais = $this->metaisPesados;

        if (!$params && !$metais) {
            return $alertas;
        }

        $limites = LimiteParametro::all()->keyBy('parametro');

        // Verificar parâmetros físico-químicos
        if ($params) {
            $parametrosVerificar = [
                'temperatura' => $params->temperatura,
                'ph' => $params->ph,
                'condutividade_eletrica' => $params->condutividade_eletrica,
                'tds' => $params->tds,
                'nitrato' => $params->nitrato,
                'cloretos' => $params->cloretos,
                'fosfato' => $params->fosfato,
            ];

            foreach ($parametrosVerificar as $nome => $valor) {
                if ($valor !== null && isset($limites[$nome])) {
                    $limite = $limites[$nome];
                    $excede = false;
                    $mensagem = '';

                    if ($limite->limite_minimo !== null && $valor < $limite->limite_minimo) {
                        $excede = true;
                        $mensagem = "Abaixo do limite mínimo ({$limite->limite_minimo} {$limite->unidade})";
                    }

                    if ($limite->limite_maximo !== null && $valor > $limite->limite_maximo) {
                        $excede = true;
                        $mensagem = "Acima do limite máximo ({$limite->limite_maximo} {$limite->unidade})";
                    }

                    if ($excede) {
                        $alertas[] = [
                            'parametro' => ucfirst(str_replace('_', ' ', $nome)),
                            'valor' => $valor,
                            'mensagem' => $mensagem,
                            'severidade' => $this->determinarSeveridade($nome, $valor, $limite),
                        ];
                    }
                }
            }
        }

        // Verificar metais pesados
        if ($metais) {
            $metaisVerificar = [
                'ferro' => $metais->ferro,
                'chumbo' => $metais->chumbo,
                'cadmio' => $metais->cadmio,
                'zinco' => $metais->zinco,
                'cobre' => $metais->cobre,
                'arsenio' => $metais->arsenio,
                'mercurio' => $metais->mercurio,
            ];

            foreach ($metaisVerificar as $nome => $valor) {
                if ($valor !== null && isset($limites[$nome])) {
                    $limite = $limites[$nome];

                    if ($limite->limite_maximo !== null && $valor > $limite->limite_maximo) {
                        $alertas[] = [
                            'parametro' => strtoupper($this->getSimbolo($nome)),
                            'valor' => $valor,
                            'mensagem' => "Acima do limite máximo ({$limite->limite_maximo} {$limite->unidade})",
                            'severidade' => 'alta', // Metais pesados sempre severidade alta
                        ];
                    }
                }
            }
        }

        return $alertas;
    }

    private function determinarSeveridade($parametro, $valor, $limite)
    {
        if (in_array($parametro, ['mercurio', 'chumbo', 'cadmio', 'arsenio'])) {
            return 'alta';
        }

        if ($limite->limite_maximo !== null) {
            $excesso = ($valor - $limite->limite_maximo) / $limite->limite_maximo * 100;
            if ($excesso > 100) return 'alta';
            if ($excesso > 50) return 'media';
        }

        return 'baixa';
    }

    private function getSimbolo($metal)
    {
        $simbolos = [
            'ferro' => 'Fe',
            'chumbo' => 'Pb',
            'cadmio' => 'Cd',
            'zinco' => 'Zn',
            'cobre' => 'Cu',
            'arsenio' => 'As',
            'mercurio' => 'Hg',
        ];

        return $simbolos[$metal] ?? $metal;
    }
}
