<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class DivisaoAdministrativa extends Model
{
    use HasFactory;

    protected $table = 'divisoes_administrativas';

    protected $fillable = [
        'nome',
        'codigo',
        'tipo',
        'parent_id',
        'area_km2',
        'populacao',
        'descricao',
        'limites_geograficos',
        'latitude',
        'longitude',
        'ativo'
    ];

    protected $casts = [
        'limites_geograficos' => 'array',
        'area_km2' => 'decimal:2',
        'latitude' => 'decimal:8',
        'longitude' => 'decimal:8',
        'ativo' => 'boolean',
        'populacao' => 'integer'
    ];

    /**
     * Relação com divisão pai (hierarquia)
     */
    public function parent()
    {
        return $this->belongsTo(DivisaoAdministrativa::class, 'parent_id');
    }

    /**
     * Relação com subdivisões filhas
     */
    public function children()
    {
        return $this->hasMany(DivisaoAdministrativa::class, 'parent_id');
    }

    /**
     * Bacias nesta divisão
     */
    public function bacias()
    {
        return $this->hasMany(Bacia::class, 'divisao_administrativa_id');
    }

    /**
     * Estações nesta divisão
     */
    public function estacoes()
    {
        return $this->hasMany(Estacao::class, 'divisao_administrativa_id');
    }

    /**
     * Barragens nesta divisão
     */
    public function barragens()
    {
        return $this->hasMany(Barragem::class, 'divisao_administrativa_id');
    }

    /**
     * Scopes
     */
    public function scopeAtivo($query)
    {
        return $query->where('ativo', true);
    }

    public function scopeTipo($query, $tipo)
    {
        return $query->where('tipo', $tipo);
    }

    public function scopeProvincias($query)
    {
        return $query->where('tipo', 'provincia');
    }

    public function scopeDistritos($query)
    {
        return $query->where('tipo', 'distrito');
    }

    /**
     * Obter hierarquia completa
     */
    public function getHierarquiaCompleta()
    {
        $hierarquia = [];
        $divisao = $this;

        while ($divisao) {
            array_unshift($hierarquia, $divisao);
            $divisao = $divisao->parent;
        }

        return $hierarquia;
    }

    /**
     * Obter nome completo com hierarquia
     */
    public function getNomeCompletoAttribute()
    {
        $hierarquia = $this->getHierarquiaCompleta();
        return collect($hierarquia)->pluck('nome')->implode(' > ');
    }

    // ============================================
    // SISTEMA DE PERMISSOES REGIONAIS
    // ============================================

    /**
     * Utilizadores atribuidos a esta divisao
     */
    public function users()
    {
        return $this->belongsToMany(
            User::class,
            'user_divisoes',
            'divisao_administrativa_id',
            'user_id'
        )->withPivot([
            'permissoes_customizadas',
            'nivel_acesso',
            'is_primary',
            'valido_de',
            'valido_ate',
            'ativo'
        ])->withTimestamps();
    }

    /**
     * Atribuicoes de utilizadores a esta divisao
     */
    public function userDivisoes()
    {
        return $this->hasMany(UserDivisao::class, 'divisao_administrativa_id');
    }

    /**
     * Permissoes customizadas por role nesta divisao
     */
    public function rolePermissions()
    {
        return $this->hasMany(DivisaoRolePermission::class, 'divisao_administrativa_id');
    }

    /**
     * Obter todos os IDs de ancestrais (para verificar acesso hierarquico)
     */
    public function getAncestorIds(): array
    {
        $ids = [];
        $parent = $this->parent;

        while ($parent) {
            $ids[] = $parent->id;
            $parent = $parent->parent;
        }

        return $ids;
    }

    /**
     * Obter todos os IDs de descendentes recursivamente
     */
    public function getDescendantIds(): array
    {
        $ids = [];

        foreach ($this->children as $child) {
            $ids[] = $child->id;
            $ids = array_merge($ids, $child->getDescendantIds());
        }

        return $ids;
    }

    /**
     * Verificar se esta divisao e ancestral de outra
     */
    public function isAncestorOf(DivisaoAdministrativa $divisao): bool
    {
        return in_array($this->id, $divisao->getAncestorIds());
    }

    /**
     * Verificar se esta divisao e descendente de outra
     */
    public function isDescendantOf(DivisaoAdministrativa $divisao): bool
    {
        return in_array($divisao->id, $this->getAncestorIds());
    }

    /**
     * Obter tipo formatado
     */
    public function getTipoFormatadoAttribute(): string
    {
        $tipos = [
            'provincia' => 'Provincia',
            'distrito' => 'Distrito',
            'municipio' => 'Municipio',
            'setor' => 'Setor',
            'zona' => 'Zona',
        ];

        return $tipos[$this->tipo] ?? ucfirst($this->tipo);
    }
}
