<?php
// app/Models/Estacao.php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\Log;
use Exception;

class Estacao extends Model
{
    use HasFactory;

    protected $table = 'estacoes';

    protected $fillable = [
        'nome',
        'codigo',
        'tipo',
        'bacia_hidrografica_id',
        'divisao_administrativa_id',
        'provincia',
        'distrito',
        'rio',
        'latitude',
        'longitude',
        'altitude',
        'nivel_alerta',
        'cota_alerta',
        'data_instalacao',
        'equipamento',
        'frequencia_leitura',
        'estado',
        'observacoes',
        'trabalha_com_cotas',
        'tipo_entrada_cotas',
        'cota_referencia',
        'formula_cota_tipo',
        'formula_cota_a',
        'formula_cota_b',
        'formula_cota_c',
        'formula_personalizada',
        'seba_station_id',
        'ativa',
    ];

    protected $casts = [
        'latitude' => 'decimal:8',
        'longitude' => 'decimal:8',
        'altitude' => 'decimal:2',
        'nivel_alerta' => 'decimal:2',
        'cota_alerta' => 'decimal:2',
        'data_instalacao' => 'date',
        'trabalha_com_cotas' => 'boolean',
        'cota_referencia' => 'decimal:3',
        'formula_cota_a' => 'decimal:6',
        'formula_cota_b' => 'decimal:6',
        'formula_cota_c' => 'decimal:3',
        'ativa' => 'boolean',
    ];

    public function bacia_hidrografica(): BelongsTo
    {
        return $this->belongsTo(BaciaHidrografica::class);
    }

    public function divisaoAdministrativa(): BelongsTo
    {
        return $this->belongsTo(DivisaoAdministrativa::class);
    }

    public function leituras(): HasMany
    {
        return $this->hasMany(LeituraEstacao::class);
    }

    public function alertas(): HasMany
    {
        return $this->hasMany(Alerta::class);
    }

    public function curvas_descarga(): HasMany
    {
        return $this->hasMany(CurvaDescarga::class);
    }

    public function formulas_caudal(): HasMany
    {
        return $this->hasMany(FormulaCaudal::class);
    }

    public function ultima_leitura()
    {
        return $this->hasOne(LeituraEstacao::class)->latestOfMany();
    }

    public function leitura_hoje()
    {
        return $this->hasOne(LeituraEstacao::class)
            ->whereDate('data_leitura', today());
    }

    public function calcularCaudal($nivel, $data = null)
    {
        $data = $data ?? today();

        // Primeiro, tentar usar fórmulas de caudal customizáveis (novo sistema)
        $formula = $this->formulas_caudal()
            ->validasPara($data)
            ->orderBy('data_inicio', 'desc')
            ->first();

        if ($formula && $formula->isValidaParaNivel($nivel)) {
            return $formula->calcularCaudal($nivel);
        }

        // Fallback: usar sistema antigo de curvas de descarga (compatibilidade)
        return $this->calcularCaudalPorCurva($nivel);
    }

    /**
     * Método antigo mantido para compatibilidade
     */
    private function calcularCaudalPorCurva($nivel)
    {
        $curva_ativa = $this->curvas_descarga()
            ->where('ativo', true)
            ->where('nivel', '<=', $nivel)
            ->orderBy('nivel', 'desc')
            ->first();

        if (!$curva_ativa) {
            return null;
        }

        $curva_superior = $this->curvas_descarga()
            ->where('ativo', true)
            ->where('nivel', '>', $nivel)
            ->orderBy('nivel', 'asc')
            ->first();

        if (!$curva_superior) {
            return $curva_ativa->caudal;
        }

        // Interpolação linear
        $delta_nivel = $curva_superior->nivel - $curva_ativa->nivel;
        $delta_caudal = $curva_superior->caudal - $curva_ativa->caudal;
        $proporcao = ($nivel - $curva_ativa->nivel) / $delta_nivel;

        return $curva_ativa->caudal + ($delta_caudal * $proporcao);
    }

    /**
     * Obter fórmula de caudal ativa atual
     */
    public function getFormulaAtiva($data = null)
    {
        return $this->formulas_caudal()
            ->validasPara($data)
            ->orderBy('data_inicio', 'desc')
            ->first();
    }

    /**
     * Verificar se tem fórmulas de caudal configuradas
     */
    public function temFormulasCaudal()
    {
        return $this->formulas_caudal()->exists();
    }

    public function getStatusNivel($nivel)
    {
        if ($this->nivel_alerta && $nivel >= $this->nivel_alerta) {
            return 'alerta';
        }
        
        return 'normal';
    }

    public function isHidrometrica()
    {
        return $this->tipo === 'hidrometrica';
    }

    public function isPluviometrica()
    {
        return $this->tipo === 'pluviometrica';
    }

    public function isMeteorological()
    {
        return in_array($this->tipo, ['meteorologica', 'climatologica']);
    }

    public function isEvaporimetrica()
    {
        return $this->tipo === 'evaporimetrica';
    }

    public function getTipoDescricaoAmigavel()
    {
        $tipos = [
            'hidrometrica' => 'Hidrométrica - Mede níveis de água',
            'pluviometrica' => 'Pluviométrica - Mede chuva',
            'meteorologica' => 'Meteorológica - Dados climáticos',
            'climatologica' => 'Climatológica - Dados de clima',
            'evaporimetrica' => 'Evaporimétrica - Mede evaporação'
        ];

        return $tipos[$this->tipo] ?? 'Tipo não definido';
    }

    public function getCamposRequeridos()
    {
        switch ($this->tipo) {
            case 'hidrometrica':
                return ['níveis de água', 'cotas (opcional)'];
            case 'pluviometrica':
                return ['precipitação'];
            case 'meteorologica':
            case 'climatologica':
                return ['temperatura', 'humidade', 'precipitação'];
            case 'evaporimetrica':
                return ['evaporação', 'temperatura (opcional)'];
            default:
                return [];
        }
    }

    public function trabalhaComCotas()
    {
        return $this->trabalha_com_cotas;
    }

    public function calculaCotasAutomaticamente()
    {
        return $this->trabalha_com_cotas && $this->tipo_entrada_cotas === 'automatica';
    }

    public function recebeCotasManualmente()
    {
        return $this->trabalha_com_cotas && $this->tipo_entrada_cotas === 'manual';
    }

    public function calcularCotaDeNivel($nivel)
    {
        if (!$this->calculaCotasAutomaticamente() || !$nivel) {
            return null;
        }

        switch ($this->formula_cota_tipo) {
            case 'simples':
                // Cota = Nível + Referência
                return $nivel + ($this->cota_referencia ?? 0);

            case 'linear':
                // Cota = (a × Nível) + b
                $a = $this->formula_cota_a ?? 1;
                $b = $this->formula_cota_b ?? 0;
                return ($a * $nivel) + $b;

            case 'completa':
                // Cota = (a × Nível²) + (b × Nível) + c
                $a = $this->formula_cota_a ?? 0;
                $b = $this->formula_cota_b ?? 1;
                $c = $this->formula_cota_c ?? 0;
                return ($a * $nivel * $nivel) + ($b * $nivel) + $c;

            case 'personalizada':
                // TODO: Implementar avaliador de fórmulas personalizadas
                return $this->avaliarFormulaPersonalizada($nivel);

            default:
                return $nivel + ($this->cota_referencia ?? 0);
        }
    }

    protected function avaliarFormulaPersonalizada($nivel)
    {
        // Implementação básica - pode ser expandida com um parser matemático
        if (!$this->formula_personalizada) {
            return null;
        }

        $formula = str_replace(['N', 'n', 'nivel'], $nivel, $this->formula_personalizada);
        $formula = str_replace(['A', 'a'], $this->formula_cota_a ?? 0, $formula);
        $formula = str_replace(['B', 'b'], $this->formula_cota_b ?? 0, $formula);
        $formula = str_replace(['C', 'c'], $this->formula_cota_c ?? 0, $formula);

        // ATENÇÃO: eval() é perigoso em produção - considere usar uma biblioteca de parsing
        try {
            return eval("return $formula;");
        } catch (Exception $e) {
            Log::error("Erro ao avaliar fórmula personalizada", [
                'estacao_id' => $this->id,
                'formula' => $formula,
                'erro' => $e->getMessage()
            ]);
            return null;
        }
    }

    public function getStatusCota($cota)
    {
        if ($this->cota_alerta && $cota >= $this->cota_alerta) {
            return 'alerta';
        }

        return 'normal';
    }

    /**
     * Obter representação textual da fórmula de cota para usuários
     */
    public function getFormulaTexto()
    {
        if (!$this->trabalhaComCotas() || !$this->calculaCotasAutomaticamente()) {
            return null;
        }

        switch ($this->formula_cota_tipo) {
            case 'simples':
                $ref = number_format($this->cota_referencia ?? 0, 3);
                return "Cota = Nível + {$ref}";
            case 'linear':
                $a = number_format($this->formula_cota_a ?? 1, 3);
                $b = number_format($this->formula_cota_b ?? 0, 3);
                return "Cota = ({$a} × Nível) + {$b}";
            case 'completa':
                $a = number_format($this->formula_cota_a ?? 0, 3);
                $b = number_format($this->formula_cota_b ?? 1, 3);
                $c = number_format($this->formula_cota_c ?? 0, 3);
                return "Cota = ({$a} × Nível²) + ({$b} × Nível) + {$c}";
            case 'personalizada':
                return $this->formula_personalizada ?: 'Fórmula personalizada não definida';
            default:
                return 'Cota = Nível + Referência';
        }
    }
}