<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class QualidadeAgua extends Model
{
    use HasFactory;

    protected $table = 'qualidade_agua';

    protected $fillable = [
        'barragem_id',
        'estacao_id',
        'data_coleta',
        'hora_coleta',
        'oxigenio_dissolvido',
        'coliformes_fecais',
        'ph',
        'dbo5',
        'nitrogenio_total',
        'fosforo_total',
        'temperatura',
        'turbidez',
        'solidos_totais',
        'condutividade',
        'salinidade',
        'clorofila_a',
        'transparencia',
        'iqa',
        'classificacao_iqa',
        'local_coleta',
        'profundidade_coleta',
        'responsavel_coleta',
        'laboratorio',
        'data_analise',
        'observacoes',
        'validado',
        'validado_por',
        'data_validacao',
    ];

    protected $casts = [
        'data_coleta' => 'date',
        'hora_coleta' => 'datetime',
        'data_analise' => 'date',
        'data_validacao' => 'datetime',
        'validado' => 'boolean',
    ];

    // Relacionamentos
    public function barragem()
    {
        return $this->belongsTo(Barragem::class);
    }

    public function estacao()
    {
        return $this->belongsTo(Estacao::class);
    }

    public function validadoPor()
    {
        return $this->belongsTo(User::class, 'validado_por');
    }

    /**
     * Calcula o IQA (Índice de Qualidade de Água) usando o método NSF-WQI
     * Formula: IQA = Π(qi^wi) onde qi são os subíndices e wi os pesos
     */
    public function calcularIQA()
    {
        // Pesos dos parâmetros segundo NSF
        $pesos = [
            'oxigenio_dissolvido' => 0.17,
            'coliformes_fecais' => 0.15,
            'ph' => 0.12,
            'dbo5' => 0.10,
            'nitrogenio_total' => 0.10,
            'fosforo_total' => 0.10,
            'temperatura' => 0.10,
            'turbidez' => 0.08,
            'solidos_totais' => 0.08,
        ];

        $iqa = 1;
        $pesoTotal = 0;

        foreach ($pesos as $parametro => $peso) {
            $valor = $this->$parametro;

            if ($valor !== null) {
                $qi = $this->calcularSubindice($parametro, $valor);
                if ($qi !== null) {
                    $iqa *= pow($qi, $peso);
                    $pesoTotal += $peso;
                }
            }
        }

        // Ajusta se nem todos os parâmetros foram medidos
        if ($pesoTotal > 0 && $pesoTotal < 1) {
            $iqa = pow($iqa, 1 / $pesoTotal);
        } elseif ($pesoTotal == 0) {
            return null;
        }

        $iqaFinal = round($iqa, 2);

        // Atualiza o modelo
        $this->iqa = $iqaFinal;
        $this->classificacao_iqa = $this->classificarIQA($iqaFinal);

        return $iqaFinal;
    }

    /**
     * Calcula o subíndice (qi) para cada parâmetro
     */
    private function calcularSubindice($parametro, $valor)
    {
        switch ($parametro) {
            case 'oxigenio_dissolvido':
                return $this->subindiceOD($valor);
            case 'coliformes_fecais':
                return $this->subindiceColiformes($valor);
            case 'ph':
                return $this->subindicePH($valor);
            case 'dbo5':
                return $this->subindiceDBO($valor);
            case 'nitrogenio_total':
                return $this->subindiceNitrogenio($valor);
            case 'fosforo_total':
                return $this->subindiceFosforo($valor);
            case 'temperatura':
                return $this->subindiceTemperatura($valor);
            case 'turbidez':
                return $this->subindiceTurbidez($valor);
            case 'solidos_totais':
                return $this->subindiceSolidos($valor);
            default:
                return null;
        }
    }

    // Funções de subíndice para cada parâmetro (curvas NSF simplificadas)

    private function subindiceOD($od)
    {
        if ($od >= 100) return 100;
        if ($od >= 90) return 90 + ($od - 90) * 1.0;
        if ($od >= 80) return 80 + ($od - 80) * 1.0;
        if ($od >= 70) return 70 + ($od - 70) * 1.0;
        if ($od >= 50) return 50 + ($od - 50) * 1.0;
        if ($od >= 30) return 30 + ($od - 30) * 1.0;
        if ($od >= 10) return 10 + ($od - 10) * 1.0;
        return $od * 1.0;
    }

    private function subindiceColiformes($cf)
    {
        if ($cf <= 1) return 100;
        if ($cf <= 10) return 95 - log10($cf) * 5;
        if ($cf <= 100) return 90 - (log10($cf) - 1) * 10;
        if ($cf <= 1000) return 80 - (log10($cf) - 2) * 15;
        if ($cf <= 10000) return 65 - (log10($cf) - 3) * 20;
        if ($cf <= 100000) return 45 - (log10($cf) - 4) * 15;
        return max(0, 30 - (log10($cf) - 5) * 10);
    }

    private function subindicePH($ph)
    {
        if ($ph < 2) return 2;
        if ($ph < 3) return 5 + ($ph - 2) * 10;
        if ($ph < 6.5) return 15 + ($ph - 3) * 24.3;
        if ($ph <= 7.5) return 100;
        if ($ph <= 8.5) return 80 + (8.5 - $ph) * 20;
        if ($ph <= 10) return 40 + (10 - $ph) * 26.7;
        if ($ph <= 12) return 10 + (12 - $ph) * 15;
        return 2;
    }

    private function subindiceDBO($dbo)
    {
        if ($dbo <= 1) return 100;
        if ($dbo <= 2) return 90 + (2 - $dbo) * 10;
        if ($dbo <= 4) return 70 + (4 - $dbo) * 10;
        if ($dbo <= 8) return 40 + (8 - $dbo) * 7.5;
        if ($dbo <= 15) return 10 + (15 - $dbo) * 4.3;
        return max(0, 10 - ($dbo - 15) * 0.67);
    }

    private function subindiceNitrogenio($n)
    {
        if ($n <= 0.5) return 100;
        if ($n <= 1) return 90 + (1 - $n) * 20;
        if ($n <= 2) return 70 + (2 - $n) * 20;
        if ($n <= 5) return 40 + (5 - $n) * 10;
        if ($n <= 10) return 10 + (10 - $n) * 6;
        return max(0, 10 - ($n - 10) * 1);
    }

    private function subindiceFosforo($p)
    {
        if ($p <= 0.01) return 100;
        if ($p <= 0.05) return 80 + (0.05 - $p) * 500;
        if ($p <= 0.1) return 60 + (0.1 - $p) * 400;
        if ($p <= 0.5) return 20 + (0.5 - $p) * 100;
        if ($p <= 1) return 5 + (1 - $p) * 30;
        return max(0, 5 - ($p - 1) * 5);
    }

    private function subindiceTemperatura($temp)
    {
        // Variação em relação à temperatura natural (assumindo 25°C como base)
        $delta = abs($temp - 25);
        if ($delta <= 1) return 100;
        if ($delta <= 5) return 90 - ($delta - 1) * 5;
        if ($delta <= 10) return 70 - ($delta - 5) * 4;
        if ($delta <= 15) return 50 - ($delta - 10) * 3;
        return max(0, 35 - ($delta - 15) * 2.33);
    }

    private function subindiceTurbidez($turb)
    {
        if ($turb <= 5) return 100;
        if ($turb <= 10) return 90 + (10 - $turb) * 2;
        if ($turb <= 25) return 70 + (25 - $turb) * 1.33;
        if ($turb <= 50) return 50 + (50 - $turb) * 0.8;
        if ($turb <= 100) return 30 + (100 - $turb) * 0.4;
        return max(0, 30 - ($turb - 100) * 0.3);
    }

    private function subindiceSolidos($st)
    {
        if ($st <= 50) return 100;
        if ($st <= 100) return 80 + (100 - $st) * 0.4;
        if ($st <= 200) return 60 + (200 - $st) * 0.2;
        if ($st <= 300) return 40 + (300 - $st) * 0.2;
        if ($st <= 500) return 20 + (500 - $st) * 0.1;
        return max(0, 20 - ($st - 500) * 0.04);
    }

    /**
     * Classifica o IQA em categorias
     */
    private function classificarIQA($iqa)
    {
        if ($iqa >= 91) return 'excelente';
        if ($iqa >= 71) return 'boa';
        if ($iqa >= 51) return 'media';
        if ($iqa >= 26) return 'ruim';
        return 'pessima';
    }

    /**
     * Retorna a descrição da classificação
     */
    public function getClassificacaoDescricaoAttribute()
    {
        $descricoes = [
            'excelente' => 'Excelente - Água própria para consumo e recreação',
            'boa' => 'Boa - Água adequada para a maioria dos usos',
            'media' => 'Média - Necessita tratamento para consumo',
            'ruim' => 'Ruim - Imprópria para consumo direto',
            'pessima' => 'Péssima - Apenas usos menos exigentes'
        ];

        return $descricoes[$this->classificacao_iqa] ?? 'Não classificado';
    }

    /**
     * Retorna a cor associada à classificação
     */
    public function getClassificacaoCorAttribute()
    {
        $cores = [
            'excelente' => '#0066CC', // Azul escuro
            'boa' => '#00CC66',       // Verde
            'media' => '#FFCC00',     // Amarelo
            'ruim' => '#FF6600',      // Laranja
            'pessima' => '#CC0000'    // Vermelho
        ];

        return $cores[$this->classificacao_iqa] ?? '#999999';
    }

    /**
     * Scope para filtrar por barragem
     */
    public function scopePorBarragem($query, $barragemId)
    {
        return $query->where('barragem_id', $barragemId);
    }

    /**
     * Scope para filtrar por estação
     */
    public function scopePorEstacao($query, $estacaoId)
    {
        return $query->where('estacao_id', $estacaoId);
    }

    /**
     * Scope para filtrar por período
     */
    public function scopePorPeriodo($query, $dataInicio, $dataFim)
    {
        return $query->whereBetween('data_coleta', [$dataInicio, $dataFim]);
    }

    /**
     * Scope para apenas validados
     */
    public function scopeValidados($query)
    {
        return $query->where('validado', true);
    }
}
