<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class SebaIntegracaoLog extends Model
{
    use HasFactory;

    protected $fillable = [
        'estacao_id',
        'tipo',
        'status',
        'mensagem',
        'registros_importados',
        'registros_erro',
        'registros_duplicados',
        'detalhes',
        'iniciado_em',
        'finalizado_em',
        'tempo_execucao',
    ];

    protected $casts = [
        'detalhes' => 'array',
        'iniciado_em' => 'datetime',
        'finalizado_em' => 'datetime',
    ];

    public function estacao(): BelongsTo
    {
        return $this->belongsTo(Estacao::class);
    }

    /**
     * Calcular tempo de execução automaticamente
     */
    protected static function boot()
    {
        parent::boot();

        static::updating(function ($log) {
            if ($log->isDirty('finalizado_em') && $log->iniciado_em && $log->finalizado_em) {
                $log->tempo_execucao = $log->finalizado_em->diffInSeconds($log->iniciado_em);
            }
        });
    }

    /**
     * Scopes
     */
    public function scopeSucesso($query)
    {
        return $query->where('status', 'sucesso');
    }

    public function scopeErro($query)
    {
        return $query->where('status', 'erro');
    }

    public function scopeRecentes($query, $dias = 7)
    {
        return $query->where('created_at', '>=', now()->subDays($dias));
    }

    public function scopePorEstacao($query, $estacaoId)
    {
        return $query->where('estacao_id', $estacaoId);
    }

    /**
     * Formatação de status
     */
    public function getStatusFormatado()
    {
        $status = [
            'executando' => 'Em Execução',
            'sucesso' => 'Sucesso',
            'erro' => 'Erro',
            'parcial' => 'Parcial'
        ];

        return $status[$this->status] ?? $this->status;
    }

    /**
     * Cor do status
     */
    public function getCorStatus()
    {
        $cores = [
            'executando' => 'blue',
            'sucesso' => 'green',
            'erro' => 'red',
            'parcial' => 'yellow'
        ];

        return $cores[$this->status] ?? 'gray';
    }

    /**
     * Formatação do tipo
     */
    public function getTipoFormatado()
    {
        $tipos = [
            'importacao_estacao' => 'Importação Estação',
            'importacao_massa' => 'Importação em Massa',
            'teste_conectividade' => 'Teste Conectividade',
            'sincronizacao' => 'Sincronização'
        ];

        return $tipos[$this->tipo] ?? $this->tipo;
    }

    /**
     * Estatísticas dos logs
     */
    public static function getEstatisticas($dias = 7)
    {
        $query = static::where('created_at', '>=', now()->subDays($dias));

        return [
            'total' => $query->count(),
            'sucesso' => $query->clone()->where('status', 'sucesso')->count(),
            'erro' => $query->clone()->where('status', 'erro')->count(),
            'executando' => $query->clone()->where('status', 'executando')->count(),
            'registros_importados' => $query->clone()->sum('registros_importados'),
            'registros_erro' => $query->clone()->sum('registros_erro'),
            'tempo_medio' => $query->clone()->whereNotNull('tempo_execucao')->avg('tempo_execucao'),
        ];
    }
}
