<?php

namespace App\Models;

use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;
use Spatie\Permission\Traits\HasRoles;

class User extends Authenticatable
{
    use HasApiTokens, HasFactory, Notifiable, HasRoles;

    protected $fillable = [
        'name',
        'email',
        'password',
        'telefone',
        'cargo',
        'departamento',
        'provincia',
        'divisao_principal_id',
        'acesso_global',
        'ativo',
        'notificacao_sms',
        'notificacao_email',
        'notificacao_emergencia',
        'niveis_notificacao',
        'nivel_escalacao',
        'receber_relatorio_diario',
        'horario_relatorio',
    ];

    protected $hidden = [
        'password',
        'remember_token',
    ];

    protected $casts = [
        'email_verified_at' => 'datetime',
        'password' => 'hashed',
        'ativo' => 'boolean',
        'acesso_global' => 'boolean',
        'notificacao_sms' => 'boolean',
        'notificacao_email' => 'boolean',
        'notificacao_emergencia' => 'boolean',
        'niveis_notificacao' => 'array',
        'receber_relatorio_diario' => 'boolean',
        'horario_relatorio' => 'datetime:H:i',
    ];

    public function leituras_barragens()
    {
        return $this->hasMany(LeituraBarragem::class, 'operador_id');
    }

    public function leituras_estacoes()
    {
        return $this->hasMany(LeituraEstacao::class, 'operador_id');
    }

    public function boletins_elaborados()
    {
        return $this->hasMany(Boletim::class, 'elaborado_por');
    }

    public function boletins_aprovados()
    {
        return $this->hasMany(Boletim::class, 'aprovado_por');
    }

    public function validacoes_barragens()
    {
        return $this->hasMany(LeituraBarragem::class, 'validado_por');
    }

    public function notificacoesEnviadas()
    {
        return $this->hasMany(NotificacaoEnviada::class);
    }

    /**
     * Verificar se deve receber notificação de um nível específico
     */
    public function deveReceberNotificacao($nivel)
    {
        if ($nivel === 'emergencia' && $this->notificacao_emergencia) {
            return true;
        }

        if ($this->niveis_notificacao) {
            return in_array($nivel, $this->niveis_notificacao);
        }

        // Default: alertas e emergências
        return in_array($nivel, ['alerta', 'emergencia']);
    }

    /**
     * Obter telefone formatado
     */
    public function getTelefoneFormatado()
    {
        if (!$this->telefone) {
            return null;
        }

        $telefone = preg_replace('/[^0-9+]/', '', $this->telefone);
        
        if (strpos($telefone, '+258') === 0) {
            return $telefone;
        }
        
        if (strlen($telefone) === 9 && $telefone[0] === '8') {
            return '+258' . $telefone;
        }
        
        return $this->telefone;
    }

    /**
     * Verificar se telefone é válido
     */
    public function temTelefoneValido()
    {
        $telefone = $this->getTelefoneFormatado();
        return $telefone && (strpos($telefone, '+258') === 0) && strlen($telefone) === 12;
    }

    /**
     * Obter nível de escalação formatado
     */
    public function getNivelEscalacaoFormatado()
    {
        $niveis = [
            'basico' => 'Básico',
            'supervisor' => 'Supervisor',
            'director' => 'Director'
        ];

        return $niveis[$this->nivel_escalacao] ?? 'Básico';
    }

    /**
     * Verificar se é supervisor ou director
     */
    public function isSupervisor()
    {
        return in_array($this->nivel_escalacao, ['supervisor', 'director']);
    }

    /**
     * Obter estatísticas de notificações do usuário
     */
    public function getEstatisticasNotificacoes($dias = 30)
    {
        return [
            'total' => $this->notificacoesEnviadas()
                ->where('created_at', '>=', now()->subDays($dias))
                ->count(),
            'sms' => $this->notificacoesEnviadas()
                ->where('tipo', 'sms')
                ->where('created_at', '>=', now()->subDays($dias))
                ->count(),
            'email' => $this->notificacoesEnviadas()
                ->where('tipo', 'email')
                ->where('created_at', '>=', now()->subDays($dias))
                ->count(),
            'entregues' => $this->notificacoesEnviadas()
                ->where('status', 'entregue')
                ->where('created_at', '>=', now()->subDays($dias))
                ->count(),
            'falhadas' => $this->notificacoesEnviadas()
                ->where('status', 'falhado')
                ->where('created_at', '>=', now()->subDays($dias))
                ->count(),
        ];
    }

    // ============================================
    // SISTEMA DE PERMISSOES REGIONAIS
    // ============================================

    /**
     * Divisoes atribuidas ao utilizador (many-to-many com dados pivot)
     */
    public function divisoes()
    {
        return $this->belongsToMany(
            DivisaoAdministrativa::class,
            'user_divisoes',
            'user_id',
            'divisao_administrativa_id'
        )->withPivot([
            'permissoes_customizadas',
            'nivel_acesso',
            'is_primary',
            'valido_de',
            'valido_ate',
            'ativo'
        ])->withTimestamps();
    }

    /**
     * Atribuicoes de divisoes activas e validas
     */
    public function divisoesAtivas()
    {
        return $this->hasMany(UserDivisao::class)->validas();
    }

    /**
     * Todas as atribuicoes de divisoes (incluindo inactivas)
     */
    public function todasDivisoes()
    {
        return $this->hasMany(UserDivisao::class);
    }

    /**
     * Divisao principal do utilizador
     */
    public function divisaoPrincipal()
    {
        return $this->belongsTo(DivisaoAdministrativa::class, 'divisao_principal_id');
    }

    /**
     * Verificar se tem acesso global (admin ou flag)
     */
    public function temAcessoGlobal(): bool
    {
        return $this->acesso_global || $this->hasRole('Administrador');
    }

    /**
     * Obter todos os IDs de divisoes acessiveis (incluindo descendentes)
     */
    public function getDivisoesAcessiveisIds(): array
    {
        // Acesso global = todas as divisoes
        if ($this->temAcessoGlobal()) {
            return DivisaoAdministrativa::pluck('id')->toArray();
        }

        $ids = [];

        // Carregar divisoes activas com a relacao divisao
        $divisoesAtivas = $this->divisoesAtivas()->with('divisao.children')->get();

        foreach ($divisoesAtivas as $userDivisao) {
            $divisao = $userDivisao->divisao;
            if (!$divisao) continue;

            $ids[] = $divisao->id;

            // Se nivel_acesso = 'full', incluir todos os descendentes
            if ($userDivisao->nivel_acesso === 'full') {
                $ids = array_merge($ids, $this->getDescendentIds($divisao));
            }
        }

        return array_unique($ids);
    }

    /**
     * Obter IDs de todos os descendentes de uma divisao
     */
    protected function getDescendentIds(DivisaoAdministrativa $divisao): array
    {
        $ids = [];

        foreach ($divisao->children as $child) {
            $ids[] = $child->id;
            $ids = array_merge($ids, $this->getDescendentIds($child));
        }

        return $ids;
    }

    /**
     * Verificar se pode aceder a uma divisao especifica
     */
    public function podeAcessarDivisao($divisaoId): bool
    {
        if ($this->temAcessoGlobal()) {
            return true;
        }

        return in_array($divisaoId, $this->getDivisoesAcessiveisIds());
    }

    /**
     * Verificar permissao no contexto de uma divisao
     */
    public function temPermissaoNaDivisao(string $permission, $divisaoId): bool
    {
        // Acesso global = todas as permissoes
        if ($this->temAcessoGlobal()) {
            return true;
        }

        // Verificar se tem acesso a divisao
        if (!$this->podeAcessarDivisao($divisaoId)) {
            return false;
        }

        // Verificar permissao base do Spatie
        $temPermissaoBase = $this->can($permission);

        // Verificar overrides por divisao/role
        $grant = DivisaoRolePermission::where('divisao_administrativa_id', $divisaoId)
            ->whereIn('role_id', $this->roles->pluck('id'))
            ->whereHas('permission', fn($q) => $q->where('name', $permission))
            ->where('tipo', 'grant')
            ->exists();

        $revoke = DivisaoRolePermission::where('divisao_administrativa_id', $divisaoId)
            ->whereIn('role_id', $this->roles->pluck('id'))
            ->whereHas('permission', fn($q) => $q->where('name', $permission))
            ->where('tipo', 'revoke')
            ->exists();

        // Revoke tem prioridade
        if ($revoke) {
            return false;
        }

        // Grant adiciona permissao
        if ($grant) {
            return true;
        }

        // Verificar permissoes customizadas no user_divisoes
        $userDivisao = $this->divisoesAtivas()
            ->where('divisao_administrativa_id', $divisaoId)
            ->first();

        if ($userDivisao) {
            if ($userDivisao->temPermissaoRevogada($permission)) {
                return false;
            }
            if ($userDivisao->temPermissaoConcedida($permission)) {
                return true;
            }
        }

        return $temPermissaoBase;
    }

    /**
     * Obter nomes das provincias acessiveis
     */
    public function getProvinciasAcessiveis(): array
    {
        if ($this->temAcessoGlobal()) {
            return DivisaoAdministrativa::provincias()->pluck('nome')->toArray();
        }

        $divisaoIds = $this->getDivisoesAcessiveisIds();

        return DivisaoAdministrativa::whereIn('id', $divisaoIds)
            ->where('tipo', 'provincia')
            ->pluck('nome')
            ->toArray();
    }

    /**
     * Obter divisao primaria ou primeira activa
     */
    public function getDivisaoPrimariaAttribute()
    {
        // Primeiro tentar divisao_principal_id
        if ($this->divisao_principal_id && $this->divisaoPrincipal) {
            return $this->divisaoPrincipal;
        }

        // Senao, buscar primeira divisao primaria nas atribuicoes
        $primaria = $this->divisoesAtivas()
            ->where('is_primary', true)
            ->with('divisao')
            ->first();

        if ($primaria) {
            return $primaria->divisao;
        }

        // Fallback: primeira divisao activa
        $primeira = $this->divisoesAtivas()->with('divisao')->first();
        return $primeira?->divisao;
    }
}