<?php

namespace App\Policies;

use App\Models\User;
use App\Models\Barragem;
use App\Services\RegionalAccessService;
use Illuminate\Auth\Access\HandlesAuthorization;

/**
 * Policy para controle de acesso a Barragens
 */
class BarragemPolicy
{
    use HandlesAuthorization;

    protected function getAccessService(User $user): RegionalAccessService
    {
        return new RegionalAccessService($user);
    }

    /**
     * Verificar se pode ver lista de barragens
     */
    public function viewAny(User $user): bool
    {
        return $user->can('ver-dashboard');
    }

    /**
     * Verificar se pode ver uma barragem especifica
     */
    public function view(User $user, Barragem $barragem): bool
    {
        $accessService = $this->getAccessService($user);
        return $accessService->canAccessBarragem($barragem);
    }

    /**
     * Verificar se pode criar barragens
     */
    public function create(User $user): bool
    {
        return $user->can('criar-barragens');
    }

    /**
     * Verificar se pode editar uma barragem
     */
    public function update(User $user, Barragem $barragem): bool
    {
        $accessService = $this->getAccessService($user);

        if (!$accessService->canAccessBarragem($barragem)) {
            return false;
        }

        return $accessService->canPerformAction(
            'editar-barragens',
            $barragem->divisao_administrativa_id
        );
    }

    /**
     * Verificar se pode eliminar uma barragem
     */
    public function delete(User $user, Barragem $barragem): bool
    {
        $accessService = $this->getAccessService($user);

        if (!$accessService->canAccessBarragem($barragem)) {
            return false;
        }

        return $accessService->canPerformAction(
            'eliminar-barragens',
            $barragem->divisao_administrativa_id
        );
    }

    /**
     * Verificar se pode gerir leituras
     */
    public function manageLeituras(User $user, Barragem $barragem): bool
    {
        $accessService = $this->getAccessService($user);

        if (!$accessService->canAccessBarragem($barragem)) {
            return false;
        }

        return $accessService->canPerformAction(
            'gerir-leituras-barragens',
            $barragem->divisao_administrativa_id
        );
    }

    /**
     * Verificar se pode validar leituras
     */
    public function validateLeituras(User $user, Barragem $barragem): bool
    {
        $accessService = $this->getAccessService($user);

        if (!$accessService->canAccessBarragem($barragem)) {
            return false;
        }

        return $accessService->canPerformAction(
            'validar-leituras-barragens',
            $barragem->divisao_administrativa_id
        );
    }
}
