<?php

namespace App\Services;

use App\Models\Boletim;
use App\Models\Barragem;
use App\Models\Estacao;
use App\Models\LeituraBarragem;
use App\Models\LeituraEstacao;
use App\Models\BaciaHidrografica;
use App\Models\QualidadeAgua;
use Barryvdh\DomPDF\Facade\Pdf;
use Carbon\Carbon;
use Illuminate\Support\Facades\Storage;
use PhpOffice\PhpWord\PhpWord;
use PhpOffice\PhpWord\IOFactory;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\BoletimExport;
use App\Models\SystemSetting;
use App\Services\AnaliseHidrologicaService;
use App\Services\GraficoBarragemService;
use App\Services\GraficoEstacaoService;

class BoletimService
{
    protected $analiseService;

    public function __construct()
    {
        $this->analiseService = new AnaliseHidrologicaService();
    }
    public function gerarBoletimHidrologicoRegional($data_inicio, $data_fim)
    {
        $dados = $this->coletarDadosHidrologicoRegional($data_inicio, $data_fim);
        
        $boletim = Boletim::create([
            'tipo' => 'hidrologico_regional',
            'numero_edicao' => '001', // Temporário
            'data_emissao' => now()->toDateString(),
            'hora_emissao' => now()->toTimeString(),
            'periodo_inicio' => $data_inicio,
            'periodo_fim' => $data_fim,
            'situacao_prevalecente' => $this->gerarSituacaoPrevalecente($dados),
            'previsao_hidrologica' => $this->gerarPrevisaoHidrologica($dados),
            'alertas_precaucoes' => $this->gerarAlertasPrecaucoes($dados),
            'dados_completos' => $dados,
            'elaborado_por' => auth()->id() ?? 1,
            'estado' => 'rascunho',
        ]);

        $boletim->gerarNumeroEdicao();
        $boletim->save();

        return $boletim;
    }

    public function gerarBoletimHidrologicoProvincial($data_emissao = null, $provincia = 'Nampula')
    {
        $data_emissao = $data_emissao ?: now()->toDateString();
        $dados = $this->coletarDadosHidrologicoProvincial($data_emissao, $provincia);

        $boletim = Boletim::create([
            'tipo' => 'hidrologico_provincial',
            'numero_edicao' => '001', // Temporário
            'data_emissao' => $data_emissao,
            'hora_emissao' => now()->toTimeString(),
            'situacao_prevalecente' => $this->gerarSituacaoProvincial($dados, $provincia),
            'previsao_hidrologica' => 'Para as próximas 24 horas, não prevê-se alterações significativas do actual cenário hidrológico',
            'alertas_precaucoes' => 'A ARA-Norte, IP, apela à sociedade para a observância de medidas de precaução, evitar a travessia de cursos de água. Apela-se igualmente ao acompanhamento da informação hidrológica disseminada pela ARA Norte, IP, e outras entidades competentes.',
            'dados_completos' => $dados,
            'elaborado_por' => auth()->id() ?? 1,
            'estado' => 'rascunho',
            'provincia' => $provincia,
        ]);

        $boletim->gerarNumeroEdicao();
        $boletim->save();

        return $boletim;
    }

    public function gerarBoletimInformativoBarragens($data_emissao = null)
    {
        $data_emissao = $data_emissao ?: now()->toDateString();
        $dados = $this->coletarDadosBarragens($data_emissao);

        $boletim = Boletim::create([
            'tipo' => 'informativo_barragens',
            'numero_edicao' => '001', // Temporário
            'data_emissao' => $data_emissao,
            'hora_emissao' => now()->toTimeString(),
            'situacao_prevalecente' => $this->gerarSituacaoBarragens($dados),
            'previsao_hidrologica' => 'Nas próximas 24 horas não se prevê alterações significativas dos níveis hidrométricos nas bacias hidrográficas da região.',
            'dados_completos' => $dados,
            'elaborado_por' => auth()->id() ?? 1,
            'estado' => 'rascunho',
        ]);

        $boletim->gerarNumeroEdicao();
        $boletim->save();

        return $boletim;
    }

    public function gerarPDF($boletim)
    {
        $dados = $boletim->dados_completos;

        if ($boletim->tipo === 'hidrologico_regional') {
            $view = 'boletins.templates.regional';
        } elseif ($boletim->tipo === 'hidrologico_provincial') {
            $view = 'boletins.templates.provincial';
        } elseif ($boletim->tipo === 'informativo_barragens') {
            $view = 'boletins.templates.informativo_barragens';
        } else {
            throw new \Exception('Tipo de boletim não suportado para geração de PDF.');
        }

        // Preparar dados do template
        $templateData = $this->prepararDadosTemplate($boletim);

        $pdf = Pdf::loadView($view, $templateData)
            ->setPaper('a4', 'portrait')
            ->set_option('defaultFont', 'DejaVu Sans')
            ->set_option('isHtml5ParserEnabled', true)
            ->set_option('isRemoteEnabled', true);

        $filename = "boletim_{$boletim->tipo}_{$boletim->numero_edicao}_{$boletim->data_emissao->format('Y-m-d')}.pdf";
        $path = "boletins/{$filename}";

        Storage::disk('local')->put($path, $pdf->output());

        $boletim->update(['arquivo_pdf' => $path]);

        return $path;
    }

    public function exportPDF($boletim)
    {
        $templateData = $this->prepararDadosTemplate($boletim);

        if ($boletim->tipo === 'hidrologico_regional') {
            $view = 'boletins.templates.regional';
        } elseif ($boletim->tipo === 'hidrologico_provincial') {
            $view = 'boletins.templates.provincial';
        } elseif ($boletim->tipo === 'informativo_barragens') {
            $view = 'boletins.templates.informativo_barragens';
        } else {
            throw new \Exception('Tipo de boletim não suportado para exportação de PDF.');
        }

        $pdf = Pdf::loadView($view, $templateData)
            ->setPaper('a4', 'portrait')
            ->set_option('defaultFont', 'DejaVu Sans')
            ->set_option('isHtml5ParserEnabled', true)
            ->set_option('isRemoteEnabled', true);

        $filename = "boletim_{$boletim->tipo}_{$boletim->numero_edicao}_{$boletim->data_emissao->format('Y-m-d')}.pdf";

        return $pdf->download($filename);
    }

    public function exportWord($boletim)
    {
        $templateData = $this->prepararDadosTemplate($boletim);

        $phpWord = new PhpWord();
        $section = $phpWord->addSection([
            'paperSize' => 'A4',
            'marginLeft' => 800,
            'marginRight' => 800,
            'marginTop' => 800,
            'marginBottom' => 800,
        ]);

        // Cabeçalho
        $headerTable = $section->addTable([
            'borderSize' => 0,
            'cellMargin' => 50,
        ]);
        $headerRow = $headerTable->addRow();
        $headerRow->addCell(8000)->addText(
            'ARA Norte,IP',
            ['name' => 'Arial', 'size' => 18, 'bold' => true, 'color' => '0066cc']
        );
        $headerRow->addCell(2000)->addText(
            'Logo Ministério',
            ['name' => 'Arial', 'size' => 10, 'color' => '666666'],
            ['alignment' => 'center']
        );

        $section->addTextBreak(1);

        // Título
        if ($boletim->tipo === 'hidrologico_regional') {
            $titulo = "BOLETIM HIDROLÓGICO REGIONAL: Nº {$templateData['numero_edicao']}";
        } elseif ($boletim->tipo === 'hidrologico_provincial') {
            $titulo = "BOLETIM HIDROLÓGICO DA PROVÍNCIA DE NAMPULA: Nº {$templateData['numero_edicao']}";
        } else {
            $titulo = "BOLETIM INFORMATIVO DE BARRAGENS: Nº {$templateData['numero_edicao']}";
        }

        $section->addText(
            $titulo,
            ['name' => 'Arial', 'size' => 14, 'bold' => true],
            ['alignment' => 'center']
        );

        $section->addText(
            'DIA ' . \Carbon\Carbon::parse($templateData['data_emissao'])->format('d.m.Y'),
            ['name' => 'Arial', 'size' => 12, 'bold' => true],
            ['alignment' => 'center']
        );

        $section->addTextBreak(2);

        // Seção 1: Situação Prevalecente
        $section->addText(
            '1. SITUAÇÃO PREVALECENTE',
            ['name' => 'Arial', 'size' => 12, 'bold' => true]
        );

        if ($boletim->tipo === 'hidrologico_provincial') {
            $section->addText(
                'Precipitação registada nas últimas 24 horas nas bacias hidrográfica da província de Nampula:',
                ['name' => 'Arial', 'size' => 11, 'bold' => true]
            );

            $section->addText(
                'Nas últimas 24 horas houve registo de precipitação na rede de observação da região Norte:',
                ['name' => 'Arial', 'size' => 11]
            );

            $section->addText(
                'BACIA HIDROGRÁFICA DO LÚRIO (P-201 Ribáuè: 2.6 mm; P-1156 Malema: 1.2 mm).',
                ['name' => 'Arial', 'size' => 11, 'italic' => true]
            );
        }

        $section->addTextBreak(1);

        // Seção 2: Enchimento das Albufeiras
        $section->addText(
            $boletim->tipo === 'hidrologico_provincial' ? '2. SITUAÇÃO DE ENCHIMENTO DAS ALBUFEIRAS' : '2. ENCHIMENTO DAS PRINCIPAIS ALBUFEIRAS',
            ['name' => 'Arial', 'size' => 12, 'bold' => true]
        );

        foreach ($templateData['barragens'] ?? [] as $barragem) {
            $nome = is_array($barragem) ? ($barragem['nome'] ?? 'N/A') : $barragem->nome;
            $enchimento = is_array($barragem) ? ($barragem['percentual_enchimento'] ?? 0) : $barragem->percentual_enchimento;
            $volume = is_array($barragem) ? ($barragem['volume_atual'] ?? 0) : $barragem->volume_atual;
            $caudal = is_array($barragem) ? ($barragem['caudal_efluente'] ?? 0) : $barragem->caudal_efluente;
            $nivel_atual = is_array($barragem) ? ($barragem['nivel_atual'] ?? 0) : $barragem->nivel_atual;
            $nivel_alerta = is_array($barragem) ? ($barragem['nivel_alerta'] ?? 0) : $barragem->nivel_alerta;

            $texto = "Barragem de {$nome}: ({$enchimento}%, {$volume} Mm³), Qef = {$caudal} m³/s; " .
                    ($nivel_atual > $nivel_alerta ? 'Nível a baixar' : 'Nível estacionário') . ".";

            $section->addText(
                "• " . $texto,
                ['name' => 'Arial', 'size' => 11]
            );
        }

        $section->addTextBreak(1);

        // Seção 3: Previsão Hidrológica
        $section->addText(
            '3. PREVISÃO HIDROLÓGICA' . ($boletim->tipo === 'hidrologico_provincial' ? '' : ' PARA AS PRÓXIMAS 24 HORAS'),
            ['name' => 'Arial', 'size' => 12, 'bold' => true]
        );

        $section->addText(
            $boletim->previsao_hidrologica,
            ['name' => 'Arial', 'size' => 11]
        );

        $section->addTextBreak(1);

        // Seção 4: Alertas e Precauções
        $section->addText(
            '4. ALERTA E PRECAUÇÕES',
            ['name' => 'Arial', 'size' => 12, 'bold' => true]
        );

        $section->addText(
            $boletim->alertas_precaucoes,
            ['name' => 'Arial', 'size' => 11]
        );

        if ($boletim->tipo === 'hidrologico_provincial') {
            $section->addTextBreak(2);

            // Legenda
            $section->addText(
                'Legenda:',
                ['name' => 'Arial', 'size' => 11, 'bold' => true]
            );

            $legendas = [
                'NA: Nível de alerta',
                'S/NA: Sem Nível de alerta',
                'SI: Sem informação',
                'Qef: Vazão efluente'
            ];

            foreach ($legendas as $legenda) {
                $section->addText(
                    $legenda,
                    ['name' => 'Arial', 'size' => 10]
                );
            }
        }

        $section->addTextBreak(3);

        // Assinatura
        $section->addText(
            'O chefe do Departamento dos Recursos Hídricos',
            ['name' => 'Arial', 'size' => 11],
            ['alignment' => 'center']
        );

        $section->addTextBreak(3);

        $section->addText(
            '________________________________',
            ['name' => 'Arial', 'size' => 11],
            ['alignment' => 'center']
        );

        $section->addText(
            $templateData['responsavel_nome'],
            ['name' => 'Arial', 'size' => 11, 'bold' => true],
            ['alignment' => 'center']
        );

        $section->addText(
            "({$templateData['responsavel_cargo']})",
            ['name' => 'Arial', 'size' => 10],
            ['alignment' => 'center']
        );

        // Salvar o arquivo
        $numeroEdicao = str_replace(['/', '\\', ':', '*', '?', '"', '<', '>', '|'], '_', $boletim->numero_edicao);
        $filename = "boletim_{$boletim->tipo}_{$numeroEdicao}_{$boletim->data_emissao->format('Y-m-d')}.docx";

        $objWriter = IOFactory::createWriter($phpWord, 'Word2007');

        // Usar output buffer para capturar o conteúdo
        ob_start();
        $objWriter->save('php://output');
        $wordContent = ob_get_clean();

        return response($wordContent)
            ->header('Content-Type', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document')
            ->header('Content-Disposition', 'attachment; filename="' . $filename . '"');
    }

    public function exportExcel($boletim)
    {
        $numeroEdicao = str_replace(['/', '\\', ':', '*', '?', '"', '<', '>', '|'], '_', $boletim->numero_edicao);
        $filename = "boletim_{$boletim->tipo}_{$numeroEdicao}_{$boletim->data_emissao}.xlsx";

        return Excel::download(new BoletimExport($boletim), $filename);
    }

    private function coletarDadosHidrologicoRegional($data_inicio, $data_fim)
    {
        $dados = [
            'bacias' => [],
            'estacoes_por_bacia' => [],
            'precipitacao_registrada' => false,
            'precipitacao_estacoes' => [],
            'resumo_niveis' => 'Os níveis hidrométricos nas bacias hidrográficas continuam a baixar.',
        ];

        $bacias = BaciaHidrografica::with(['estacoes' => function($query) {
            $query->where('estado', 'activa')->where('tipo', 'hidrometrica');
        }])->get();

        foreach ($bacias as $bacia) {
            $dados['bacias'][$bacia->nome] = [
                'nome' => $bacia->nome,
                'codigo' => $bacia->codigo,
                'estacoes' => [],
            ];

            // Buscar estação pluviométrica da mesma bacia
            $pluvStation = Estacao::where('bacia_hidrografica_id', $bacia->id)
                ->where('tipo', 'pluviometrica')
                ->where('estado', 'activa')
                ->first();

            foreach ($bacia->estacoes as $estacao) {
                $leituras = LeituraEstacao::where('estacao_id', $estacao->id)
                    ->whereBetween('data_leitura', [$data_inicio, $data_fim])
                    ->orderBy('data_leitura', 'desc')
                    ->get();

                $dados['bacias'][$bacia->nome]['estacoes'][] = [
                    'nome' => $estacao->nome,
                    'codigo' => $estacao->codigo,
                    'nivel_alerta' => $estacao->nivel_alerta,
                    'codigo_pluviometrica' => $pluvStation ? $pluvStation->codigo : 'S/P',
                    'leituras' => $leituras->map(function($leitura) {
                        return [
                            'data' => $leitura->data_leitura->format('d/m/y'),
                            'nivel_6h' => $leitura->nivel_6h,
                            'nivel_12h' => $leitura->nivel_12h,
                            'nivel_18h' => $leitura->nivel_18h,
                            'precipitacao' => $leitura->precipitacao_mm,
                        ];
                    })->toArray(),
                ];

                // Verificar se houve precipitação
                if ($leituras->where('precipitacao_mm', '>', 0)->count() > 0) {
                    $dados['precipitacao_registrada'] = true;
                }
            }
        }

        // Coletar dados específicos de precipitação das estações pluviométricas
        $estacoes_precipitacao = Estacao::where('estado', 'activa')
            ->where('tipo', 'pluviometrica')
            ->whereIn('provincia', ['Nampula', 'Cabo Delgado', 'Niassa'])
            ->with(['leituras' => function($query) use ($data_inicio, $data_fim) {
                $query->whereBetween('data_leitura', [$data_inicio, $data_fim])
                    ->where('precipitacao_mm', '>', 0)
                    ->orderBy('data_leitura', 'desc');
            }, 'bacia_hidrografica'])
            ->get();

        foreach ($estacoes_precipitacao as $estacao) {
            foreach ($estacao->leituras as $leitura) {
                $dados['precipitacao_estacoes'][] = [
                    'estacao' => $estacao->nome,
                    'codigo' => $estacao->codigo,
                    'precipitacao' => $leitura->precipitacao_mm,
                    'data' => $leitura->data_leitura->format('d/m/Y'),
                    'bacia' => $estacao->bacia_hidrografica->nome ?? 'N/D',
                ];
                $dados['precipitacao_registrada'] = true;
            }
        }

        // Gerar análise dinâmica dos níveis hidrométricos
        $dados['analise_niveis'] = $this->analisarNiveisHidrometricos($dados);

        // Estruturar dados para exibição detalhada
        $dados['estacoes'] = $this->estruturarDadosEstacoes($dados);

        return $dados;
    }

    private function analisarNiveisHidrometricos($dados)
    {
        // Usar serviço de análise inteligente
        $analise = [
            'texto' => $this->analiseService->gerarAnaliseNiveis()
        ];

        // Verificar estações sem dados para adicionar nota
        $estacoes_sem_dados = [];
        $cabo_delgado_sem_dados = 0;

        foreach ($dados['bacias'] ?? [] as $bacia) {
            foreach ($bacia['estacoes'] ?? [] as $estacao) {
                if (empty($estacao['leituras'])) {
                    $estacoes_sem_dados[] = $estacao['nome'];
                    if (stripos($estacao['nome'], 'pemba') !== false ||
                        stripos($estacao['nome'], 'palma') !== false ||
                        stripos($estacao['nome'], 'montaquez') !== false) {
                        $cabo_delgado_sem_dados++;
                    }
                }
            }
        }

        // Adicionar nota sobre comunicações se há estações sem dados
        if (count($estacoes_sem_dados) > 0) {
            if ($cabo_delgado_sem_dados > 0) {
                $analise['texto'] .= " Importa referir que para a província de Cabo Delgado enfrenta-se dificuldades na obtenção de dados, dificultando assim o fluxo de informação.";
            } else {
                $analise['texto'] .= " Algumas estações da região não reportaram dados para o período em análise.";
            }
        }

        return $analise;
    }

    private function estruturarDadosEstacoes($dados)
    {
        $estacoes_estruturadas = [];

        // Adicionar estações de precipitação
        foreach ($dados['precipitacao_estacoes'] ?? [] as $estacao) {
            $estacoes_estruturadas[] = [
                'nome' => $estacao['estacao'],
                'codigo' => $estacao['codigo'],
                'tipo' => 'pluviometrica',
                'ultima_leitura' => $estacao['data'],
                'precipitacao_mm' => $estacao['precipitacao'],
                'valor_leitura' => $estacao['precipitacao'] . ' mm',
                'estado' => 'activa',
                'bacia' => $estacao['bacia']
            ];
        }

        // Adicionar estações hidrométricas das bacias
        foreach ($dados['bacias'] ?? [] as $bacia) {
            foreach ($bacia['estacoes'] ?? [] as $estacao) {
                if (!empty($estacao['leituras'])) {
                    $ultima_leitura = collect($estacao['leituras'])->first();
                    $nivel = $ultima_leitura['nivel_18h'] ?? $ultima_leitura['nivel_12h'] ?? $ultima_leitura['nivel_6h'];

                    $estacoes_estruturadas[] = [
                        'nome' => $estacao['nome'],
                        'codigo' => $estacao['codigo'],
                        'tipo' => 'hidrometrica',
                        'ultima_leitura' => $ultima_leitura['data'],
                        'nivel_hidrometrico' => $nivel,
                        'valor_leitura' => $nivel ? $nivel . ' m' : 'N/A',
                        'estado' => 'activa',
                        'bacia' => $bacia['nome']
                    ];
                } else {
                    $estacoes_estruturadas[] = [
                        'nome' => $estacao['nome'],
                        'codigo' => $estacao['codigo'],
                        'tipo' => 'hidrometrica',
                        'ultima_leitura' => 'Sem dados',
                        'nivel_hidrometrico' => null,
                        'valor_leitura' => 'N/A',
                        'estado' => 'sem_dados',
                        'bacia' => $bacia['nome']
                    ];
                }
            }
        }

        return $estacoes_estruturadas;
    }

    private function coletarDadosBarragens($data_emissao)
    {
        $dados = [
            'barragens' => [],
            'data_emissao' => $data_emissao,
            'tem_dados' => false,
        ];

        // Coletar todas as barragens ativas
        $barragens = Barragem::where('estado', 'activa')
            ->with('bacia_hidrografica')
            ->get();

        foreach ($barragens as $barragem) {
            // Buscar as 3 leituras mais recentes diretamente (limit não funciona no eager loading)
            $leituras = LeituraBarragem::where('barragem_id', $barragem->id)
                ->whereDate('data_leitura', '<=', $data_emissao)
                ->orderBy('data_leitura', 'desc')
                ->orderBy('hora_leitura', 'desc')
                ->take(3)
                ->get();

            $leitura_recente = $leituras->first();

            if ($leitura_recente) {
                // Determinar status do nível
                $status_nivel = 'normal';
                if ($leitura_recente->cota_actual >= $barragem->nivel_emergencia) {
                    $status_nivel = 'emergencia';
                } elseif ($leitura_recente->cota_actual >= $barragem->nivel_alerta) {
                    $status_nivel = 'alerta';
                }

                // Obter estações pluviométricas vinculadas à barragem
                $estacoes_pluv = $barragem->estacoes_pluviometricas ?? [];
                if (is_string($estacoes_pluv)) {
                    $estacoes_pluv = json_decode($estacoes_pluv, true) ?? [];
                }

                // Preparar array de leituras (3 datas)
                $leituras_array = $leituras->map(function($leitura) use ($estacoes_pluv) {
                    // Buscar precipitações das estações vinculadas para esta data
                    $precipitacoes = [];
                    if (!empty($estacoes_pluv)) {
                        $estacoes = \App\Models\Estacao::whereIn('codigo', $estacoes_pluv)->get();
                        foreach ($estacoes as $estacao) {
                            $leitura_estacao = \App\Models\LeituraEstacao::where('estacao_id', $estacao->id)
                                ->whereDate('data_leitura', $leitura->data_leitura)
                                ->first();
                            $precipitacoes[$estacao->codigo] = $leitura_estacao->precipitacao_mm ?? $leitura_estacao->precipitacao ?? 0;
                        }
                    }

                    return [
                        'data' => $leitura->data_leitura->format('d/m/Y'),
                        'hora' => $leitura->hora_leitura ? $leitura->hora_leitura->format('H \h') : '6 h',
                        'cota' => $leitura->cota_actual,
                        'nivel_hidro' => $leitura->nivel_hidrometrico ?? $leitura->cota_actual,
                        'caudal_afluente' => $leitura->caudal_afluente ?? 0,
                        'caudal_efluente' => $leitura->caudal_efluente ?? 0,
                        'volume_total' => $leitura->volume_actual ?? 0,
                        'percentagem_total' => $leitura->percentagem_enchimento ?? 0,
                        'volume_morto' => $leitura->volume_morto ?? 0,
                        'volume_util' => $leitura->volume_util ?? 0,
                        'percentagem_util' => $leitura->percentagem_util ?? 0,
                        'cota_fipa' => $leitura->cota_fipa ?? 0,
                        'precipitacao_p1' => $leitura->precipitacao_p1 ?? $leitura->precipitacao ?? 0,
                        'precipitacao' => $leitura->precipitacao_p1 ?? $leitura->precipitacao ?? 0,
                        'precipitacoes' => $precipitacoes,
                    ];
                })->toArray();

                $dados['barragens'][] = [
                    'id' => $barragem->id,
                    'nome' => $barragem->nome,
                    'codigo' => $barragem->codigo,
                    'cota_npa' => $barragem->cota_npa,
                    'cota_nme' => $barragem->cota_nme,
                    'volume_armazenamento' => $barragem->capacidade_total ?? 0,
                    'cota' => $leitura_recente->cota_actual,
                    'cota_actual' => $leitura_recente->cota_actual,
                    'enchimento' => $leitura_recente->percentagem_enchimento,
                    'percentagem_enchimento' => $leitura_recente->percentagem_enchimento,
                    'volume_atual' => $leitura_recente->volume_actual,
                    'volume_actual' => $leitura_recente->volume_actual,
                    'comparacao_ano_anterior' => $leitura_recente->comparacao_ano_anterior ?? 'N/D',
                    'data_leitura' => $leitura_recente->data_leitura->format('d/m/Y'),
                    'hora_leitura' => $leitura_recente->hora_leitura ? $leitura_recente->hora_leitura->format('H:i') : '06:00',
                    'caudal_ecologico' => $leitura_recente->caudal_ecologico,
                    'caudal_afluente' => $leitura_recente->caudal_afluente ?? 0,
                    'caudal_efluente' => $leitura_recente->caudal_efluente ?? 0,
                    'descarga_total' => $leitura_recente->descarga_total,
                    'descarga_superficie' => $leitura_recente->descarga_superficie ?? 0,
                    'descarga_fundo' => $leitura_recente->descarga_fundo ?? 0,
                    'precipitacao' => $leitura_recente->precipitacao ?? 0,
                    'nivel_alerta' => $barragem->nivel_alerta,
                    'nivel_emergencia' => $barragem->nivel_emergencia,
                    'status_nivel' => $status_nivel,
                    'bacia' => $barragem->bacia_hidrografica->nome ?? 'N/D',
                    'leituras' => $leituras_array, // Array com 3 leituras
                ];
                $dados['tem_dados'] = true;
            } else {
                // Incluir barragem sem dados recentes mas informar status
                $dados['barragens'][] = [
                    'id' => $barragem->id,
                    'nome' => $barragem->nome,
                    'codigo' => $barragem->codigo,
                    'cota_npa' => $barragem->cota_npa,
                    'cota_nme' => $barragem->cota_nme,
                    'cota' => null,
                    'enchimento' => null,
                    'volume_atual' => null,
                    'comparacao_ano_anterior' => 'Sem dados',
                    'data_leitura' => 'N/D',
                    'hora_leitura' => 'N/D',
                    'caudal_ecologico' => null,
                    'descarga_total' => null,
                    'nivel_alerta' => $barragem->nivel_alerta,
                    'bacia' => $barragem->bacia_hidrografica->nome ?? 'N/D',
                    'leituras' => [],
                ];
            }
        }

        return $dados;
    }

    private function gerarSituacaoPrevalecente($dados)
    {
        $situacao = "**Precipitação**\n\n";

        if ($dados['precipitacao_registrada']) {
            $situacao .= "Na rede regional de observação hidroclimatológica foram registadas precipitações em algumas estações.\n\n";
        } else {
            $situacao .= "Na rede regional de observação hidroclimatológica não houve registo de precipitação.\n\n";
        }

        $situacao .= "**Níveis Hidrométricos**\n\n";

        // Usar análise inteligente
        $situacao .= $this->analiseService->gerarResumoGeral();

        return $situacao;
    }

    private function gerarSituacaoBarragens($dados)
    {
        $situacao = "**Níveis Hidrométricos**\n\n";

        if (!empty($dados['barragens']) && $dados['tem_dados']) {
            $situacao .= "Situação atual das principais barragens da região:\n\n";

            foreach ($dados['barragens'] as $barragem) {
                if ($barragem['enchimento'] !== null) {
                    $situacao .= "• **{$barragem['nome']}**: {$barragem['enchimento']}% de enchimento";
                    if ($barragem['volume_atual']) {
                        $situacao .= " ({$barragem['volume_atual']} Mm³)";
                    }
                    if ($barragem['caudal_ecologico']) {
                        $situacao .= ", Qef = {$barragem['caudal_ecologico']} m³/s";
                    }
                    $situacao .= ".\n";
                } else {
                    $situacao .= "• **{$barragem['nome']}**: Sem dados disponíveis para o período.\n";
                }
            }
        } else {
            $situacao .= "Não há dados disponíveis de barragens para o período em análise.";
        }

        return $situacao;
    }

    private function gerarPrevisaoHidrologica($dados)
    {
        // Usar análise inteligente baseada nos dados reais
        $previsao = $this->analiseService->gerarPrevisao();

        // Adicionar informação sobre estação climatológica
        $mes_atual = now()->month;
        if ($mes_atual >= 11 || $mes_atual <= 4) {
            $previsao .= "\n\nNota: Estamos na época chuvosa (Novembro-Abril), período de maior atenção hidrológica na região.";
        } else {
            $previsao .= "\n\nNota: Estamos na época seca (Maio-Outubro), período de menor atividade hidrológica na região.";
        }

        return $previsao;
    }

    private function gerarAlertasPrecaucoes($dados)
    {
        // Usar análise inteligente para gerar alertas baseados em dados reais
        $textoBase = $this->analiseService->gerarTextoPrecaucoes();

        $alertas = [];

        // Verificar se há precipitação significativa
        if ($dados['precipitacao_registrada'] ?? false) {
            $alertas[] = "Devido à precipitação registrada, recomenda-se atenção redobrada na travessia de cursos de água.";
        }

        // Verificar época do ano
        $mes_atual = now()->month;
        if ($mes_atual >= 11 || $mes_atual <= 4) {
            $alertas[] = "Durante a época chuvosa, mantenha-se informado sobre as condições hidrológicas através dos canais oficiais.";
        }

        // Adicionar texto gerado pela análise
        $alertas[] = $textoBase;

        return implode("\n\n", $alertas);
    }

    private function coletarDadosHidrologicoProvincial($data_emissao, $provincia = 'Nampula')
    {
        $dados = [
            'precipitacao_estacoes' => [],
            'barragens' => [],
            'data_emissao' => $data_emissao,
            'provincia' => $provincia,
            'tem_dados' => false,
        ];

        // Coletar dados reais de precipitação das estações DA PROVÍNCIA ESPECÍFICA
        $estacoes = Estacao::where('estado', 'activa')
            ->where('tipo', 'pluviometrica')
            ->where('provincia', $provincia)
            ->with(['leituras' => function($query) use ($data_emissao) {
                $query->whereDate('data_leitura', $data_emissao)
                    ->where('precipitacao_mm', '>', 0);
            }])
            ->get();

        foreach ($estacoes as $estacao) {
            foreach ($estacao->leituras as $leitura) {
                $dados['precipitacao_estacoes'][] = [
                    'estacao' => $estacao->nome,
                    'codigo' => $estacao->codigo,
                    'precipitacao' => $leitura->precipitacao_mm,
                    'data' => $leitura->data_leitura->format('d/m/Y'),
                    'bacia' => $estacao->bacia_hidrografica->nome ?? 'N/D',
                ];
                $dados['tem_dados'] = true;
            }
        }

        // Coletar dados reais das barragens DA PROVÍNCIA ESPECÍFICA
        $barragens = Barragem::where('estado', 'activa')
            ->where('provincia', $provincia)
            ->with(['ultima_leitura', 'bacia_hidrografica'])
            ->get();

        foreach ($barragens as $barragem) {
            $ultima_leitura = $barragem->ultima_leitura;

            if ($ultima_leitura) {
                // Calcular caudal efluente total conforme especificação ARA Norte
                // Qef = caudal_efluente + descarga_total (se não houver caudal_efluente, usar caudal_ecologico)
                $caudal_efluente = $ultima_leitura->caudal_efluente ?? $ultima_leitura->caudal_ecologico ?? 0;

                $dados['barragens'][] = [
                    'nome' => $barragem->nome,
                    'codigo' => $barragem->codigo,
                    'percentual_enchimento' => $ultima_leitura->percentagem_enchimento,
                    'volume_atual' => $ultima_leitura->volume_actual,
                    'caudal_efluente' => $caudal_efluente,
                    'caudal_ecologico' => $ultima_leitura->caudal_ecologico ?? 0,
                    'caudal_afluente' => $ultima_leitura->caudal_afluente ?? 0,
                    'descarga_total' => $ultima_leitura->descarga_total ?? 0,
                    'descarga_fundo' => $ultima_leitura->descarga_fundo ?? 0,
                    'descarga_superficie' => $ultima_leitura->descarga_superficie ?? 0,
                    'nivel_atual' => $ultima_leitura->cota_actual,
                    'nivel_alerta' => $barragem->nivel_alerta,
                    'nivel_emergencia' => $barragem->nivel_emergencia ?? $barragem->cota_nmc,
                    'data_leitura' => $ultima_leitura->data_leitura->format('d/m/Y'),
                    'hora_leitura' => $ultima_leitura->hora_leitura ? $ultima_leitura->hora_leitura->format('H:i') : '00:00',
                    'bacia' => $barragem->bacia_hidrografica->nome ?? 'N/D',
                ];
                $dados['tem_dados'] = true;
            }
        }

        // Adicionar análise de níveis hidrométricos
        $dados['analise_niveis'] = $this->analisarNiveisProvincial($dados);

        return $dados;
    }

    private function gerarSituacaoProvincial($dados, $provincia = 'Nampula')
    {
        $situacao = "**Precipitação registada nas últimas 24 horas nas bacias hidrográficas da província de {$provincia}:**\n\n";

        if (!empty($dados['precipitacao_estacoes'])) {
            $situacao .= "Nas últimas 24 horas houve registo de precipitação na rede de observação da província de {$provincia}:\n\n";

            // Agrupar por bacia
            $precipitacao_por_bacia = [];
            foreach ($dados['precipitacao_estacoes'] as $registro) {
                $bacia = $registro['bacia'];
                if (!isset($precipitacao_por_bacia[$bacia])) {
                    $precipitacao_por_bacia[$bacia] = [];
                }
                $precipitacao_por_bacia[$bacia][] = $registro;
            }

            foreach ($precipitacao_por_bacia as $bacia => $registros) {
                $situacao .= "**BACIA HIDROGRÁFICA " . strtoupper($bacia) . "** (";
                $estacoes_texto = [];
                foreach ($registros as $registro) {
                    $estacoes_texto[] = "{$registro['codigo']} {$registro['estacao']}: {$registro['precipitacao']} mm";
                }
                $situacao .= implode('; ', $estacoes_texto) . ").\n\n";
            }
        } else {
            $situacao .= "Nas últimas 24 horas não houve registo de precipitação na rede de observação da província de {$provincia}.\n\n";
        }

        $situacao .= "**Comportamento dos níveis hidrométricos nas principais bacias da província:**\n\n";

        if (!empty($dados['barragens'])) {
            $situacao .= "Nas últimas 24h observou-se comportamento variado dos níveis nas barragens da província de {$provincia}:\n\n";

            foreach ($dados['barragens'] as $barragem) {
                $tendencia = $barragem['nivel_atual'] > $barragem['nivel_alerta'] ? 'nível elevado' : 'nível normal';
                $situacao .= "• **{$barragem['nome']}**: {$barragem['percentual_enchimento']}% de enchimento, {$tendencia}.\n";
            }
        } else {
            $situacao .= "Não há dados disponíveis de níveis hidrométricos para o período em análise na província de {$provincia}.";
        }

        return $situacao;
    }

    private function prepararDadosTemplate($boletim)
    {
        $dados = $boletim->dados_completos;

        // Get system settings
        $settings = SystemSetting::all_settings();

        $templateData = [
            'boletim' => $boletim,
            'numero_edicao' => $boletim->numero_edicao,
            'data_emissao' => $boletim->data_emissao,
            'dados' => $dados,
            'responsavel_nome' => $settings['responsible_person_name'] ?? 'João Alberto Miguel Sitoe',
            'responsavel_cargo' => $settings['responsible_person_title'] ?? 'Técnico Superior N1',
            'settings' => $settings,
        ];

        if ($boletim->tipo === 'hidrologico_regional') {
            $templateData['estacoes'] = $this->gerarListaEstacoes();
            $templateData['barragens'] = $this->prepararBarragensRegional($dados);

            // Gerar gráficos com protecção contra timeout
            try {
                set_time_limit(120); // 2 minutos para gráficos
                $graficoService = new GraficoBarragemService();
                $templateData['graficos'] = $graficoService->gerarGraficosParaBoletim($templateData['barragens']);
            } catch (\Exception $e) {
                \Log::error("Erro ao gerar gráficos de barragens: " . $e->getMessage());
                $templateData['graficos'] = [];
            }

            try {
                $graficoEstacaoService = new GraficoEstacaoService();
                $templateData['graficos_estacoes'] = $graficoEstacaoService->gerarGraficosParaBoletim($dados['bacias'] ?? []);
            } catch (\Exception $e) {
                \Log::error("Erro ao gerar gráficos de estações: " . $e->getMessage());
                $templateData['graficos_estacoes'] = [];
            }
        } elseif ($boletim->tipo === 'hidrologico_provincial') {
            $templateData['barragens'] = $dados['barragens'] ?? [];
            // Adicionar análise de níveis se não existir nos dados
            if (!isset($dados['analise_niveis'])) {
                $templateData['dados']['analise_niveis'] = $this->analisarNiveisProvincial($dados);
            }
        } elseif ($boletim->tipo === 'informativo_barragens') {
            // Gerar gráficos para as barragens
            $graficoService = new GraficoBarragemService();
            $templateData['graficos'] = $graficoService->gerarGraficosParaBoletim($dados['barragens'] ?? []);
        }

        return $templateData;
    }

    private function analisarNiveisProvincial($dados)
    {
        // Usar serviço de análise inteligente
        return [
            'texto' => $this->analiseService->gerarAnaliseNiveis()
        ];
    }

    private function gerarListaEstacoes()
    {
        // Buscar estações ativas da base de dados
        $estacoes = Estacao::where('estado', 'activa')
            ->whereIn('provincia', ['Nampula', 'Cabo Delgado', 'Niassa'])
            ->orderBy('nome')
            ->get(['id', 'nome', 'codigo', 'tipo', 'provincia', 'bacia_hidrografica_id'])
            ->map(function ($estacao) {
                return [
                    'nome' => $estacao->nome,
                    'codigo' => $estacao->codigo,
                    'tipo' => $estacao->tipo,
                    'provincia' => $estacao->provincia,
                ];
            })
            ->toArray();

        // Se não houver estações na BD, retornar lista vazia
        if (empty($estacoes)) {
            return [];
        }

        return $estacoes;
    }

    private function prepararBarragensRegional($dados)
    {
        // Coletar dados reais das barragens da região
        $barragens = Barragem::where('estado', 'activa')
            ->with('ultima_leitura')
            ->get();

        $barragens_preparadas = [];

        foreach ($barragens as $barragem) {
            $leitura_recente = $barragem->ultima_leitura;

            // Dados da barragem (capacidades)
            $cap_total = $barragem->capacidade_total ?? $barragem->volume_maximo ?? 0;
            $cap_util = $barragem->volume_util ?? 0;
            $cota_npa = $barragem->cota_npa ?? 0;
            $volume_morto = $barragem->volume_morto ?? 0;

            if ($leitura_recente) {
                $volume_actual = $leitura_recente->volume_actual ?? 0;
                $volume_util_actual = max(0, $volume_actual - $volume_morto);

                $barragens_preparadas[] = (object)[
                    'id' => $barragem->id,
                    'nome' => $barragem->nome,
                    'cota_npa' => $cota_npa,
                    'cap_total' => $cap_total,
                    'cap_util' => $cap_util,
                    'nivel_hidrometrico' => $leitura_recente->nivel_hidrometrico ?? null,
                    'cota_actual' => $leitura_recente->cota_actual ?? 0,
                    'volume_total' => $volume_actual,
                    'percentagem_total' => $leitura_recente->percentagem_total ?? 0,
                    'volume_util_actual' => $volume_util_actual,
                    'percentual_enchimento' => $leitura_recente->percentagem_enchimento ?? 0,
                    'caudal_ecologico' => $leitura_recente->caudal_ecologico ?? 0,
                    // Campos legados para compatibilidade com narrativa
                    'volume_atual' => $volume_actual,
                    'caudal_efluente' => $leitura_recente->caudal_ecologico ?? 0,
                    'nivel_atual' => $leitura_recente->cota_actual ?? 0,
                    'nivel_alerta' => $barragem->nivel_alerta,
                    'data_leitura' => $leitura_recente->data_leitura->format('d/m/Y'),
                    'descarga_total' => $leitura_recente->descarga_total ?? 0,
                    'comparacao_ano_anterior' => $leitura_recente->comparacao_ano_anterior ?? null,
                ];
            } else {
                $barragens_preparadas[] = (object)[
                    'id' => $barragem->id,
                    'nome' => $barragem->nome,
                    'cota_npa' => $cota_npa,
                    'cap_total' => $cap_total,
                    'cap_util' => $cap_util,
                    'nivel_hidrometrico' => null,
                    'cota_actual' => 0,
                    'volume_total' => 0,
                    'percentagem_total' => 0,
                    'volume_util_actual' => 0,
                    'percentual_enchimento' => 0,
                    'caudal_ecologico' => 0,
                    'volume_atual' => 0,
                    'caudal_efluente' => 0,
                    'nivel_atual' => 0,
                    'nivel_alerta' => $barragem->nivel_alerta,
                    'data_leitura' => 'N/D',
                    'descarga_total' => 0,
                    'comparacao_ano_anterior' => null,
                ];
            }
        }

        // Se não há barragens com dados, retornar array vazio para evitar erro no template
        return $barragens_preparadas;
    }

    /**
     * Coletar dados de qualidade de água para inclusão no boletim
     */
    private function coletarDadosQualidadeAgua($data_inicio, $data_fim)
    {
        $dados = [
            'qualidade_agua' => [],
            'estatisticas_iqa' => [],
            'tem_dados' => false,
        ];

        // Coletar dados de qualidade de água do período
        $qualidades = QualidadeAgua::with(['barragem', 'estacao'])
            ->whereBetween('data_coleta', [$data_inicio, $data_fim])
            ->where('validado', true)
            ->orderBy('data_coleta', 'desc')
            ->get();

        foreach ($qualidades as $qualidade) {
            $local = $qualidade->barragem ? $qualidade->barragem->nome : ($qualidade->estacao ? $qualidade->estacao->nome : 'N/D');

            $dados['qualidade_agua'][] = [
                'local' => $local,
                'data_coleta' => $qualidade->data_coleta->format('d/m/Y'),
                'iqa' => $qualidade->iqa,
                'classificacao' => $qualidade->classificacao_iqa,
                'classificacao_descricao' => $qualidade->classificacao_descricao,
                'oxigenio_dissolvido' => $qualidade->oxigenio_dissolvido,
                'ph' => $qualidade->ph,
                'turbidez' => $qualidade->turbidez,
                'temperatura' => $qualidade->temperatura,
            ];
            $dados['tem_dados'] = true;
        }

        // Estatísticas gerais do IQA
        if ($qualidades->count() > 0) {
            $dados['estatisticas_iqa'] = [
                'total_amostras' => $qualidades->count(),
                'iqa_medio' => round($qualidades->avg('iqa'), 2),
                'iqa_minimo' => $qualidades->min('iqa'),
                'iqa_maximo' => $qualidades->max('iqa'),
                'excelente' => $qualidades->where('classificacao_iqa', 'excelente')->count(),
                'boa' => $qualidades->where('classificacao_iqa', 'boa')->count(),
                'media' => $qualidades->where('classificacao_iqa', 'media')->count(),
                'ruim' => $qualidades->where('classificacao_iqa', 'ruim')->count(),
                'pessima' => $qualidades->where('classificacao_iqa', 'pessima')->count(),
            ];
        }

        return $dados;
    }

    /**
     * Gerar seção de qualidade de água para o boletim
     */
    public function gerarSecaoQualidadeAgua($data_inicio, $data_fim)
    {
        $dados = $this->coletarDadosQualidadeAgua($data_inicio, $data_fim);

        if (!$dados['tem_dados']) {
            return "Não há dados de qualidade de água disponíveis para o período em análise.";
        }

        $secao = "**Qualidade de Água**\n\n";
        $secao .= "Durante o período em análise, foram realizadas {$dados['estatisticas_iqa']['total_amostras']} coletas de qualidade de água.\n\n";

        $secao .= "**Resumo das Análises:**\n";
        $secao .= "• IQA Médio: {$dados['estatisticas_iqa']['iqa_medio']}\n";
        $secao .= "• Classificação: ";

        if ($dados['estatisticas_iqa']['excelente'] > 0) {
            $secao .= "{$dados['estatisticas_iqa']['excelente']} excelente(s), ";
        }
        if ($dados['estatisticas_iqa']['boa'] > 0) {
            $secao .= "{$dados['estatisticas_iqa']['boa']} boa(s), ";
        }
        if ($dados['estatisticas_iqa']['media'] > 0) {
            $secao .= "{$dados['estatisticas_iqa']['media']} média(s), ";
        }
        if ($dados['estatisticas_iqa']['ruim'] > 0) {
            $secao .= "{$dados['estatisticas_iqa']['ruim']} ruim/ruins, ";
        }
        if ($dados['estatisticas_iqa']['pessima'] > 0) {
            $secao .= "{$dados['estatisticas_iqa']['pessima']} péssima(s)";
        }

        $secao = rtrim($secao, ', ') . "\n\n";

        $secao .= "**Detalhamento por Local:**\n";
        foreach ($dados['qualidade_agua'] as $qualidade) {
            $secao .= "• **{$qualidade['local']}** ({$qualidade['data_coleta']}): ";
            $secao .= "IQA = {$qualidade['iqa']} ({$qualidade['classificacao']})\n";
        }

        return $secao;
    }
}