<?php

namespace App\Services;

/**
 * Gerador de gráficos estilo Excel para barragens
 * Replica o formato exato dos gráficos do Excel da ARA Norte
 */
class ChartGenerator
{
    protected $width = 550;
    protected $height = 350;
    protected $padding = ['top' => 40, 'right' => 70, 'bottom' => 55, 'left' => 55];

    // Cores exatas do Excel
    protected $lineColors = [
        '2022/23' => [0, 176, 80],      // Verde
        '2023/24' => [0, 112, 192],     // Azul
        '2024/25' => [112, 48, 160],    // Roxo
        '2025/26' => [255, 0, 0],       // Vermelho
    ];

    /**
     * Gerar gráfico de barragem estilo Excel
     */
    public function generateLineChart(array $config): ?string
    {
        $labels = $config['labels'] ?? [];
        $datasets = $config['datasets'] ?? [];
        $title = $config['title'] ?? '';
        $yMin = $config['yMin'] ?? null;
        $yMax = $config['yMax'] ?? null;
        $precipitacao = $config['precipitacao'] ?? [];

        if (empty($labels) || empty($datasets)) {
            return null;
        }

        // Criar imagem
        $image = imagecreatetruecolor($this->width, $this->height);

        // Cores básicas
        $white = imagecolorallocate($image, 255, 255, 255);
        $black = imagecolorallocate($image, 0, 0, 0);
        $gray = imagecolorallocate($image, 128, 128, 128);
        $lightGray = imagecolorallocate($image, 192, 192, 192);
        $gridColor = imagecolorallocate($image, 192, 192, 192);

        // Cores específicas
        $greenColor = imagecolorallocate($image, 0, 176, 80);
        $blueColor = imagecolorallocate($image, 0, 112, 192);
        $purpleColor = imagecolorallocate($image, 112, 48, 160);
        $redColor = imagecolorallocate($image, 255, 0, 0);
        $darkBlueColor = imagecolorallocate($image, 0, 32, 96);
        $precipColor = imagecolorallocate($image, 128, 0, 128); // Roxo para precipitação

        // Fundo branco
        imagefill($image, 0, 0, $white);

        // Área do gráfico
        $chartLeft = $this->padding['left'];
        $chartRight = $this->width - $this->padding['right'];
        $chartTop = $this->padding['top'];
        $chartBottom = $this->height - $this->padding['bottom'];
        $chartWidth = $chartRight - $chartLeft;
        $chartHeight = $chartBottom - $chartTop;

        // Calcular range dos dados de cota
        $allValues = [];
        foreach ($datasets as $dataset) {
            if (!isset($dataset['borderDash'])) { // Ignorar linhas de referência no cálculo
                foreach ($dataset['data'] as $value) {
                    if ($value !== null && is_numeric($value)) {
                        $allValues[] = (float) $value;
                    }
                }
            }
        }

        // Incluir NPA e NME no range
        foreach ($datasets as $dataset) {
            if (isset($dataset['borderDash']) && !empty($dataset['data'])) {
                $val = $dataset['data'][0];
                if ($val !== null && is_numeric($val)) {
                    $allValues[] = (float) $val;
                }
            }
        }

        if (empty($allValues)) {
            imagedestroy($image);
            return null;
        }

        // Calcular min/max com margem
        $dataMin = $yMin ?? floor(min($allValues) - 2);
        $dataMax = $yMax ?? ceil(max($allValues) + 2);

        // Arredondar para valores bonitos
        $range = $dataMax - $dataMin;
        $step = $this->calculateNiceStep($range, 5);
        $dataMin = floor($dataMin / $step) * $step;
        $dataMax = ceil($dataMax / $step) * $step;
        $dataRange = $dataMax - $dataMin;

        if ($dataRange == 0) {
            $dataRange = 10;
            $dataMax = $dataMin + 10;
            $step = 2;
        }

        // Desenhar borda do gráfico
        imagerectangle($image, $chartLeft, $chartTop, $chartRight, $chartBottom, $black);

        // Desenhar grid e labels do eixo Y esquerdo (Cota)
        $numGridLines = (int)($dataRange / $step);
        for ($i = 0; $i <= $numGridLines; $i++) {
            $value = $dataMin + ($i * $step);
            $y = $chartBottom - (($value - $dataMin) / $dataRange) * $chartHeight;

            // Linha do grid
            if ($y > $chartTop && $y < $chartBottom) {
                imagesetthickness($image, 1);
                imageline($image, $chartLeft, (int)$y, $chartRight, (int)$y, $gridColor);
            }

            // Label eixo Y esquerdo
            $label = number_format($value, 0);
            imagestring($image, 2, $chartLeft - 30, (int)$y - 6, $label, $black);
        }

        // Eixo Y direito (Precipitação) - 0 a 120mm invertido
        $precipMax = 120;
        $precipStep = 20;
        for ($i = 0; $i <= 6; $i++) {
            $value = $i * $precipStep;
            // Invertido: 0 no topo, 120 embaixo
            $y = $chartTop + ($value / $precipMax) * $chartHeight;

            // Label eixo Y direito
            $label = number_format($value, 1);
            imagestring($image, 2, $chartRight + 8, (int)$y - 6, $label, $black);
        }

        // Labels dos eixos
        imagestringup($image, 2, 5, $chartBottom - 60, 'Cota (m)', $black);
        imagestringup($image, 2, $this->width - 12, $chartBottom - 30, 'Precipitacao (mm)', $black);

        // Calcular posições X
        $numPoints = count($labels);
        $pointSpacing = $chartWidth / $numPoints;
        $xPositions = [];
        for ($i = 0; $i < $numPoints; $i++) {
            $xPositions[$i] = $chartLeft + ($i * $pointSpacing) + ($pointSpacing / 2);
        }

        // Labels do eixo X
        for ($i = 0; $i < $numPoints; $i++) {
            $label = $labels[$i];
            $x = $xPositions[$i] - (strlen($label) * 3);
            imagestring($image, 2, (int)$x, $chartBottom + 5, $label, $black);
        }

        // Desenhar barras de precipitação (invertidas, do topo)
        if (!empty($precipitacao)) {
            $barWidth = $pointSpacing * 0.6;
            foreach ($precipitacao as $i => $precip) {
                if ($precip > 0 && isset($xPositions[$i])) {
                    $barHeight = ($precip / $precipMax) * $chartHeight;
                    $x1 = $xPositions[$i] - $barWidth / 2;
                    $x2 = $xPositions[$i] + $barWidth / 2;
                    $y1 = $chartTop;
                    $y2 = $chartTop + $barHeight;

                    imagefilledrectangle($image, (int)$x1, (int)$y1, (int)$x2, (int)$y2, $precipColor);
                }
            }
        }

        // Desenhar datasets (linhas)
        $legendItems = [];
        $colorMap = [
            '#00B050' => $greenColor,
            '#0070C0' => $blueColor,
            '#7030A0' => $purpleColor,
            '#FF0000' => $redColor,
            '#002060' => $darkBlueColor,
            '#4472C4' => $blueColor,
            '#2196F3' => $blueColor,
            '#4CAF50' => $greenColor,
            '#9C27B0' => $purpleColor,
            '#800000' => imagecolorallocate($image, 128, 0, 0),
        ];

        foreach ($datasets as $dataset) {
            $data = $dataset['data'];
            $label = $dataset['label'] ?? '';
            $isDashed = isset($dataset['borderDash']) && !empty($dataset['borderDash']);

            // Determinar cor
            $hexColor = $dataset['borderColor'] ?? '#000000';
            if (isset($colorMap[$hexColor])) {
                $lineColor = $colorMap[$hexColor];
            } else {
                $rgb = $this->hexToRgb($hexColor);
                $lineColor = imagecolorallocate($image, $rgb[0], $rgb[1], $rgb[2]);
            }

            // Coletar pontos válidos
            $points = [];
            for ($i = 0; $i < count($data); $i++) {
                if (isset($xPositions[$i]) && $data[$i] !== null && is_numeric($data[$i])) {
                    $x = $xPositions[$i];
                    $y = $chartBottom - (($data[$i] - $dataMin) / $dataRange) * $chartHeight;
                    $points[] = ['x' => $x, 'y' => $y];
                }
            }

            // Desenhar linha
            imagesetthickness($image, $isDashed ? 1 : 2);

            for ($i = 1; $i < count($points); $i++) {
                $x1 = (int)$points[$i - 1]['x'];
                $y1 = (int)$points[$i - 1]['y'];
                $x2 = (int)$points[$i]['x'];
                $y2 = (int)$points[$i]['y'];

                if ($isDashed) {
                    $this->drawDashedLine($image, $x1, $y1, $x2, $y2, $lineColor, 8, 4);
                } else {
                    imageline($image, $x1, $y1, $x2, $y2, $lineColor);
                    // Segunda linha para mais espessura
                    imageline($image, $x1, $y1 + 1, $x2, $y2 + 1, $lineColor);
                }
            }

            // Desenhar marcadores (quadrados para linhas normais)
            if (!$isDashed) {
                foreach ($points as $point) {
                    $size = 4;
                    imagefilledrectangle($image,
                        (int)$point['x'] - $size, (int)$point['y'] - $size,
                        (int)$point['x'] + $size, (int)$point['y'] + $size,
                        $lineColor);
                }
            }

            $legendItems[] = ['label' => $label, 'color' => $lineColor, 'isDashed' => $isDashed];
        }

        // Título
        if (!empty($title)) {
            $titleX = ($this->width - strlen($title) * 6) / 2;
            imagestring($image, 3, (int)$titleX, 8, $title, $black);
        }

        // Legenda na parte inferior
        $legendY = $this->height - 18;
        $legendX = $chartLeft;

        foreach ($legendItems as $item) {
            // Linha/símbolo
            if ($item['isDashed']) {
                $this->drawDashedLine($image, (int)$legendX, $legendY, (int)$legendX + 20, $legendY, $item['color'], 4, 3);
            } else {
                imageline($image, (int)$legendX, $legendY, (int)$legendX + 20, $legendY, $item['color']);
                imagefilledrectangle($image, (int)$legendX + 8, $legendY - 3, (int)$legendX + 14, $legendY + 3, $item['color']);
            }

            // Texto
            imagestring($image, 1, (int)$legendX + 24, $legendY - 4, $item['label'], $black);
            $legendX += strlen($item['label']) * 5 + 40;
        }

        // Converter para base64
        ob_start();
        imagepng($image, null, 9);
        $imageData = ob_get_clean();
        imagedestroy($image);

        return 'data:image/png;base64,' . base64_encode($imageData);
    }

    protected function calculateNiceStep(float $range, int $targetSteps): float
    {
        if ($range <= 0) return 1;

        $roughStep = $range / $targetSteps;
        $magnitude = pow(10, floor(log10($roughStep)));
        $residual = $roughStep / $magnitude;

        if ($residual > 5) return 10 * $magnitude;
        if ($residual > 2) return 5 * $magnitude;
        if ($residual > 1) return 2 * $magnitude;
        return $magnitude;
    }

    protected function drawDashedLine($image, int $x1, int $y1, int $x2, int $y2, $color, int $dash = 5, int $gap = 3): void
    {
        $dx = $x2 - $x1;
        $dy = $y2 - $y1;
        $length = sqrt($dx * $dx + $dy * $dy);
        if ($length == 0) return;

        $dx /= $length;
        $dy /= $length;

        $x = (float)$x1;
        $y = (float)$y1;
        $traveled = 0;
        $draw = true;

        while ($traveled < $length) {
            $segLen = $draw ? $dash : $gap;
            $segLen = min($segLen, $length - $traveled);

            $nx = $x + $dx * $segLen;
            $ny = $y + $dy * $segLen;

            if ($draw) {
                imageline($image, (int)$x, (int)$y, (int)$nx, (int)$ny, $color);
            }

            $x = $nx;
            $y = $ny;
            $traveled += $segLen;
            $draw = !$draw;
        }
    }

    protected function hexToRgb(string $hex): array
    {
        $hex = ltrim($hex, '#');
        $named = ['red'=>'FF0000','green'=>'00FF00','blue'=>'0000FF','maroon'=>'800000','black'=>'000000','white'=>'FFFFFF'];
        if (isset($named[strtolower($hex)])) $hex = $named[strtolower($hex)];
        if (strlen($hex) === 3) $hex = $hex[0].$hex[0].$hex[1].$hex[1].$hex[2].$hex[2];
        if (strlen($hex) !== 6) return [0, 0, 0];
        return [hexdec(substr($hex,0,2)), hexdec(substr($hex,2,2)), hexdec(substr($hex,4,2))];
    }

    public function setDimensions(int $width, int $height): self
    {
        $this->width = $width;
        $this->height = $height;
        return $this;
    }
}
