<?php

namespace App\Services;

use App\Models\Barragem;
use App\Models\Estacao;
use App\Models\LeituraBarragem;
use App\Models\LeituraEstacao;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class EstatisticasService
{
    /**
     * Calcular médias aritméticas de precipitação para estações
     * Fase I: Médias diárias, semanais e mensais
     */
    public function calcularMediasPrecipitacao($estacao_id = null, $periodo = '30d', $data_inicio = null, $data_fim = null)
    {
        $query = LeituraEstacao::query();

        if ($estacao_id) {
            $query->where('estacao_id', $estacao_id);
        }

        // Definir intervalo de datas
        if ($data_inicio && $data_fim) {
            $query->whereBetween('data_leitura', [$data_inicio, $data_fim]);
        } else {
            $startDate = $this->getStartDate($periodo);
            $query->where('data_leitura', '>=', $startDate);
        }

        $leituras = $query->orderBy('data_leitura')->get();

        if ($leituras->isEmpty()) {
            return [
                'media_total' => 0,
                'media_diaria' => [],
                'media_semanal' => [],
                'media_mensal' => [],
                'total_precipitacao' => 0,
                'dias_com_chuva' => 0,
                'dias_sem_chuva' => 0
            ];
        }

        // Calcular precipitação total e média geral
        $precipitacaoTotal = $leituras->sum(function ($leitura) {
            return $leitura->precipitacao_mm ?? $leitura->precipitacao ?? 0;
        });

        $diasComChuva = $leituras->filter(function ($leitura) {
            return ($leitura->precipitacao_mm ?? $leitura->precipitacao ?? 0) > 0;
        })->count();

        $diasSemChuva = $leituras->count() - $diasComChuva;

        // Médias diárias (agrupar por dia)
        $mediasDiarias = $leituras->groupBy(function ($leitura) {
            return $leitura->data_leitura->format('Y-m-d');
        })->map(function ($grupo, $data) {
            $media = $grupo->avg(function ($leitura) {
                return $leitura->precipitacao_mm ?? $leitura->precipitacao ?? 0;
            });

            return [
                'data' => Carbon::parse($data)->format('d/m/Y'),
                'media' => round($media, 2),
                'total' => round($grupo->sum(function ($leitura) {
                    return $leitura->precipitacao_mm ?? $leitura->precipitacao ?? 0;
                }), 2),
                'leituras' => $grupo->count()
            ];
        })->values();

        // Médias semanais (agrupar por semana)
        $mediasSemanais = $leituras->groupBy(function ($leitura) {
            return $leitura->data_leitura->format('Y-W');
        })->map(function ($grupo, $semana) {
            $media = $grupo->avg(function ($leitura) {
                return $leitura->precipitacao_mm ?? $leitura->precipitacao ?? 0;
            });

            $primeiroDia = $grupo->first()->data_leitura;
            $ultimoDia = $grupo->last()->data_leitura;

            return [
                'semana' => "Semana " . $primeiroDia->weekOfYear . "/" . $primeiroDia->year,
                'periodo' => $primeiroDia->format('d/m') . ' - ' . $ultimoDia->format('d/m/Y'),
                'media' => round($media, 2),
                'total' => round($grupo->sum(function ($leitura) {
                    return $leitura->precipitacao_mm ?? $leitura->precipitacao ?? 0;
                }), 2),
                'leituras' => $grupo->count()
            ];
        })->values();

        // Médias mensais (agrupar por mês)
        $mediasMensais = $leituras->groupBy(function ($leitura) {
            return $leitura->data_leitura->format('Y-m');
        })->map(function ($grupo, $mes) {
            $media = $grupo->avg(function ($leitura) {
                return $leitura->precipitacao_mm ?? $leitura->precipitacao ?? 0;
            });

            return [
                'mes' => Carbon::parse($mes . '-01')->locale('pt')->isoFormat('MMMM/YYYY'),
                'media' => round($media, 2),
                'total' => round($grupo->sum(function ($leitura) {
                    return $leitura->precipitacao_mm ?? $leitura->precipitacao ?? 0;
                }), 2),
                'leituras' => $grupo->count()
            ];
        })->values();

        return [
            'media_total' => round($precipitacaoTotal / $leituras->count(), 2),
            'media_diaria' => $mediasDiarias,
            'media_semanal' => $mediasSemanais,
            'media_mensal' => $mediasMensais,
            'total_precipitacao' => round($precipitacaoTotal, 2),
            'dias_com_chuva' => $diasComChuva,
            'dias_sem_chuva' => $diasSemChuva,
            'total_leituras' => $leituras->count()
        ];
    }

    /**
     * Calcular médias aritméticas de cotas nas barragens
     * Fase I: Médias diárias, semanais e mensais
     */
    public function calcularMediasCotasBarragens($barragem_id = null, $periodo = '30d', $data_inicio = null, $data_fim = null)
    {
        $query = LeituraBarragem::query();

        if ($barragem_id) {
            $query->where('barragem_id', $barragem_id);
        }

        // Definir intervalo de datas
        if ($data_inicio && $data_fim) {
            $query->whereBetween('data_leitura', [$data_inicio, $data_fim]);
        } else {
            $startDate = $this->getStartDate($periodo);
            $query->where('data_leitura', '>=', $startDate);
        }

        $leituras = $query->orderBy('data_leitura')->get();

        if ($leituras->isEmpty()) {
            return [
                'media_total' => 0,
                'media_diaria' => [],
                'media_semanal' => [],
                'media_mensal' => [],
                'cota_maxima' => 0,
                'cota_minima' => 0,
                'variacao' => 0
            ];
        }

        // Calcular estatísticas gerais
        $cotaMaxima = $leituras->max('cota_actual');
        $cotaMinima = $leituras->min('cota_actual');
        $cotaMedia = $leituras->avg('cota_actual');

        // Médias diárias (agrupar por dia)
        $mediasDiarias = $leituras->groupBy(function ($leitura) {
            return $leitura->data_leitura->format('Y-m-d');
        })->map(function ($grupo, $data) {
            $media = $grupo->avg('cota_actual');
            $maxima = $grupo->max('cota_actual');
            $minima = $grupo->min('cota_actual');

            return [
                'data' => Carbon::parse($data)->format('d/m/Y'),
                'media' => round($media, 2),
                'maxima' => round($maxima, 2),
                'minima' => round($minima, 2),
                'variacao' => round($maxima - $minima, 2),
                'leituras' => $grupo->count()
            ];
        })->values();

        // Médias semanais (agrupar por semana)
        $mediasSemanais = $leituras->groupBy(function ($leitura) {
            return $leitura->data_leitura->format('Y-W');
        })->map(function ($grupo, $semana) {
            $media = $grupo->avg('cota_actual');
            $maxima = $grupo->max('cota_actual');
            $minima = $grupo->min('cota_actual');

            $primeiroDia = $grupo->first()->data_leitura;
            $ultimoDia = $grupo->last()->data_leitura;

            return [
                'semana' => "Semana " . $primeiroDia->weekOfYear . "/" . $primeiroDia->year,
                'periodo' => $primeiroDia->format('d/m') . ' - ' . $ultimoDia->format('d/m/Y'),
                'media' => round($media, 2),
                'maxima' => round($maxima, 2),
                'minima' => round($minima, 2),
                'variacao' => round($maxima - $minima, 2),
                'leituras' => $grupo->count()
            ];
        })->values();

        // Médias mensais (agrupar por mês)
        $mediasMensais = $leituras->groupBy(function ($leitura) {
            return $leitura->data_leitura->format('Y-m');
        })->map(function ($grupo, $mes) {
            $media = $grupo->avg('cota_actual');
            $maxima = $grupo->max('cota_actual');
            $minima = $grupo->min('cota_actual');

            return [
                'mes' => Carbon::parse($mes . '-01')->locale('pt')->isoFormat('MMMM/YYYY'),
                'media' => round($media, 2),
                'maxima' => round($maxima, 2),
                'minima' => round($minima, 2),
                'variacao' => round($maxima - $minima, 2),
                'leituras' => $grupo->count()
            ];
        })->values();

        return [
            'media_total' => round($cotaMedia, 2),
            'media_diaria' => $mediasDiarias,
            'media_semanal' => $mediasSemanais,
            'media_mensal' => $mediasMensais,
            'cota_maxima' => round($cotaMaxima, 2),
            'cota_minima' => round($cotaMinima, 2),
            'variacao' => round($cotaMaxima - $cotaMinima, 2),
            'total_leituras' => $leituras->count()
        ];
    }

    /**
     * Calcular médias aritméticas de níveis nas estações hidrométricas
     * Fase I: Médias diárias, semanais e mensais
     */
    public function calcularMediasNiveisEstacoes($estacao_id = null, $periodo = '30d', $data_inicio = null, $data_fim = null)
    {
        $query = LeituraEstacao::query();

        if ($estacao_id) {
            $query->where('estacao_id', $estacao_id);
        }

        // Definir intervalo de datas
        if ($data_inicio && $data_fim) {
            $query->whereBetween('data_leitura', [$data_inicio, $data_fim]);
        } else {
            $startDate = $this->getStartDate($periodo);
            $query->where('data_leitura', '>=', $startDate);
        }

        $leituras = $query->orderBy('data_leitura')->get();

        if ($leituras->isEmpty()) {
            return [
                'media_total' => 0,
                'media_diaria' => [],
                'media_semanal' => [],
                'media_mensal' => [],
                'nivel_maximo' => 0,
                'nivel_minimo' => 0,
                'variacao' => 0
            ];
        }

        // Calcular estatísticas gerais (usar nivel_hidrometrico ou média dos níveis horários)
        $niveisValidos = $leituras->filter(function ($leitura) {
            return $leitura->nivel_hidrometrico !== null ||
                   $leitura->nivel_6h !== null ||
                   $leitura->nivel_12h !== null ||
                   $leitura->nivel_18h !== null;
        });

        if ($niveisValidos->isEmpty()) {
            return [
                'media_total' => 0,
                'media_diaria' => [],
                'media_semanal' => [],
                'media_mensal' => [],
                'nivel_maximo' => 0,
                'nivel_minimo' => 0,
                'variacao' => 0,
                'message' => 'Não há leituras de nível no período selecionado'
            ];
        }

        $nivelMaximo = $niveisValidos->max('nivel_hidrometrico');
        $nivelMinimo = $niveisValidos->min('nivel_hidrometrico');
        $nivelMedio = $niveisValidos->avg('nivel_hidrometrico');

        // Médias diárias (agrupar por dia)
        $mediasDiarias = $leituras->groupBy(function ($leitura) {
            return $leitura->data_leitura->format('Y-m-d');
        })->map(function ($grupo, $data) {
            $niveisGrupo = $grupo->filter(function ($leitura) {
                return $leitura->nivel_hidrometrico !== null;
            });

            if ($niveisGrupo->isEmpty()) {
                return null;
            }

            $media = $niveisGrupo->avg('nivel_hidrometrico');
            $maxima = $niveisGrupo->max('nivel_hidrometrico');
            $minima = $niveisGrupo->min('nivel_hidrometrico');

            return [
                'data' => Carbon::parse($data)->format('d/m/Y'),
                'media' => round($media, 2),
                'maxima' => round($maxima, 2),
                'minima' => round($minima, 2),
                'variacao' => round($maxima - $minima, 2),
                'leituras' => $niveisGrupo->count()
            ];
        })->filter()->values();

        // Médias semanais (agrupar por semana)
        $mediasSemanais = $leituras->groupBy(function ($leitura) {
            return $leitura->data_leitura->format('Y-W');
        })->map(function ($grupo, $semana) {
            $niveisGrupo = $grupo->filter(function ($leitura) {
                return $leitura->nivel_hidrometrico !== null;
            });

            if ($niveisGrupo->isEmpty()) {
                return null;
            }

            $media = $niveisGrupo->avg('nivel_hidrometrico');
            $maxima = $niveisGrupo->max('nivel_hidrometrico');
            $minima = $niveisGrupo->min('nivel_hidrometrico');

            $primeiroDia = $grupo->first()->data_leitura;
            $ultimoDia = $grupo->last()->data_leitura;

            return [
                'semana' => "Semana " . $primeiroDia->weekOfYear . "/" . $primeiroDia->year,
                'periodo' => $primeiroDia->format('d/m') . ' - ' . $ultimoDia->format('d/m/Y'),
                'media' => round($media, 2),
                'maxima' => round($maxima, 2),
                'minima' => round($minima, 2),
                'variacao' => round($maxima - $minima, 2),
                'leituras' => $niveisGrupo->count()
            ];
        })->filter()->values();

        // Médias mensais (agrupar por mês)
        $mediasMensais = $leituras->groupBy(function ($leitura) {
            return $leitura->data_leitura->format('Y-m');
        })->map(function ($grupo, $mes) {
            $niveisGrupo = $grupo->filter(function ($leitura) {
                return $leitura->nivel_hidrometrico !== null;
            });

            if ($niveisGrupo->isEmpty()) {
                return null;
            }

            $media = $niveisGrupo->avg('nivel_hidrometrico');
            $maxima = $niveisGrupo->max('nivel_hidrometrico');
            $minima = $niveisGrupo->min('nivel_hidrometrico');

            return [
                'mes' => Carbon::parse($mes . '-01')->locale('pt')->isoFormat('MMMM/YYYY'),
                'media' => round($media, 2),
                'maxima' => round($maxima, 2),
                'minima' => round($minima, 2),
                'variacao' => round($maxima - $minima, 2),
                'leituras' => $niveisGrupo->count()
            ];
        })->filter()->values();

        return [
            'media_total' => round($nivelMedio, 2),
            'media_diaria' => $mediasDiarias,
            'media_semanal' => $mediasSemanais,
            'media_mensal' => $mediasMensais,
            'nivel_maximo' => round($nivelMaximo, 2),
            'nivel_minimo' => round($nivelMinimo, 2),
            'variacao' => round($nivelMaximo - $nivelMinimo, 2),
            'total_leituras' => $niveisValidos->count()
        ];
    }

    /**
     * Helper para determinar data de início baseado no período
     */
    private function getStartDate($periodo)
    {
        return match($periodo) {
            '7d' => Carbon::now()->subDays(7),
            '30d' => Carbon::now()->subDays(30),
            '90d' => Carbon::now()->subDays(90),
            '6m' => Carbon::now()->subMonths(6),
            '1y' => Carbon::now()->subYear(),
            default => Carbon::now()->subDays(30)
        };
    }
}
