<?php

namespace App\Services;

/**
 * Gerador de gráficos estilo Excel usando Chart.js + Node.js
 * Produz gráficos visuais idênticos aos gerados pelo Excel
 */
class ExcelChartGenerator
{
    protected $width = 700;
    protected $height = 400;
    protected $nodePath = 'node';
    protected $scriptPath;

    public function __construct()
    {
        // Usar Chart.js para gráficos estilo Excel
        $this->scriptPath = base_path('scripts/generate-chart-chartjs.cjs');
    }

    /**
     * Gerar gráfico de barragem estilo Excel
     */
    public function generateLineChart(array $config): ?string
    {
        $labels = $config['labels'] ?? [];
        $datasets = $config['datasets'] ?? [];

        if (empty($labels) || empty($datasets)) {
            return null;
        }

        // Preparar configuração para o script Node.js (Chart.js)
        $chartConfig = [
            'width' => $this->width,
            'height' => $this->height,
            'title' => $config['title'] ?? '',
            'labels' => $labels,
            'datasets' => $datasets,
            'yMin' => $config['yMin'] ?? null,
            'yMax' => $config['yMax'] ?? null,
            'precipitacao' => $config['precipitacao'] ?? [],
            'precipMax' => $config['precipMax'] ?? 100,
        ];

        // Criar arquivos temporários
        $configFile = sys_get_temp_dir() . '/chart_config_' . uniqid() . '.json';
        $outputFile = sys_get_temp_dir() . '/chart_output_' . uniqid() . '.png';

        // Salvar configuração
        file_put_contents($configFile, json_encode($chartConfig, JSON_PRETTY_PRINT));

        try {
            // Executar script Node.js
            $command = escapeshellcmd($this->nodePath) . ' ' .
                       escapeshellarg($this->scriptPath) . ' ' .
                       escapeshellarg($configFile) . ' ' .
                       escapeshellarg($outputFile) . ' 2>&1';

            $output = [];
            $returnCode = 0;
            exec($command, $output, $returnCode);

            if ($returnCode !== 0) {
                \Log::error('Erro ao gerar gráfico via Node.js', [
                    'command' => $command,
                    'output' => implode("\n", $output),
                    'returnCode' => $returnCode,
                ]);

                // Fallback para gerador PHP GD
                return $this->fallbackToGD($config);
            }

            // Ler imagem gerada
            if (!file_exists($outputFile)) {
                \Log::error('Arquivo de gráfico não foi criado: ' . $outputFile);
                return $this->fallbackToGD($config);
            }

            $imageData = file_get_contents($outputFile);
            $base64 = 'data:image/png;base64,' . base64_encode($imageData);

            return $base64;

        } finally {
            // Limpar arquivos temporários
            @unlink($configFile);
            @unlink($outputFile);
        }
    }

    /**
     * Fallback para o gerador GD caso Node.js falhe
     */
    protected function fallbackToGD(array $config): ?string
    {
        $gdGenerator = new ChartGenerator();
        $gdGenerator->setDimensions($this->width, $this->height);
        return $gdGenerator->generateLineChart($config);
    }

    /**
     * Definir dimensões do gráfico
     */
    public function setDimensions(int $width, int $height): self
    {
        $this->width = $width;
        $this->height = $height;
        return $this;
    }

    /**
     * Verificar se Node.js está disponível
     */
    public static function isNodeAvailable(): bool
    {
        $output = [];
        $returnCode = 0;
        exec('node --version 2>&1', $output, $returnCode);
        return $returnCode === 0;
    }
}
