<?php

namespace App\Services;

use App\Models\Barragem;
use App\Models\LeituraBarragem;
use Carbon\Carbon;

class GraficoBarragemService
{
    protected $chartGenerator;

    public function __construct()
    {
        // Usar ExcelChartGenerator se Node.js estiver disponível
        if (ExcelChartGenerator::isNodeAvailable()) {
            $this->chartGenerator = new ExcelChartGenerator();
        } else {
            $this->chartGenerator = new ChartGenerator();
        }
    }

    /**
     * Gerar gráfico de evolução de cotas para uma barragem
     * Usa dados diários para gráficos estilo Excel
     */
    public function gerarGraficoBarragem(Barragem $barragem, int $anoAtual = null): ?string
    {
        $anoAtual = $anoAtual ?? now()->year;

        // Coletar dados diários dos últimos 3 anos hidrológicos
        $dadosDiarios = $this->coletarDadosDiarios($barragem, $anoAtual, 3);

        // Se não há dados, tentar gráfico simples
        if (empty($dadosDiarios['datasets'])) {
            return $this->gerarGraficoSimples($barragem);
        }

        // Valores de referência
        $npaValue = (float) $barragem->cota_npa;
        $nmeValue = (float) $barragem->cota_nme;
        $numPontos = count($dadosDiarios['labels']);

        // Adicionar linhas de referência NPA e NME
        $dadosDiarios['datasets'][] = [
            'label' => 'NPA',
            'data' => array_fill(0, $numPontos, $npaValue),
            'borderColor' => '#FF0000',
            'borderDash' => [5, 5],
        ];

        $dadosDiarios['datasets'][] = [
            'label' => 'NME',
            'data' => array_fill(0, $numPontos, $nmeValue),
            'borderColor' => '#800000',
            'borderDash' => [5, 5],
        ];

        // Calcular min/max para os eixos
        $todasCotas = [];
        foreach ($dadosDiarios['datasets'] as $ds) {
            if (!isset($ds['borderDash'])) { // Apenas linhas de dados, não NPA/NME
                $todasCotas = array_merge($todasCotas, array_filter($ds['data'], fn($v) => $v !== null));
            }
        }
        $todasCotas[] = $npaValue;
        $todasCotas[] = $nmeValue;

        $minCota = !empty($todasCotas) ? floor(min($todasCotas) - 3) : 0;
        $maxCota = !empty($todasCotas) ? ceil(max($todasCotas) + 2) : 100;

        // Configuração do gráfico
        $chartConfig = [
            'labels' => $dadosDiarios['labels'],
            'datasets' => $dadosDiarios['datasets'],
            'title' => 'BARRAGEM DE ' . strtoupper($barragem->nome),
            'yMin' => $minCota,
            'yMax' => $maxCota,
            'precipitacao' => $dadosDiarios['precipitacao'],
            'precipMax' => 100,
        ];

        return $this->chartGenerator->generateLineChart($chartConfig);
    }

    /**
     * Coletar dados diários de uma barragem (estilo Excel)
     */
    protected function coletarDadosDiarios(Barragem $barragem, int $anoAtual, int $numAnos = 3): array
    {
        $datasets = [];
        $labels = [];
        $precipitacao = [];

        // Cores estilo Excel
        $cores = ['#00B0F0', '#00B050', '#002060'];

        for ($i = 0; $i < $numAnos; $i++) {
            $anoFim = $anoAtual - $i;
            $anoInicio = $anoFim - 1;
            $anoHidrologico = "{$anoInicio}/" . substr($anoFim, 2);

            $dataInicio = Carbon::create($anoInicio, 10, 1);
            $dataFim = Carbon::create($anoFim, 9, 30);

            $leituras = LeituraBarragem::where('barragem_id', $barragem->id)
                ->whereBetween('data_leitura', [$dataInicio, $dataFim])
                ->orderBy('data_leitura')
                ->get();

            if ($leituras->isEmpty()) {
                continue;
            }

            // Extrair cotas
            $cotas = $leituras->pluck('cota_actual')->map(fn($v) => $v ? (float) $v : null)->toArray();

            $datasets[] = [
                'label' => $anoHidrologico,
                'data' => $cotas,
                'borderColor' => $cores[$i] ?? '#666666',
            ];

            // Labels e precipitação do ano mais recente
            if (empty($labels)) {
                $labels = array_fill(0, count($cotas), '');
                $precipitacao = $leituras->pluck('precipitacao')
                    ->map(fn($v) => min((float) ($v ?? 0), 100))
                    ->toArray();
            }
        }

        // Normalizar tamanhos (usar o menor)
        if (!empty($datasets)) {
            $minLen = min(array_map(fn($ds) => count($ds['data']), $datasets));
            foreach ($datasets as &$ds) {
                $ds['data'] = array_slice($ds['data'], 0, $minLen);
            }
            $labels = array_slice($labels, 0, $minLen);
            $precipitacao = array_slice($precipitacao, 0, $minLen);
        }

        return [
            'datasets' => $datasets,
            'labels' => $labels,
            'precipitacao' => $precipitacao,
        ];
    }

    /**
     * Coletar dados históricos de uma barragem por ano hidrológico
     */
    protected function coletarDadosHistoricos(Barragem $barragem, int $anoAtual, int $numAnos = 3): array
    {
        $dados = [];

        for ($i = 0; $i < $numAnos; $i++) {
            $anoFim = $anoAtual - $i;
            $anoInicio = $anoFim - 1;
            $anoHidrologico = "{$anoInicio}/" . substr($anoFim, 2);

            // Período do ano hidrológico: Outubro do ano anterior a Setembro do ano atual
            $dataInicio = Carbon::create($anoInicio, 10, 1);
            $dataFim = Carbon::create($anoFim, 9, 30);

            // Buscar leituras do período
            $leituras = LeituraBarragem::where('barragem_id', $barragem->id)
                ->whereBetween('data_leitura', [$dataInicio, $dataFim])
                ->orderBy('data_leitura')
                ->get();

            if ($leituras->isEmpty()) {
                continue;
            }

            // Agrupar por mês e calcular média
            $cotasPorMes = [];
            $precipitacaoPorMes = [];

            // Inicializar meses (Outubro=10 a Setembro=9)
            $mesesOrdem = [10, 11, 12, 1, 2, 3, 4, 5, 6, 7, 8, 9];
            foreach ($mesesOrdem as $mes) {
                $cotasPorMes[$mes] = [];
                $precipitacaoPorMes[$mes] = [];
            }

            foreach ($leituras as $leitura) {
                $mes = $leitura->data_leitura->month;
                if (isset($cotasPorMes[$mes])) {
                    if ($leitura->cota_actual) {
                        $cotasPorMes[$mes][] = (float) $leitura->cota_actual;
                    }
                    if ($leitura->precipitacao) {
                        $precipitacaoPorMes[$mes][] = (float) $leitura->precipitacao;
                    }
                }
            }

            // Calcular médias mensais
            $cotasMedia = [];
            $precipitacaoMedia = [];
            foreach ($mesesOrdem as $mes) {
                $cotasMedia[] = !empty($cotasPorMes[$mes]) ? round(array_sum($cotasPorMes[$mes]) / count($cotasPorMes[$mes]), 2) : null;
                $precipitacaoMedia[] = !empty($precipitacaoPorMes[$mes]) ? round(array_sum($precipitacaoPorMes[$mes]), 1) : 0;
            }

            $dados[$anoHidrologico] = [
                'cotas' => $cotasMedia,
                'precipitacao' => $precipitacaoMedia,
            ];
        }

        return $dados;
    }

    /**
     * Gerar gráficos para todas as barragens de um boletim
     */
    public function gerarGraficosParaBoletim(array $barragens): array
    {
        $graficos = [];

        foreach ($barragens as $barragemData) {
            $barragemId = is_array($barragemData) ? ($barragemData['id'] ?? null) : ($barragemData->id ?? null);
            $barragemNome = is_array($barragemData) ? ($barragemData['nome'] ?? 'N/A') : ($barragemData->nome ?? 'N/A');

            try {
                if (!$barragemId) {
                    // Tentar buscar pelo nome
                    $barragem = Barragem::where('nome', $barragemNome)->first();
                } else {
                    $barragem = Barragem::find($barragemId);
                }

                if ($barragem) {
                    $grafico = $this->gerarGraficoBarragem($barragem);
                    if ($grafico) {
                        $graficos[$barragem->nome] = $grafico;
                    } else {
                        \Log::info("Sem dados históricos para gráfico: {$barragem->nome}");
                    }
                }
            } catch (\Exception $e) {
                \Log::error("Erro ao gerar gráfico para {$barragemNome}: " . $e->getMessage());
                continue;
            }
        }

        return $graficos;
    }

    /**
     * Gerar gráfico simples com dados recentes disponíveis
     * Usado quando não há dados históricos suficientes
     */
    protected function gerarGraficoSimples(Barragem $barragem): ?string
    {
        // Buscar últimas 30 leituras
        $leituras = LeituraBarragem::where('barragem_id', $barragem->id)
            ->orderBy('data_leitura', 'desc')
            ->limit(30)
            ->get()
            ->reverse();

        if ($leituras->isEmpty()) {
            return null;
        }

        // Preparar dados
        $labels = [];
        $cotas = [];

        foreach ($leituras as $leitura) {
            $labels[] = $leitura->data_leitura->format('d/m');
            $cotas[] = (float) $leitura->cota_actual;
        }

        // Calcular min/max
        $cotasValidas = array_filter($cotas);
        $npaValue = (float) $barragem->cota_npa;
        $nmeValue = (float) $barragem->cota_nme;

        $allValues = array_merge($cotasValidas, [$npaValue, $nmeValue]);
        $minCota = !empty($allValues) ? floor(min($allValues) - 2) : 0;
        $maxCota = !empty($allValues) ? ceil(max($allValues) + 2) : 100;

        // Datasets
        $datasets = [
            [
                'label' => 'Cota Actual',
                'data' => $cotas,
                'borderColor' => '#2196F3',
            ],
            [
                'label' => 'NPA',
                'data' => array_fill(0, count($labels), $npaValue),
                'borderColor' => '#FF0000',
                'borderDash' => [5, 5],
            ],
            [
                'label' => 'NME',
                'data' => array_fill(0, count($labels), $nmeValue),
                'borderColor' => '#800000',
                'borderDash' => [5, 5],
            ],
        ];

        $chartConfig = [
            'labels' => $labels,
            'datasets' => $datasets,
            'title' => 'BARRAGEM DE ' . strtoupper($barragem->nome),
            'yMin' => $minCota,
            'yMax' => $maxCota,
        ];

        return $this->chartGenerator->generateLineChart($chartConfig);
    }
}
