<?php

namespace App\Services;

use App\Models\Estacao;
use App\Models\LeituraEstacao;
use Carbon\Carbon;

class GraficoEstacaoService
{
    protected $chartGenerator;

    public function __construct()
    {
        if (ExcelChartGenerator::isNodeAvailable()) {
            $this->chartGenerator = new ExcelChartGenerator();
        } else {
            $this->chartGenerator = new ChartGenerator();
        }
    }

    /**
     * Gerar gráfico de evolução de níveis para uma estação hidrométrica
     * Mostra comparação de anos hidrológicos com N. Alerta e precipitação
     */
    public function gerarGraficoEstacao(Estacao $estacao, int $anoAtual = null): ?string
    {
        $anoAtual = $anoAtual ?? now()->year;

        // Coletar dados diários dos últimos 3 anos hidrológicos
        $dadosDiarios = $this->coletarDadosDiarios($estacao, $anoAtual, 3);

        if (empty($dadosDiarios['datasets'])) {
            return $this->gerarGraficoSimples($estacao);
        }

        // Linha de N. Alerta
        $nivelAlerta = (float) $estacao->nivel_alerta;
        $numPontos = count($dadosDiarios['labels']);

        if ($nivelAlerta > 0) {
            $dadosDiarios['datasets'][] = [
                'label' => 'N .Alerta',
                'data' => array_fill(0, $numPontos, $nivelAlerta),
                'borderColor' => '#FF0000',
                'borderDash' => [5, 5],
            ];
        }

        // Calcular min/max para os eixos
        $todosNiveis = [];
        foreach ($dadosDiarios['datasets'] as $ds) {
            if (!isset($ds['borderDash'])) {
                $todosNiveis = array_merge($todosNiveis, array_filter($ds['data'], fn($v) => $v !== null));
            }
        }
        if ($nivelAlerta > 0) {
            $todosNiveis[] = $nivelAlerta;
        }

        $minNivel = !empty($todosNiveis) ? floor(min($todosNiveis) * 10) / 10 : 0;
        $maxNivel = !empty($todosNiveis) ? ceil(max($todosNiveis) + 1) : 10;
        $minNivel = max(0, $minNivel - 0.5);

        // Buscar bacia para o título
        $bacia = $estacao->bacia_hidrografica;
        $nomeBacia = $bacia ? $bacia->nome : '';
        $titulo = $estacao->codigo . ', em ' . $estacao->nome . ' / B. ' . $nomeBacia;

        // Buscar código da pluviométrica associada
        $pluvStation = null;
        if ($bacia) {
            $pluvStation = Estacao::where('bacia_hidrografica_id', $bacia->id)
                ->where('tipo', 'pluviometrica')
                ->where('estado', 'activa')
                ->first();
        }

        // Configuração do gráfico
        $chartConfig = [
            'labels' => $dadosDiarios['labels'],
            'datasets' => $dadosDiarios['datasets'],
            'title' => $titulo,
            'yMin' => $minNivel,
            'yMax' => $maxNivel,
            'precipitacao' => $dadosDiarios['precipitacao'],
            'precipMax' => 100,
            'yLeftLabel' => 'Niveis Hidrometricas ( m )',
            'yRightLabel' => 'Precipitacao ( mm )',
            'precipLabel' => $pluvStation ? $pluvStation->codigo : null,
        ];

        return $this->chartGenerator->generateLineChart($chartConfig);
    }

    /**
     * Coletar dados diários de uma estação (nível médio de 6h/12h/18h)
     */
    protected function coletarDadosDiarios(Estacao $estacao, int $anoAtual, int $numAnos = 3): array
    {
        $datasets = [];
        $labels = [];
        $precipitacao = [];

        // Cores para os diferentes anos
        $cores = ['#00B0F0', '#7030A0', '#00B050'];

        for ($i = 0; $i < $numAnos; $i++) {
            $anoFim = $anoAtual - $i;
            $anoInicio = $anoFim - 1;
            $anoHidrologico = "{$anoInicio}/" . substr($anoFim, 2);

            $dataInicio = Carbon::create($anoInicio, 10, 1);
            $dataFim = Carbon::create($anoFim, 9, 30);

            $leituras = LeituraEstacao::where('estacao_id', $estacao->id)
                ->whereBetween('data_leitura', [$dataInicio, $dataFim])
                ->orderBy('data_leitura')
                ->get();

            if ($leituras->isEmpty()) {
                continue;
            }

            // Calcular nível médio diário (média de 6h, 12h, 18h)
            $niveis = $leituras->map(function ($l) {
                $valores = array_filter([
                    $l->nivel_6h ? (float)$l->nivel_6h : null,
                    $l->nivel_12h ? (float)$l->nivel_12h : null,
                    $l->nivel_18h ? (float)$l->nivel_18h : null,
                ], fn($v) => $v !== null);

                return !empty($valores) ? round(array_sum($valores) / count($valores), 2) : null;
            })->toArray();

            $datasets[] = [
                'label' => $anoHidrologico,
                'data' => $niveis,
                'borderColor' => $cores[$i] ?? '#666666',
            ];

            // Labels (meses) e precipitação do ano mais recente
            if (empty($labels)) {
                $labels = $leituras->map(function ($l) {
                    return $l->data_leitura->format('d/m');
                })->toArray();

                $precipitacao = $leituras->map(function ($l) {
                    return min((float)($l->precipitacao_mm ?? 0), 100);
                })->toArray();
            }
        }

        // Normalizar tamanhos
        if (!empty($datasets)) {
            $minLen = min(array_map(fn($ds) => count($ds['data']), $datasets));
            foreach ($datasets as &$ds) {
                $ds['data'] = array_slice($ds['data'], 0, $minLen);
            }
            $labels = array_slice($labels, 0, $minLen);
            $precipitacao = array_slice($precipitacao, 0, $minLen);
        }

        return [
            'datasets' => $datasets,
            'labels' => $labels,
            'precipitacao' => $precipitacao,
        ];
    }

    /**
     * Gerar gráficos para todas as estações de um boletim
     */
    public function gerarGraficosParaBoletim(array $dados_bacias): array
    {
        $graficos = [];

        foreach ($dados_bacias as $nomeBacia => $bacia) {
            foreach ($bacia['estacoes'] ?? [] as $estacaoData) {
                $codigo = $estacaoData['codigo'] ?? null;
                $nome = $estacaoData['nome'] ?? 'N/A';

                try {
                    $estacao = Estacao::where('codigo', $codigo)
                        ->where('tipo', 'hidrometrica')
                        ->first();

                    if (!$estacao) {
                        $estacao = Estacao::where('nome', $nome)
                            ->where('tipo', 'hidrometrica')
                            ->first();
                    }

                    if ($estacao) {
                        $grafico = $this->gerarGraficoEstacao($estacao);
                        if ($grafico) {
                            $chave = $estacao->codigo . ' ' . $estacao->nome;
                            $graficos[$chave] = $grafico;
                        }
                    }
                } catch (\Exception $e) {
                    \Log::error("Erro ao gerar gráfico para estação {$nome}: " . $e->getMessage());
                    continue;
                }
            }
        }

        return $graficos;
    }

    /**
     * Gerar gráfico simples com dados recentes (fallback)
     */
    protected function gerarGraficoSimples(Estacao $estacao): ?string
    {
        $leituras = LeituraEstacao::where('estacao_id', $estacao->id)
            ->orderBy('data_leitura', 'desc')
            ->limit(30)
            ->get()
            ->reverse();

        if ($leituras->isEmpty()) {
            return null;
        }

        $labels = [];
        $niveis = [];

        foreach ($leituras as $leitura) {
            $labels[] = $leitura->data_leitura->format('d/m');

            $valores = array_filter([
                $leitura->nivel_6h ? (float)$leitura->nivel_6h : null,
                $leitura->nivel_12h ? (float)$leitura->nivel_12h : null,
                $leitura->nivel_18h ? (float)$leitura->nivel_18h : null,
            ], fn($v) => $v !== null);

            $niveis[] = !empty($valores) ? round(array_sum($valores) / count($valores), 2) : null;
        }

        $niveisValidos = array_filter($niveis, fn($v) => $v !== null);
        $nivelAlerta = (float) $estacao->nivel_alerta;

        $allValues = array_merge($niveisValidos, $nivelAlerta > 0 ? [$nivelAlerta] : []);
        $minNivel = !empty($allValues) ? max(0, floor(min($allValues) - 1)) : 0;
        $maxNivel = !empty($allValues) ? ceil(max($allValues) + 1) : 10;

        $datasets = [
            [
                'label' => 'Nível Actual',
                'data' => $niveis,
                'borderColor' => '#00B0F0',
            ],
        ];

        if ($nivelAlerta > 0) {
            $datasets[] = [
                'label' => 'N .Alerta',
                'data' => array_fill(0, count($labels), $nivelAlerta),
                'borderColor' => '#FF0000',
                'borderDash' => [5, 5],
            ];
        }

        $bacia = $estacao->bacia_hidrografica;
        $nomeBacia = $bacia ? $bacia->nome : '';
        $titulo = $estacao->codigo . ', em ' . $estacao->nome . ' / B. ' . $nomeBacia;

        $chartConfig = [
            'labels' => $labels,
            'datasets' => $datasets,
            'title' => $titulo,
            'yMin' => $minNivel,
            'yMax' => $maxNivel,
        ];

        return $this->chartGenerator->generateLineChart($chartConfig);
    }
}
