<?php

namespace App\Services;

use App\Models\User;
use App\Models\DivisaoAdministrativa;
use App\Models\Barragem;
use App\Models\Estacao;
use App\Models\BaciaHidrografica;
use App\Models\LeituraBarragem;
use App\Models\LeituraEstacao;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Cache;

/**
 * Servico para controle de acesso regional
 * Filtra dados baseado nas divisoes atribuidas ao utilizador
 */
class RegionalAccessService
{
    protected ?User $user;
    protected array $accessibleDivisaoIds = [];

    public function __construct(?User $user = null)
    {
        $this->user = $user ?? auth()->user();
        $this->loadAccessibleDivisoes();
    }

    /**
     * Carregar e cachear IDs das divisoes acessiveis
     */
    protected function loadAccessibleDivisoes(): void
    {
        if (!$this->user) {
            return;
        }

        $cacheKey = "user_{$this->user->id}_divisoes_acessiveis";

        $this->accessibleDivisaoIds = Cache::remember($cacheKey, 300, function () {
            return $this->user->getDivisoesAcessiveisIds();
        });
    }

    /**
     * Limpar cache de acesso para um utilizador
     */
    public static function clearUserCache(int $userId): void
    {
        Cache::forget("user_{$userId}_divisoes_acessiveis");
    }

    /**
     * Verificar se tem acesso global
     */
    public function hasGlobalAccess(): bool
    {
        return $this->user && $this->user->temAcessoGlobal();
    }

    /**
     * Obter IDs das divisoes acessiveis
     */
    public function getAccessibleDivisaoIds(): array
    {
        return $this->accessibleDivisaoIds;
    }

    /**
     * Verificar se pode acessar uma divisao especifica
     */
    public function canAccessDivisao(int $divisaoId): bool
    {
        if ($this->hasGlobalAccess()) {
            return true;
        }

        return in_array($divisaoId, $this->accessibleDivisaoIds);
    }

    /**
     * Verificar permissao para uma accao numa divisao
     */
    public function canPerformAction(string $permission, ?int $divisaoId = null): bool
    {
        if (!$this->user) {
            return false;
        }

        if ($this->hasGlobalAccess()) {
            return true;
        }

        // Se nenhuma divisao especificada, verificar permissao base
        if (!$divisaoId) {
            return $this->user->can($permission);
        }

        return $this->user->temPermissaoNaDivisao($permission, $divisaoId);
    }

    /**
     * Aplicar filtro regional a query de Barragens
     */
    public function filterBarragens(Builder $query): Builder
    {
        if ($this->hasGlobalAccess()) {
            return $query;
        }

        return $query->where(function ($q) {
            // Filtrar por divisao_administrativa_id se definido
            $q->whereIn('divisao_administrativa_id', $this->accessibleDivisaoIds);

            // Tambem verificar campo provincia string para dados legados
            $provincias = $this->getProvinciaNames();
            if (!empty($provincias)) {
                $q->orWhere(function ($subQ) use ($provincias) {
                    foreach ($provincias as $provincia) {
                        $subQ->orWhere('provincia', 'like', "%{$provincia}%");
                    }
                });
            }
        });
    }

    /**
     * Aplicar filtro regional a query de Estacoes
     */
    public function filterEstacoes(Builder $query): Builder
    {
        if ($this->hasGlobalAccess()) {
            return $query;
        }

        return $query->where(function ($q) {
            $q->whereIn('divisao_administrativa_id', $this->accessibleDivisaoIds);

            $provincias = $this->getProvinciaNames();
            if (!empty($provincias)) {
                $q->orWhere(function ($subQ) use ($provincias) {
                    foreach ($provincias as $provincia) {
                        $subQ->orWhere('provincia', 'like', "%{$provincia}%");
                    }
                });
            }
        });
    }

    /**
     * Aplicar filtro regional a query de Bacias
     */
    public function filterBacias(Builder $query): Builder
    {
        if ($this->hasGlobalAccess()) {
            return $query;
        }

        return $query->whereIn('divisao_administrativa_id', $this->accessibleDivisaoIds);
    }

    /**
     * Aplicar filtro regional a query de Leituras de Barragens
     */
    public function filterLeiturasBarragens(Builder $query): Builder
    {
        if ($this->hasGlobalAccess()) {
            return $query;
        }

        return $query->whereHas('barragem', function ($q) {
            $this->filterBarragens($q);
        });
    }

    /**
     * Aplicar filtro regional a query de Leituras de Estacoes
     */
    public function filterLeiturasEstacoes(Builder $query): Builder
    {
        if ($this->hasGlobalAccess()) {
            return $query;
        }

        return $query->whereHas('estacao', function ($q) {
            $this->filterEstacoes($q);
        });
    }

    /**
     * Obter nomes das provincias acessiveis
     */
    public function getProvinciaNames(): array
    {
        return DivisaoAdministrativa::whereIn('id', $this->accessibleDivisaoIds)
            ->where('tipo', 'provincia')
            ->pluck('nome')
            ->toArray();
    }

    /**
     * Verificar se pode acessar uma Barragem especifica
     */
    public function canAccessBarragem(Barragem $barragem): bool
    {
        if ($this->hasGlobalAccess()) {
            return true;
        }

        // Verificar divisao_administrativa_id
        if ($barragem->divisao_administrativa_id) {
            return $this->canAccessDivisao($barragem->divisao_administrativa_id);
        }

        // Fallback para campo provincia string
        $provincias = $this->getProvinciaNames();
        foreach ($provincias as $provincia) {
            if (stripos($barragem->provincia ?? '', $provincia) !== false) {
                return true;
            }
        }

        return false;
    }

    /**
     * Verificar se pode acessar uma Estacao especifica
     */
    public function canAccessEstacao(Estacao $estacao): bool
    {
        if ($this->hasGlobalAccess()) {
            return true;
        }

        if ($estacao->divisao_administrativa_id) {
            return $this->canAccessDivisao($estacao->divisao_administrativa_id);
        }

        $provincias = $this->getProvinciaNames();
        foreach ($provincias as $provincia) {
            if (stripos($estacao->provincia ?? '', $provincia) !== false) {
                return true;
            }
        }

        return false;
    }

    /**
     * Obter estatisticas regionais para o utilizador
     */
    public function getRegionalStats(): array
    {
        $barragensQuery = Barragem::query();
        $estacoesQuery = Estacao::query();

        $this->filterBarragens($barragensQuery);
        $this->filterEstacoes($estacoesQuery);

        return [
            'total_barragens' => $barragensQuery->count(),
            'barragens_ativas' => (clone $barragensQuery)->where('estado', 'activa')->count(),
            'total_estacoes' => $estacoesQuery->count(),
            'estacoes_ativas' => (clone $estacoesQuery)->where('estado', 'activa')->count(),
            'divisoes_acessiveis' => count($this->accessibleDivisaoIds),
            'provincias' => $this->getProvinciaNames(),
        ];
    }

    /**
     * Obter divisoes acessiveis agrupadas por tipo
     */
    public function getDivisoesAcessiveisPorTipo(): array
    {
        if ($this->hasGlobalAccess()) {
            return DivisaoAdministrativa::ativo()
                ->orderBy('tipo')
                ->orderBy('nome')
                ->get()
                ->groupBy('tipo')
                ->toArray();
        }

        return DivisaoAdministrativa::whereIn('id', $this->accessibleDivisaoIds)
            ->ativo()
            ->orderBy('tipo')
            ->orderBy('nome')
            ->get()
            ->groupBy('tipo')
            ->toArray();
    }

    /**
     * Obter utilizador actual
     */
    public function getUser(): ?User
    {
        return $this->user;
    }

    /**
     * Definir utilizador (util para testes)
     */
    public function setUser(?User $user): self
    {
        $this->user = $user;
        $this->loadAccessibleDivisoes();
        return $this;
    }
}
