<?php

namespace App\Console\Commands;

use App\Models\Estacao;
use App\Services\SebaHydrometrieService;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class SebaImportarEstacao extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'seba:importar-estacao
                           {estacao_id : ID da estação a importar}
                           {--inicio= : Data de início (Y-m-d H:i)}
                           {--fim= : Data de fim (Y-m-d H:i)}
                           {--force : Forçar importação mesmo se desabilitada}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Importar dados de uma estação específica do SEBA-HYDROMETRIE';

    protected $sebaService;

    public function __construct(SebaHydrometrieService $sebaService)
    {
        parent::__construct();
        $this->sebaService = $sebaService;
    }

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $estacaoId = $this->argument('estacao_id');

        // Buscar estação
        $estacao = Estacao::find($estacaoId);
        if (!$estacao) {
            $this->error("Estação com ID {$estacaoId} não encontrada.");
            return Command::FAILURE;
        }

        // Verificar se tem integração SEBA configurada
        if (!$estacao->seba_station_id) {
            $this->error("Estação '{$estacao->nome}' não possui integração SEBA configurada.");
            return Command::FAILURE;
        }

        // Verificar se a importação automática está ativa (a menos que forçado)
        if (!$this->option('force') && !$estacao->auto_importacao_ativa) {
            $this->warn("Estação '{$estacao->nome}' não tem importação automática ativa. Use --force para forçar.");
            return Command::SUCCESS;
        }

        $this->info("Iniciando importação da estação: {$estacao->nome}");
        $this->info("SEBA Station ID: {$estacao->seba_station_id}");

        // Determinar período
        $dataInicio = $this->option('inicio') ? Carbon::createFromFormat('Y-m-d H:i', $this->option('inicio')) : null;
        $dataFim = $this->option('fim') ? Carbon::createFromFormat('Y-m-d H:i', $this->option('fim')) : null;

        if ($dataInicio && $dataFim) {
            $this->info("Período: {$dataInicio->format('d/m/Y H:i')} - {$dataFim->format('d/m/Y H:i')}");
        } else {
            $this->info("Período: Últimas 24 horas");
        }

        // Barra de progresso indeterminada
        $this->output->write('Importando dados');

        try {
            $resultado = $this->sebaService->importarDadosEstacao($estacao, $dataInicio, $dataFim);

            $this->newLine();

            if ($resultado['success']) {
                $this->info("✅ " . $resultado['message']);

                if ($resultado['imported'] > 0) {
                    $this->table(
                        ['Métrica', 'Valor'],
                        [
                            ['Registros importados', $resultado['imported']],
                            ['Duplicados ignorados', $resultado['duplicates'] ?? 0],
                            ['Erros', $resultado['errors'] ?? 0]
                        ]
                    );

                    // Atualizar última sincronização
                    $estacao->update(['ultima_sincronizacao_seba' => now()]);
                }

                return Command::SUCCESS;
            } else {
                $this->error("❌ " . $resultado['message']);
                return Command::FAILURE;
            }

        } catch (\Exception $e) {
            $this->newLine();
            $this->error("❌ Erro durante importação: " . $e->getMessage());

            Log::error('Erro no comando seba:importar-estacao', [
                'estacao_id' => $estacaoId,
                'erro' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return Command::FAILURE;
        }
    }
}
