<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Services\ExternalDataService;
use App\Models\Estacao;
use App\Models\LeituraEstacao;

class UpdateExternalData extends Command
{
    protected $signature = 'data:update-external';
    protected $description = 'Atualizar dados hidrológicos de APIs externas';

    public function handle()
    {
        $this->info('Iniciando atualização de dados externos...');

        $externalService = new ExternalDataService();

        // Obter estações ativas do norte
        $estacoes = Estacao::where('estado', 'activa')
            ->whereNotNull('latitude')
            ->whereNotNull('longitude')
            ->where('provincia', 'like', '%Cabo Delgado%')
            ->orWhere('provincia', 'like', '%Nampula%')
            ->orWhere('provincia', 'like', '%Niassa%')
            ->get();

        $this->info("Encontradas {$estacoes->count()} estações para atualizar");

        foreach ($estacoes as $estacao) {
            try {
                $this->info("Atualizando dados para: {$estacao->nome}");

                $dadosConsolidados = $externalService->getConsolidatedData(
                    $estacao->id,
                    $estacao->latitude,
                    $estacao->longitude
                );

                if ($dadosConsolidados) {
                    // Criar leitura automática
                    LeituraEstacao::create([
                        'estacao_id' => $estacao->id,
                        'data_leitura' => now(),
                        'precipitacao' => $dadosConsolidados['meteorologia']['precipitacao'] ?? null,
                        'temperatura' => $dadosConsolidados['meteorologia']['temperatura'] ?? null,
                        'nivel_agua' => $dadosConsolidados['nivel_rio_simulado']['nivel_atual'] ?? null,
                        'validado' => false,
                        'observacoes' => 'Dados obtidos via API externa - ' . $dadosConsolidados['fonte_dados'],
                        'operador' => 'Sistema Automático'
                    ]);

                    $this->info("✓ Dados atualizados para {$estacao->nome}");
                } else {
                    $this->warn("✗ Falha ao obter dados para {$estacao->nome}");
                }

            } catch (\Exception $e) {
                $this->error("Erro ao atualizar {$estacao->nome}: " . $e->getMessage());
            }
        }

        $this->info('Atualização de dados externos concluída!');

        return 0;
    }
}