<?php

namespace App\Exports;

use App\Models\Boletim;
use Illuminate\Contracts\View\View;
use Maatwebsite\Excel\Concerns\FromView;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithColumnWidths;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Fill;

class BoletimExport implements FromView, ShouldAutoSize, WithStyles, WithColumnWidths
{
    protected $boletim;

    public function __construct(Boletim $boletim)
    {
        $this->boletim = $boletim;
    }

    public function view(): View
    {
        // Get data based on bulletin type
        $data = $this->getBoletimData();

        return view('exports.boletim', [
            'boletim' => $this->boletim,
            'data' => $data
        ]);
    }

    private function getBoletimData()
    {
        switch ($this->boletim->tipo) {
            case 'hidrologico_regional':
                return $this->getRegionalData();
            case 'hidrologico_provincial':
                return $this->getProvincialData();
            default:
                return $this->getInformativoBarragensData();
        }
    }

    private function getRegionalData()
    {
        return [
            'estacoes_hidrograficas' => [
                ['nome' => 'Rio Lúrio - Ponte da EN1', 'nivel' => '2.45m', 'volume' => '85%', 'status' => 'Normal'],
                ['nome' => 'Rio Monapo - Monapo Sede', 'nivel' => '1.80m', 'volume' => '65%', 'status' => 'Normal'],
                ['nome' => 'Rio Melúli - Mecubúri', 'nivel' => '1.25m', 'volume' => '45%', 'status' => 'Baixo'],
                ['nome' => 'Rio Ligonha - Lugela', 'nivel' => '2.10m', 'volume' => '70%', 'status' => 'Normal'],
            ],
            'barragens' => [
                ['nome' => 'Barragem de Corumana', 'capacidade' => '879 Mm³', 'nivel_atual' => '75%', 'entrada' => '15 m³/s', 'saida' => '12 m³/s'],
                ['nome' => 'Barragem de Pequenos Libombos', 'capacidade' => '400 Mm³', 'nivel_atual' => '68%', 'entrada' => '8 m³/s', 'saida' => '6 m³/s'],
                ['nome' => 'Barragem de Massingir', 'capacidade' => '2800 Mm³', 'nivel_atual' => '82%', 'entrada' => '25 m³/s', 'saida' => '20 m³/s'],
            ],
            'precipitacao' => [
                ['estacao' => 'Nampula', 'precipitacao_24h' => '12.5mm', 'precipitacao_mensal' => '125.8mm'],
                ['estacao' => 'Pemba', 'precipitacao_24h' => '8.2mm', 'precipitacao_mensal' => '98.4mm'],
                ['estacao' => 'Lichinga', 'precipitacao_24h' => '15.7mm', 'precipitacao_mensal' => '156.3mm'],
            ]
        ];
    }

    private function getProvincialData()
    {
        return [
            'barragens_nampula' => [
                ['nome' => 'Barragem de Mutamba', 'nivel_atual' => '72%', 'volume' => '28.8 Mm³', 'status' => 'Normal'],
                ['nome' => 'Barragem de Nacala', 'nivel_atual' => '65%', 'volume' => '19.5 Mm³', 'status' => 'Normal'],
                ['nome' => 'Mini Barragem de Ribaué', 'nivel_atual' => '45%', 'volume' => '4.5 Mm³', 'status' => 'Baixo'],
            ],
            'postos_udograficos' => [
                ['posto' => 'Nampula Cidade', 'nivel' => '1.85m', 'caudal' => '15.2 m³/s'],
                ['posto' => 'Monapo', 'nivel' => '2.10m', 'caudal' => '18.7 m³/s'],
                ['posto' => 'Angoche', 'nivel' => '1.65m', 'caudal' => '12.3 m³/s'],
            ]
        ];
    }

    private function getInformativoBarragensData()
    {
        return [
            'barragens_principais' => [
                ['nome' => 'Cahora Bassa', 'capacidade' => '63000 Mm³', 'nivel_atual' => '88%', 'producao_energia' => '2075 MW'],
                ['nome' => 'Chicamba Real', 'capacidade' => '540 Mm³', 'nivel_atual' => '76%', 'producao_energia' => '38 MW'],
                ['nome' => 'Mavuzi', 'capacidade' => '345 Mm³', 'nivel_atual' => '82%', 'producao_energia' => '52 MW'],
            ],
            'alertas' => [
                'Níveis normais em todas as barragens principais',
                'Caudais de entrada estáveis',
                'Produção energética dentro do esperado'
            ]
        ];
    }

    public function styles(Worksheet $sheet)
    {
        return [
            // Header style
            1 => [
                'font' => [
                    'bold' => true,
                    'size' => 16,
                    'color' => ['rgb' => 'FFFFFF']
                ],
                'fill' => [
                    'fillType' => Fill::FILL_SOLID,
                    'startColor' => ['rgb' => '2563EB']
                ],
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_CENTER,
                    'vertical' => Alignment::VERTICAL_CENTER
                ]
            ],
            // Sub-header style
            3 => [
                'font' => [
                    'bold' => true,
                    'size' => 14,
                    'color' => ['rgb' => '1E40AF']
                ],
                'fill' => [
                    'fillType' => Fill::FILL_SOLID,
                    'startColor' => ['rgb' => 'EBF4FF']
                ]
            ],
            // Table headers
            'A5:Z5' => [
                'font' => [
                    'bold' => true,
                    'color' => ['rgb' => 'FFFFFF']
                ],
                'fill' => [
                    'fillType' => Fill::FILL_SOLID,
                    'startColor' => ['rgb' => '3B82F6']
                ],
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_CENTER
                ]
            ]
        ];
    }

    public function columnWidths(): array
    {
        return [
            'A' => 25,
            'B' => 15,
            'C' => 15,
            'D' => 15,
            'E' => 15,
            'F' => 15,
            'G' => 20,
        ];
    }
}