<?php

namespace App\Http\Controllers;

use App\Models\BaciaHidrografica;
use Illuminate\Http\Request;

class BaciaHidrograficaController extends Controller
{
    public function index()
    {
        $query = BaciaHidrografica::query();

        // Apply filters
        if (request('search')) {
            $query->where(function($q) {
                $q->where('nome', 'like', '%' . request('search') . '%')
                  ->orWhere('codigo', 'like', '%' . request('search') . '%');
            });
        }

        if (request('provincia')) {
            $query->where('provincias_abrangidas', 'like', '%' . request('provincia') . '%');
        }

        $bacias = $query->withCount(['barragens', 'estacoes'])
                       ->orderBy('nome')
                       ->paginate(15);

        return view('bacias.index', compact('bacias'));
    }

    public function create()
    {
        return view('bacias.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'nome' => 'required|string|max:255',
            'codigo' => 'required|string|max:50|unique:bacias_hidrograficas',
            'area_km2' => 'nullable|numeric|min:0',
            'comprimento_rio_principal' => 'nullable|numeric|min:0',
            'provincias_abrangidas' => 'required|array|min:1',
            'provincias_abrangidas.*' => 'string',
            'descricao' => 'nullable|string',
        ]);

        // Converter array de províncias em string separada por vírgulas
        $validated['provincias_abrangidas'] = implode(',', $validated['provincias_abrangidas']);

        BaciaHidrografica::create($validated);

        return redirect()->route('bacias.index')
            ->with('success', 'Bacia hidrográfica criada com sucesso!');
    }

    public function show(BaciaHidrografica $bacia)
    {
        // Simplificado temporariamente para resolver timeout
        return view('bacias.show', compact('bacia'));
    }

    public function edit(BaciaHidrografica $bacia)
    {
        return view('bacias.edit', compact('bacia'));
    }

    public function update(Request $request, BaciaHidrografica $bacia)
    {
        $validated = $request->validate([
            'nome' => 'required|string|max:255',
            'codigo' => 'required|string|max:50|unique:bacias_hidrograficas,codigo,' . $bacia->id,
            'area_km2' => 'nullable|numeric|min:0',
            'comprimento_rio_principal' => 'nullable|numeric|min:0',
            'provincias_abrangidas' => 'required|array|min:1',
            'provincias_abrangidas.*' => 'string',
            'descricao' => 'nullable|string',
        ]);

        // Converter array de províncias em string separada por vírgulas
        $validated['provincias_abrangidas'] = implode(',', $validated['provincias_abrangidas']);

        $bacia->update($validated);

        return redirect()->route('bacias.show', $bacia)
            ->with('success', 'Bacia hidrográfica atualizada com sucesso!');
    }

    public function destroy(BaciaHidrografica $bacia)
    {
        try {
            $bacia->delete();
            return redirect()->route('bacias.index')
                ->with('success', 'Bacia hidrográfica removida com sucesso!');
        } catch (\Exception $e) {
            return redirect()->route('bacias.index')
                ->with('error', 'Erro ao remover bacia. Verifique se não há barragens ou estações dependentes.');
        }
    }

    /**
     * API endpoint para buscar bacias
     */
    public function api()
    {
        $bacias = BaciaHidrografica::select('id', 'nome', 'codigo', 'provincias_abrangidas')
                                 ->orderBy('nome')
                                 ->get()
                                 ->map(function ($bacia) {
                                     return [
                                         'id' => $bacia->id,
                                         'nome' => $bacia->nome,
                                         'codigo' => $bacia->codigo,
                                         'provincias' => $bacia->getProvinciasList(),
                                         'display' => $bacia->nome . ' (' . $bacia->codigo . ')'
                                     ];
                                 });

        return response()->json($bacias);
    }
}