<?php

namespace App\Http\Controllers;

use App\Models\Barragem;
use App\Models\BaciaHidrografica;
use App\Models\LeituraBarragem;
use App\Models\Alerta;
use App\Helpers\HydrologicalYear;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class BarragemController extends Controller
{
    public function index()
    {
        $query = Barragem::with(['bacia_hidrografica', 'ultima_leitura']);

        // Apply filters
        if (request('search')) {
            $query->where(function($q) {
                $q->where('nome', 'like', '%' . request('search') . '%')
                  ->orWhere('codigo', 'like', '%' . request('search') . '%');
            });
        }

        if (request('estado')) {
            $query->where('estado', request('estado'));
        }

        if (request('provincia')) {
            $query->where('provincia', 'like', '%' . request('provincia') . '%');
        }

        $barragens = $query->orderBy('nome')->paginate(15);

        return view('barragens.index', compact('barragens'));
    }

    public function create()
    {
        $bacias = BaciaHidrografica::orderBy('nome')->get();
        return view('barragens.create', compact('bacias'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            // Identificação
            'nome' => 'required|string|max:255',
            'codigo' => 'required|string|max:50|unique:barragens',
            'bacia_hidrografica_id' => 'nullable|exists:bacias_hidrograficas,id',
            'provincia' => 'nullable|string|max:100',
            'distrito' => 'nullable|string|max:100',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180',
            
            // Características de Volume
            'volume_maximo' => 'nullable|numeric|min:0',
            'volume_morto' => 'nullable|numeric|min:0',
            'volume_util' => 'nullable|numeric|min:0',
            'capacidade_total' => 'nullable|numeric|min:0',
            
            // Cotas e Níveis
            'cota_maxima' => 'nullable|numeric|min:0',
            'cota_minima' => 'nullable|numeric|min:0',
            'cota_npa' => 'nullable|numeric|min:0',
            'cota_nme' => 'nullable|numeric|min:0',
            'cota_nmc' => 'nullable|numeric|min:0',
            'nivel_descargador' => 'nullable|numeric|min:0',
            'cota_seguranca' => 'nullable|numeric|min:0',
            'nivel_alerta' => 'nullable|numeric|min:0',
            'nivel_emergencia' => 'nullable|numeric|min:0',
            
            // Características Físicas
            'area_bacia_hidraulica' => 'nullable|numeric|min:0',
            'area_inundacao' => 'nullable|numeric|min:0',
            'altura_barragem' => 'nullable|numeric|min:0',
            'comprimento_crista' => 'nullable|numeric|min:0',
            
            // Características Hidrológicas
            'rio_principal' => 'nullable|string|max:100',
            'afluentes' => 'nullable|string',
            'estacoes_pluviometricas' => 'nullable|array',
            'estacoes_pluviometricas.*' => 'exists:estacoes,id',
            
            // Classificação Técnica
            'tipo_barragem' => 'nullable|in:terra,betao,enrocamento,mista,gravidade,arco,contraforte',
            'finalidade_principal' => 'nullable|in:abastecimento,irrigacao,energia,controle_cheias,recreacao,piscicultura,multipla',
            'tipo_descargador' => 'nullable|in:superficie,fundo,lateral,tunel,misto',
            'classe_risco' => 'nullable|in:baixo,medio,alto',
            'categoria_dano' => 'nullable|in:baixo,medio,alto',
            
            // Informações de Projeto
            'ano_construcao' => 'nullable|integer|min:1900|max:' . date('Y'),
            'empresa_projetista' => 'nullable|string|max:200',
            'empresa_construtora' => 'nullable|string|max:200',
            'ano_projeto' => 'nullable|integer|min:1900|max:' . date('Y'),
            'ano_inicio_construcao' => 'nullable|integer|min:1900|max:' . date('Y'),
            'ano_conclusao' => 'nullable|integer|min:1900|max:' . date('Y'),
            
            // Características Operacionais
            'sistema_automatizado' => 'nullable|boolean',
            'telemetria' => 'nullable|boolean',
            'frequencia_monitoramento' => 'nullable|in:diario,semanal,mensal,continuo',
            'estado' => 'required|in:activa,inactiva,manutencao',
            'observacoes' => 'nullable|string',
        ]);
        
        // Processar arrays para campos JSON
        if (isset($validated['afluentes']) && is_string($validated['afluentes'])) {
            $validated['afluentes'] = array_map('trim', explode(',', $validated['afluentes']));
        }
        
        // estacoes_pluviometricas já vem como array dos IDs selecionados

        Barragem::create($validated);

        return redirect()->route('barragens.index')
            ->with('success', 'Barragem criada com sucesso!');
    }

    public function show(Barragem $barragem)
    {
        $barragem->load(['bacia_hidrografica', 'ultima_leitura']);
        
        // Últimas 10 leituras
        $leituras_recentes = $barragem->leituras()
            ->with('operador')
            ->orderBy('data_leitura', 'desc')
            ->orderBy('hora_leitura', 'desc')
            ->take(10)
            ->get();

        // Alertas ativos
        $alertas_ativos = $barragem->alertas()
            ->where('estado', 'activo')
            ->orderBy('created_at', 'desc')
            ->get();

        // Dados para gráfico de enchimento (últimos 30 dias)
        $historico_enchimento = $barragem->leituras()
            ->where('data_leitura', '>=', now()->subDays(30))
            ->orderBy('data_leitura')
            ->get(['data_leitura', 'percentagem_enchimento', 'cota_actual']);

        return view('barragens.show', compact(
            'barragem',
            'leituras_recentes',
            'alertas_ativos',
            'historico_enchimento'
        ));
    }

    public function edit(Barragem $barragem)
    {
        $bacias = BaciaHidrografica::orderBy('nome')->get();
        return view('barragens.edit', compact('barragem', 'bacias'));
    }

    public function update(Request $request, Barragem $barragem)
    {
        $validated = $request->validate([
            // Identificação
            'nome' => 'required|string|max:255',
            'codigo' => 'required|string|max:50|unique:barragens,codigo,' . $barragem->id,
            'bacia_hidrografica_id' => 'nullable|exists:bacias_hidrograficas,id',
            'provincia' => 'nullable|string|max:100',
            'distrito' => 'nullable|string|max:100',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180',
            
            // Características de Volume
            'volume_maximo' => 'nullable|numeric|min:0',
            'volume_morto' => 'nullable|numeric|min:0',
            'volume_util' => 'nullable|numeric|min:0',
            'capacidade_total' => 'nullable|numeric|min:0',
            
            // Cotas e Níveis
            'cota_maxima' => 'nullable|numeric|min:0',
            'cota_minima' => 'nullable|numeric|min:0',
            'cota_npa' => 'nullable|numeric|min:0',
            'cota_nme' => 'nullable|numeric|min:0',
            'cota_nmc' => 'nullable|numeric|min:0',
            'nivel_descargador' => 'nullable|numeric|min:0',
            'cota_seguranca' => 'nullable|numeric|min:0',
            'nivel_alerta' => 'nullable|numeric|min:0',
            'nivel_emergencia' => 'nullable|numeric|min:0',
            
            // Características Físicas
            'area_bacia_hidraulica' => 'nullable|numeric|min:0',
            'area_inundacao' => 'nullable|numeric|min:0',
            'altura_barragem' => 'nullable|numeric|min:0',
            'comprimento_crista' => 'nullable|numeric|min:0',
            
            // Características Hidrológicas
            'rio_principal' => 'nullable|string|max:100',
            'afluentes' => 'nullable|string',
            'estacoes_pluviometricas' => 'nullable|array',
            'estacoes_pluviometricas.*' => 'exists:estacoes,id',
            
            // Classificação Técnica
            'tipo_barragem' => 'nullable|in:terra,betao,enrocamento,mista,gravidade,arco,contraforte',
            'finalidade_principal' => 'nullable|in:abastecimento,irrigacao,energia,controle_cheias,recreacao,piscicultura,multipla',
            'tipo_descargador' => 'nullable|in:superficie,fundo,lateral,tunel,misto',
            'classe_risco' => 'nullable|in:baixo,medio,alto',
            'categoria_dano' => 'nullable|in:baixo,medio,alto',
            
            // Informações de Projeto
            'ano_construcao' => 'nullable|integer|min:1900|max:' . date('Y'),
            'empresa_projetista' => 'nullable|string|max:200',
            'empresa_construtora' => 'nullable|string|max:200',
            'ano_projeto' => 'nullable|integer|min:1900|max:' . date('Y'),
            'ano_inicio_construcao' => 'nullable|integer|min:1900|max:' . date('Y'),
            'ano_conclusao' => 'nullable|integer|min:1900|max:' . date('Y'),
            
            // Características Operacionais
            'sistema_automatizado' => 'nullable|boolean',
            'telemetria' => 'nullable|boolean',
            'frequencia_monitoramento' => 'nullable|in:diario,semanal,mensal,continuo',
            'estado' => 'required|in:activa,inactiva,manutencao',
            'observacoes' => 'nullable|string',
        ]);
        
        // Processar arrays para campos JSON
        if (isset($validated['afluentes']) && is_string($validated['afluentes'])) {
            $validated['afluentes'] = array_map('trim', explode(',', $validated['afluentes']));
        }
        
        // estacoes_pluviometricas já vem como array dos IDs selecionados

        $barragem->update($validated);

        return redirect()->route('barragens.show', $barragem)
            ->with('success', 'Barragem atualizada com sucesso!');
    }

    public function destroy(Barragem $barragem)
    {
        try {
            $barragem->delete();
            return redirect()->route('barragens.index')
                ->with('success', 'Barragem removida com sucesso!');
        } catch (\Exception $e) {
            return redirect()->route('barragens.index')
                ->with('error', 'Erro ao remover barragem. Verifique se não há dados dependentes.');
        }
    }

    public function leituras(Barragem $barragem)
    {
        $leituras = $barragem->leituras()
            ->with(['operador', 'validador'])
            ->orderBy('data_leitura', 'desc')
            ->orderBy('hora_leitura', 'desc')
            ->paginate(20);

        return view('barragens.leituras', compact('barragem', 'leituras'));
    }

    public function storeLeitura(Request $request, Barragem $barragem)
    {
        $validated = $request->validate([
            'data_leitura' => 'required|date',
            'hora_leitura' => 'required|date_format:H:i',
            'cota_actual' => 'required|numeric|min:0',
            'volume_actual' => 'nullable|numeric|min:0',
            'volume_morto' => 'nullable|numeric|min:0',
            'percentagem_total' => 'nullable|numeric|min:0|max:100',
            'volume_armazenado' => 'nullable|numeric|min:0',
            'caudal_captacao' => 'nullable|numeric|min:0',
            'caudal_afluente' => 'nullable|numeric|min:0',
            'caudal_efluente' => 'nullable|numeric|min:0',
            'descarga_fundo' => 'nullable|numeric|min:0',
            'descarga_superficie' => 'nullable|numeric|min:0',
            'caudal_ecologico' => 'nullable|numeric|min:0',
            'caudal_turbinado' => 'nullable|numeric|min:0',
            'caudal_descarregador' => 'nullable|numeric|min:0',
            'evaporacao' => 'nullable|numeric|min:0',
            'infiltracao' => 'nullable|numeric|min:0',
            'precipitacao' => 'nullable|numeric|min:0',
            'metodo_leitura' => 'required|in:manual,automatico,telemetria',
            'observacoes' => 'nullable|string',
        ]);

        $validated['barragem_id'] = $barragem->id;
        $validated['operador_id'] = Auth::id();

        $leitura = LeituraBarragem::create($validated);

        // Verificar se deve criar alerta
        Alerta::criarAlertaBarragem($barragem, $leitura);

        return redirect()->route('barragens.leituras', $barragem)
            ->with('success', 'Leitura registrada com sucesso!');
    }

    public function validateLeitura(Request $request, Barragem $barragem, LeituraBarragem $leitura)
    {
        $leitura->update([
            'validado' => true,
            'validado_por' => Auth::id(),
            'data_validacao' => now(),
        ]);

        return redirect()->back()
            ->with('success', 'Leitura validada com sucesso!');
    }

    public function getHistorico(Barragem $barragem, Request $request)
    {
        $periodo = $request->get('periodo', 30); // dias
        
        $historico = $barragem->leituras()
            ->where('data_leitura', '>=', now()->subDays($periodo))
            ->orderBy('data_leitura')
            ->get(['data_leitura', 'cota_actual', 'percentagem_enchimento', 'volume_actual']);

        return response()->json($historico);
    }

    public function getChartData(Barragem $barragem, Request $request)
    {
        $period = $request->get('period', '7d'); // 7d, 30d, 1y

        $startDate = match($period) {
            '7d' => now()->subDays(7),
            '30d' => now()->subDays(30),
            '1y' => now()->subYear(),
            default => now()->subDays(7)
        };

        $leituras = $barragem->leituras()
            ->where('data_leitura', '>=', $startDate)
            ->orderBy('data_leitura')
            ->orderBy('hora_leitura')
            ->get([
                'data_leitura',
                'hora_leitura',
                'cota_actual',
                'volume_actual',
                'percentagem_enchimento',
                'caudal_afluente',
                'caudal_efluente'
            ]);

        // Formatar dados para Chart.js
        $data = [
            'labels' => $leituras->map(function($leitura) {
                return $leitura->data_leitura->format('d/m');
            })->values(),
            'datasets' => [
                [
                    'label' => 'Cota Atual (m)',
                    'data' => $leituras->pluck('cota_actual')->values(),
                    'borderColor' => 'rgb(59, 130, 246)',
                    'backgroundColor' => 'rgba(59, 130, 246, 0.1)',
                    'tension' => 0.1,
                    'yAxisID' => 'y'
                ],
                [
                    'label' => 'Volume Atual (Mm³)',
                    'data' => $leituras->pluck('volume_actual')->values(),
                    'borderColor' => 'rgb(34, 197, 94)',
                    'backgroundColor' => 'rgba(34, 197, 94, 0.1)',
                    'tension' => 0.1,
                    'yAxisID' => 'y1'
                ],
                [
                    'label' => 'Enchimento (%)',
                    'data' => $leituras->pluck('percentagem_enchimento')->values(),
                    'borderColor' => 'rgb(168, 85, 247)',
                    'backgroundColor' => 'rgba(168, 85, 247, 0.1)',
                    'tension' => 0.1,
                    'yAxisID' => 'y2'
                ]
            ],
            'safety_levels' => [
                'npa' => $barragem->cota_npa,
                'nme' => $barragem->cota_nme,
                'nmc' => $barragem->cota_nmc,
                'alerta' => $barragem->nivel_alerta,
                'emergencia' => $barragem->nivel_emergencia,
                'maxima' => $barragem->cota_maxima
            ]
        ];

        return response()->json($data);
    }

    public function editLeitura(Barragem $barragem, LeituraBarragem $leitura)
    {
        // Verificar se a leitura pertence à barragem
        if ($leitura->barragem_id !== $barragem->id) {
            abort(404);
        }

        return view('barragens.edit-leitura', compact('barragem', 'leitura'));
    }

    public function updateLeitura(Request $request, Barragem $barragem, LeituraBarragem $leitura)
    {
        // Verificar se a leitura pertence à barragem
        if ($leitura->barragem_id !== $barragem->id) {
            abort(404);
        }

        $validated = $request->validate([
            'data_leitura' => 'required|date',
            'hora_leitura' => 'required|date_format:H:i',
            'cota_actual' => 'required|numeric|min:0',
            'volume_actual' => 'nullable|numeric|min:0',
            'volume_morto' => 'nullable|numeric|min:0',
            'percentagem_total' => 'nullable|numeric|min:0|max:100',
            'volume_armazenado' => 'nullable|numeric|min:0',
            'caudal_captacao' => 'nullable|numeric|min:0',
            'caudal_afluente' => 'nullable|numeric|min:0',
            'caudal_efluente' => 'nullable|numeric|min:0',
            'descarga_fundo' => 'nullable|numeric|min:0',
            'descarga_superficie' => 'nullable|numeric|min:0',
            'caudal_ecologico' => 'nullable|numeric|min:0',
            'caudal_turbinado' => 'nullable|numeric|min:0',
            'caudal_descarregador' => 'nullable|numeric|min:0',
            'evaporacao' => 'nullable|numeric|min:0',
            'infiltracao' => 'nullable|numeric|min:0',
            'precipitacao' => 'nullable|numeric|min:0',
            'metodo_leitura' => 'required|in:manual,automatico,telemetria',
            'observacoes' => 'nullable|string',
        ]);

        $leitura->update($validated);

        // Verificar se deve criar/atualizar alerta
        Alerta::criarAlertaBarragem($barragem, $leitura);

        return redirect()->route('barragens.leituras', $barragem)
            ->with('success', 'Leitura atualizada com sucesso!');
    }

    public function destroyLeitura(Barragem $barragem, LeituraBarragem $leitura)
    {
        // Verificar se a leitura pertence à barragem
        if ($leitura->barragem_id !== $barragem->id) {
            abort(404);
        }

        try {
            $leitura->delete();
            return redirect()->route('barragens.leituras', $barragem)
                ->with('success', 'Leitura removida com sucesso!');
        } catch (\Exception $e) {
            return redirect()->route('barragens.leituras', $barragem)
                ->with('error', 'Erro ao remover leitura. Verifique se não há dados dependentes.');
        }
    }
}