<?php

namespace App\Http\Controllers;

use App\Models\Barragem;
use App\Models\CurvaVolume;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class CurvaVolumeController extends Controller
{
    public function index(Barragem $barragem)
    {
        $curvas = $barragem->curvas_volume()
            ->with('criador')
            ->orderBy('cota', 'asc')
            ->get();

        return view('barragens.curvas-volume', compact('barragem', 'curvas'));
    }

    public function store(Request $request, Barragem $barragem)
    {
        $validated = $request->validate([
            'cota' => 'required|numeric|min:0',
            'volume' => 'required|numeric|min:0',
            'area' => 'nullable|numeric|min:0',
            'observacoes' => 'nullable|string',
        ]);

        $validated['barragem_id'] = $barragem->id;
        $validated['criado_por'] = Auth::id();

        CurvaVolume::create($validated);

        return redirect()->route('barragens.curvas-volume.index', $barragem)
            ->with('success', 'Ponto da curva adicionado com sucesso!');
    }

    public function update(Request $request, Barragem $barragem, CurvaVolume $curva)
    {
        $validated = $request->validate([
            'cota' => 'required|numeric|min:0',
            'volume' => 'required|numeric|min:0',
            'area' => 'nullable|numeric|min:0',
            'observacoes' => 'nullable|string',
        ]);

        $curva->update($validated);

        return redirect()->route('barragens.curvas-volume.index', $barragem)
            ->with('success', 'Ponto da curva atualizado com sucesso!');
    }

    public function destroy(Barragem $barragem, CurvaVolume $curva)
    {
        $curva->delete();

        return redirect()->route('barragens.curvas-volume.index', $barragem)
            ->with('success', 'Ponto da curva removido com sucesso!');
    }

    public function getChartData(Barragem $barragem)
    {
        $curvas = $barragem->curvas_volume()
            ->orderBy('cota', 'asc')
            ->get(['cota', 'volume', 'area']);

        $data = [
            'labels' => $curvas->pluck('cota')->map(fn($cota) => number_format($cota, 2))->values(),
            'datasets' => [
                [
                    'label' => 'Volume (Mm³)',
                    'data' => $curvas->pluck('volume')->values(),
                    'borderColor' => 'rgb(59, 130, 246)',
                    'backgroundColor' => 'rgba(59, 130, 246, 0.1)',
                    'tension' => 0.2,
                    'yAxisID' => 'y'
                ]
            ]
        ];

        if ($curvas->contains('area', '!=', null)) {
            $data['datasets'][] = [
                'label' => 'Área (km²)',
                'data' => $curvas->pluck('area')->values(),
                'borderColor' => 'rgb(34, 197, 94)',
                'backgroundColor' => 'rgba(34, 197, 94, 0.1)',
                'tension' => 0.2,
                'yAxisID' => 'y1'
            ];
        }

        return response()->json($data);
    }
}