<?php

namespace App\Http\Controllers;

use App\Models\Estacao;
use App\Models\Barragem;
use Illuminate\Http\Request;
use Illuminate\View\View;

class MapController extends Controller
{
    /**
     * Exibir mapa interativo com localização específica
     */
    public function show(Request $request): View
    {
        $tipo = $request->get('tipo');
        $id = $request->get('id');
        $lat = $request->get('lat');
        $lng = $request->get('lng');

        $itemSelecionado = null;
        $todasEstacoes = collect();
        $todasBarragens = collect();

        // Buscar o item específico selecionado
        if ($tipo && $id) {
            if ($tipo === 'estacao') {
                $itemSelecionado = Estacao::with('ultima_leitura')->findOrFail($id);
            } elseif ($tipo === 'barragem') {
                $itemSelecionado = Barragem::with('ultima_leitura')->findOrFail($id);
            }
        }

        // Buscar todas as estações e barragens para mostrar no contexto
        $todasEstacoes = Estacao::where('estado', 'activa')
                               ->whereNotNull('latitude')
                               ->whereNotNull('longitude')
                               ->get();

        $todasBarragens = Barragem::where('estado', 'activa')
                                 ->whereNotNull('latitude')
                                 ->whereNotNull('longitude')
                                 ->get();

        return view('mapas.show', compact(
            'tipo',
            'id',
            'lat',
            'lng',
            'itemSelecionado',
            'todasEstacoes',
            'todasBarragens'
        ));
    }

    /**
     * API para obter dados do mapa em JSON
     */
    public function dados(Request $request)
    {
        $estacoes = Estacao::where('estado', 'activa')
                          ->whereNotNull('latitude')
                          ->whereNotNull('longitude')
                          ->with('ultima_leitura')
                          ->get()
                          ->map(function($estacao) {
                              return [
                                  'id' => $estacao->id,
                                  'nome' => $estacao->nome,
                                  'tipo' => 'estacao',
                                  'subtipo' => $estacao->tipo,
                                  'lat' => (float) $estacao->latitude,
                                  'lng' => (float) $estacao->longitude,
                                  'status' => $estacao->estado,
                                  'nivel_atual' => $estacao->ultima_leitura?->nivel_hidrometrico,
                                  'url' => route('estacoes.show', $estacao->id)
                              ];
                          });

        $barragens = Barragem::where('estado', 'activa')
                            ->whereNotNull('latitude')
                            ->whereNotNull('longitude')
                            ->with('ultima_leitura')
                            ->get()
                            ->map(function($barragem) {
                                return [
                                    'id' => $barragem->id,
                                    'nome' => $barragem->nome,
                                    'tipo' => 'barragem',
                                    'lat' => (float) $barragem->latitude,
                                    'lng' => (float) $barragem->longitude,
                                    'cota_atual' => $barragem->ultima_leitura?->cota_actual,
                                    'nivel_alerta' => $barragem->nivel_alerta,
                                    'capacidade' => $barragem->capacidade_total,
                                    'url' => route('barragens.show', $barragem->id)
                                ];
                            });

        return response()->json([
            'estacoes' => $estacoes,
            'barragens' => $barragens
        ]);
    }

    /**
     * Servir shapes como GeoJSON para o mapa
     */
    public function getShape(Request $request, $shape)
    {
        $shapesPath = base_path('ficheiros/Shaps ARA-N,IP/');
        $allowedShapes = [
            'ara_norte' => 'ARA_Norte_IP.shp',
            'rios' => 'Rios.shp',
            'barragens' => 'Barragens.shp',
            'albufeiras' => 'Albufeiras.shp',
            'bacia_licungo' => 'Bacia_Licungo.shp',
            'bacia_messalo' => 'Bacia_Messalo.shp',
            'bacia_rovuma' => 'Bacia_Rovuma.shp',
            'bacia_monapo' => 'Bacia_Monapo.shp',
            'bacia_meluli' => 'Bacia_Meluli.shp',
            'bacia_mecuburi' => 'Bacia_Mecuburi.shp',
            'megarruma' => 'Megarruma.shp',
            'dgbl' => 'DGBL.shp',
            'dgblic' => 'DGBLic.shp',
            'dgbm' => 'DGBM.shp',
            'dgbr' => 'DGBR.shp',
        ];

        if (!isset($allowedShapes[$shape])) {
            return response()->json(['error' => 'Shape não encontrado'], 404);
        }

        $shapeFile = $shapesPath . $allowedShapes[$shape];

        if (!file_exists($shapeFile)) {
            return response()->json(['error' => 'Arquivo shape não encontrado'], 404);
        }

        // Cache simples - verificar se já temos GeoJSON convertido
        $cacheFile = storage_path("app/shapes_cache/{$shape}.geojson");

        if (file_exists($cacheFile) && filemtime($cacheFile) > filemtime($shapeFile)) {
            $geojson = file_get_contents($cacheFile);
            return response($geojson)->header('Content-Type', 'application/json');
        }

        // Para demonstração, retornar dados simulados baseados no tipo
        $geojson = $this->generateSimulatedShape($shape);

        // Salvar no cache
        if (!is_dir(dirname($cacheFile))) {
            mkdir(dirname($cacheFile), 0755, true);
        }
        file_put_contents($cacheFile, $geojson);

        return response($geojson)->header('Content-Type', 'application/json');
    }

    private function generateSimulatedShape($shape)
    {
        // Dados simulados baseados nas coordenadas reais da ARA Norte
        $shapes = [
            'ara_norte' => [
                'type' => 'FeatureCollection',
                'features' => [[
                    'type' => 'Feature',
                    'properties' => ['name' => 'ARA Norte', 'description' => 'Administração Regional de Águas do Norte'],
                    'geometry' => [
                        'type' => 'Polygon',
                        'coordinates' => [[
                            [36.0, -10.5], [41.0, -10.5], [41.0, -17.0], [36.0, -17.0], [36.0, -10.5]
                        ]]
                    ]
                ]]
            ],
            'bacia_rovuma' => [
                'type' => 'FeatureCollection',
                'features' => [[
                    'type' => 'Feature',
                    'properties' => ['name' => 'Bacia do Rovuma', 'area_km2' => 99524],
                    'geometry' => [
                        'type' => 'Polygon',
                        'coordinates' => [[
                            [38.0, -10.47], [40.44, -10.47], [40.44, -13.5], [37.0, -13.5], [38.0, -10.47]
                        ]]
                    ]
                ]]
            ],
            'rios' => [
                'type' => 'FeatureCollection',
                'features' => [
                    [
                        'type' => 'Feature',
                        'properties' => ['name' => 'Rio Rovuma', 'tipo' => 'internacional'],
                        'geometry' => [
                            'type' => 'LineString',
                            'coordinates' => [[38.0, -10.47], [39.0, -10.6], [40.0, -10.8], [40.44, -10.9]]
                        ]
                    ],
                    [
                        'type' => 'Feature',
                        'properties' => ['name' => 'Rio Lúrio', 'tipo' => 'nacional'],
                        'geometry' => [
                            'type' => 'LineString',
                            'coordinates' => [[36.8, -13.2], [38.0, -14.0], [39.2, -14.6], [40.1, -15.0]]
                        ]
                    ]
                ]
            ]
        ];

        return json_encode($shapes[$shape] ?? [
            'type' => 'FeatureCollection',
            'features' => []
        ]);
    }

    /**
     * Servir arquivos shapefile (.shp, .dbf, .shx) diretamente
     */
    public function getShapefile(Request $request, $shape, $type)
    {
        $shapesPath = base_path('ficheiros/Shaps ARA-N,IP/');
        $allowedShapes = [
            'ara_norte' => 'ARA_Norte_IP',
            'rios' => 'Rios',
            'barragens' => 'Barragens',
            'albufeiras' => 'Albufeiras',
            'bacia_licungo' => 'Bacia_Licungo',
            'bacia_messalo' => 'Bacia_Messalo',
            'bacia_rovuma' => 'Bacia_Rovuma',
            'bacia_monapo' => 'Bacia_Monapo',
            'bacia_meluli' => 'Bacia_Meluli',
            'bacia_mecuburi' => 'Bacia_Mecuburi',
            'megarruma' => 'Megarruma',
            'dgbl' => 'DGBL',
            'dgblic' => 'DGBLic',
            'dgbm' => 'DGBM',
            'dgbr' => 'DGBR',
        ];

        $allowedTypes = ['shp', 'dbf', 'shx', 'prj'];

        if (!isset($allowedShapes[$shape]) || !in_array($type, $allowedTypes)) {
            return response()->json(['error' => 'Arquivo não permitido'], 404);
        }

        $fileName = $allowedShapes[$shape] . '.' . $type;
        $filePath = $shapesPath . $fileName;

        if (!file_exists($filePath)) {
            return response()->json(['error' => 'Arquivo não encontrado'], 404);
        }

        // Definir content type apropriado
        $contentTypes = [
            'shp' => 'application/octet-stream',
            'dbf' => 'application/dbf',
            'shx' => 'application/octet-stream',
            'prj' => 'text/plain'
        ];

        return response()->file($filePath, [
            'Content-Type' => $contentTypes[$type] ?? 'application/octet-stream',
            'Content-Disposition' => 'inline; filename="' . $fileName . '"'
        ]);
    }
}