<?php
// app/Models/Boletim.php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class Boletim extends Model
{
    use HasFactory;

    protected $table = 'boletins';

    protected $fillable = [
        'numero_edicao',
        'tipo',
        'data_emissao',
        'hora_emissao',
        'periodo_inicio',
        'periodo_fim',
        'situacao_prevalecente',
        'previsao_hidrologica',
        'alertas_precaucoes',
        'dados_completos',
        'elaborado_por',
        'aprovado_por',
        'estado',
        'arquivo_pdf',
    ];

    protected $casts = [
        'data_emissao' => 'date',
        'periodo_inicio' => 'date',
        'periodo_fim' => 'date',
        'dados_completos' => 'array',
    ];

    public function elaborador(): BelongsTo
    {
        return $this->belongsTo(User::class, 'elaborado_por');
    }

    public function aprovador(): BelongsTo
    {
        return $this->belongsTo(User::class, 'aprovado_por');
    }

    public function gerarNumeroEdicao()
    {
        $ano = $this->data_emissao->year;

        // Meses em português
        $meses = [
            1 => 'JAN', 2 => 'FEV', 3 => 'MAR', 4 => 'ABR',
            5 => 'MAI', 6 => 'JUN', 7 => 'JUL', 8 => 'AGO',
            9 => 'SET', 10 => 'OUT', 11 => 'NOV', 12 => 'DEZ'
        ];

        $mes = $meses[$this->data_emissao->month];

        // Pegar o último número sequencial do ano
        $ultimo_numero = static::where('tipo', $this->tipo)
            ->whereYear('data_emissao', $ano)
            ->whereRaw("numero_edicao LIKE '%/{$mes}/DRH%'")
            ->count();

        $numero_sequencial = $ultimo_numero + 1;

        // Determinar o ano letivo/hidrológico
        $ano_hidrologico = $this->data_emissao->month >= 10 ? $ano . '-' . ($ano + 1) : ($ano - 1) . '-' . $ano;

        // Formato: {sequencial}/{MES}/DRH - {numero_padronizado} ({ano_hidrologico})
        $numero_padronizado = sprintf('%03d', $numero_sequencial);

        $this->numero_edicao = "{$numero_sequencial}/{$mes}/DRH - {$numero_padronizado} ({$ano_hidrologico})";
    }

    public function podeSerEditado()
    {
        return in_array($this->estado, ['rascunho', 'revisao']);
    }

    public function podeSerAprovado()
    {
        return $this->estado === 'revisao';
    }

    public function podeSerPublicado()
    {
        return $this->estado === 'aprovado';
    }

    public function getTipoFormatado()
    {
        $tipos = [
            'diario' => 'Boletim Diário',
            'semanal' => 'Boletim Semanal',
            'mensal' => 'Boletim Mensal',
            'emergencia' => 'Boletim de Emergência',
            'hidrologico_regional' => 'Boletim Hidrológico Regional',
            'hidrologico_provincial' => 'Boletim Hidrológico da Província',
            'informativo_barragens' => 'Boletim Informativo de Barragens',
        ];

        return $tipos[$this->tipo] ?? $this->tipo;
    }

    public function getEstadoFormatado()
    {
        $estados = [
            'rascunho' => 'Rascunho',
            'revisao' => 'Em Revisão',
            'aprovado' => 'Aprovado',
            'publicado' => 'Publicado',
        ];

        return $estados[$this->estado] ?? $this->estado;
    }
}
