<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Carbon\Carbon;

class FormulaCaudal extends Model
{
    use HasFactory;

    protected $table = 'formulas_caudal';

    protected $fillable = [
        'estacao_id',
        'nome',
        'descricao',
        'coeficiente_a',
        'coeficiente_b',
        'coeficiente_c',
        'data_inicio',
        'data_fim',
        'ativa',
        'nivel_min',
        'nivel_max',
        'criado_por',
        'motivo_mudanca',
        'r_quadrado',
    ];

    protected $casts = [
        'data_inicio' => 'date',
        'data_fim' => 'date',
        'ativa' => 'boolean',
        'coeficiente_a' => 'decimal:6',
        'coeficiente_b' => 'decimal:6',
        'coeficiente_c' => 'decimal:6',
        'nivel_min' => 'decimal:3',
        'nivel_max' => 'decimal:3',
        'r_quadrado' => 'decimal:4',
    ];

    public function estacao(): BelongsTo
    {
        return $this->belongsTo(Estacao::class);
    }

    public function criador(): BelongsTo
    {
        return $this->belongsTo(User::class, 'criado_por');
    }

    /**
     * Calcular caudal usando a fórmula A(H-C)^B
     */
    public function calcularCaudal($nivel)
    {
        if (!$this->isValidaParaNivel($nivel)) {
            return null;
        }

        // Fórmula: Q = A × (H - C)^B
        // Onde: Q = Caudal, H = Nível, A,B,C = Coeficientes
        $h_menos_c = $nivel - $this->coeficiente_c;

        // Evitar valores negativos ou zero na base da potência
        if ($h_menos_c <= 0) {
            return 0;
        }

        $caudal = $this->coeficiente_a * pow($h_menos_c, $this->coeficiente_b);

        return max(0, $caudal); // Caudal não pode ser negativo
    }

    /**
     * Verificar se a fórmula é válida para o nível informado
     */
    public function isValidaParaNivel($nivel)
    {
        // Verificar se está ativa
        if (!$this->ativa) {
            return false;
        }

        // Verificar validade temporal
        if (!$this->isValidaParaData()) {
            return false;
        }

        // Verificar faixa de níveis
        if ($this->nivel_min && $nivel < $this->nivel_min) {
            return false;
        }

        if ($this->nivel_max && $nivel > $this->nivel_max) {
            return false;
        }

        return true;
    }

    /**
     * Verificar se a fórmula é válida para a data atual
     */
    public function isValidaParaData($data = null)
    {
        $data = $data ?? today();

        // Verificar se já começou
        if ($this->data_inicio && $data < $this->data_inicio) {
            return false;
        }

        // Verificar se não expirou
        if ($this->data_fim && $data > $this->data_fim) {
            return false;
        }

        return true;
    }

    /**
     * Obter representação textual da fórmula para usuários
     */
    public function getFormulaTexto()
    {
        $a = number_format($this->coeficiente_a, 3);
        $b = number_format($this->coeficiente_b, 3);
        $c = number_format($this->coeficiente_c, 3);

        if ($this->coeficiente_c == 0) {
            return "Q = {$a} × H^{$b}";
        } else {
            return "Q = {$a} × (H - {$c})^{$b}";
        }
    }

    /**
     * Obter status da fórmula em linguagem simples
     */
    public function getStatus()
    {
        if (!$this->ativa) {
            return [
                'status' => 'inativa',
                'texto' => 'Inativa',
                'cor' => '#6b7280' // Cinza
            ];
        }

        if (!$this->isValidaParaData()) {
            $hoje = today();

            if ($this->data_inicio > $hoje) {
                return [
                    'status' => 'futura',
                    'texto' => 'Entra em vigor em ' . $this->data_inicio->format('d/m/Y'),
                    'cor' => '#3b82f6' // Azul
                ];
            }

            if ($this->data_fim && $this->data_fim < $hoje) {
                return [
                    'status' => 'expirada',
                    'texto' => 'Expirou em ' . $this->data_fim->format('d/m/Y'),
                    'cor' => '#ef4444' // Vermelho
                ];
            }
        }

        return [
            'status' => 'ativa',
            'texto' => 'Ativa',
            'cor' => '#22c55e' // Verde
        ];
    }

    /**
     * Obter qualidade do ajuste baseado no R²
     */
    public function getQualidadeAjuste()
    {
        if (!$this->r_quadrado) {
            return [
                'qualidade' => 'desconhecida',
                'texto' => 'Não informado',
                'cor' => '#9ca3af'
            ];
        }

        $r2 = (float) $this->r_quadrado;

        if ($r2 >= 0.9) {
            return [
                'qualidade' => 'excelente',
                'texto' => 'Excelente (R² = ' . number_format($r2, 3) . ')',
                'cor' => '#22c55e'
            ];
        } elseif ($r2 >= 0.8) {
            return [
                'qualidade' => 'boa',
                'texto' => 'Boa (R² = ' . number_format($r2, 3) . ')',
                'cor' => '#3b82f6'
            ];
        } elseif ($r2 >= 0.6) {
            return [
                'qualidade' => 'regular',
                'texto' => 'Regular (R² = ' . number_format($r2, 3) . ')',
                'cor' => '#eab308'
            ];
        } else {
            return [
                'qualidade' => 'ruim',
                'texto' => 'Ruim (R² = ' . number_format($r2, 3) . ')',
                'cor' => '#ef4444'
            ];
        }
    }

    /**
     * Scope para buscar fórmulas ativas
     */
    public function scopeAtivas($query)
    {
        return $query->where('ativa', true);
    }

    /**
     * Scope para buscar fórmulas válidas para uma data
     */
    public function scopeValidasPara($query, $data = null)
    {
        $data = $data ?? today();

        return $query->where('ativa', true)
                    ->where(function ($q) use ($data) {
                        $q->whereNull('data_inicio')
                          ->orWhere('data_inicio', '<=', $data);
                    })
                    ->where(function ($q) use ($data) {
                        $q->whereNull('data_fim')
                          ->orWhere('data_fim', '>=', $data);
                    });
    }
}