<?php
// app/Models/LeituraBarragem.php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class LeituraBarragem extends Model
{
    use HasFactory;

    protected $table = 'leituras_barragens';

    protected $fillable = [
        'barragem_id',
        'data_leitura',
        'hora_leitura',
        'cota_actual',
        'volume_actual',
        'volume_morto',
        'percentagem_enchimento',
        'percentagem_total',
        'volume_armazenado',
        'caudal_captacao',
        'caudal_afluente',
        'caudal_efluente',
        'descarga_fundo',
        'descarga_superficie',
        'caudal_ecologico',
        'caudal_turbinado',
        'caudal_descarregador',
        'evaporacao',
        'infiltracao',
        'variacao_volume',
        'comparacao_ano_anterior',
        'descarga_total',
        'precipitacao',
        'cota_npa',
        'nivel_minimo',
        'nivel_maximo',
        'data_nivel_minimo',
        'data_nivel_maximo',
        'operador_id',
        'metodo_leitura',
        'observacoes',
        'validado',
        'validado_por',
        'data_validacao',
    ];

    protected $casts = [
        'data_leitura' => 'date',
        'hora_leitura' => 'datetime',
        'cota_actual' => 'decimal:2',
        'volume_actual' => 'decimal:2',
        'percentagem_enchimento' => 'decimal:2',
        'caudal_afluente' => 'decimal:3',
        'caudal_efluente' => 'decimal:3',
        'descarga_fundo' => 'decimal:3',
        'descarga_superficie' => 'decimal:3',
        'caudal_ecologico' => 'decimal:3',
        'caudal_turbinado' => 'decimal:3',
        'caudal_descarregador' => 'decimal:3',
        'evaporacao' => 'decimal:3',
        'infiltracao' => 'decimal:3',
        'variacao_volume' => 'decimal:2',
        'comparacao_ano_anterior' => 'decimal:2',
        'descarga_total' => 'decimal:3',
        'precipitacao' => 'decimal:2',
        'validado' => 'boolean',
        'data_validacao' => 'datetime',
    ];

    public function barragem(): BelongsTo
    {
        return $this->belongsTo(Barragem::class);
    }

    public function operador(): BelongsTo
    {
        return $this->belongsTo(User::class, 'operador_id');
    }

    public function validador(): BelongsTo
    {
        return $this->belongsTo(User::class, 'validado_por');
    }

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($leitura) {
            $leitura->calcularValoresAutomaticos();
        });

        static::updating(function ($leitura) {
            $leitura->calcularValoresAutomaticos();
        });
    }

    public function calcularValoresAutomaticos()
    {
        if ($this->barragem && $this->cota_actual) {
            // Calcular volume atual usando método da barragem
            $this->volume_actual = $this->barragem->calcularVolumeAtual($this->cota_actual);

            // FÓRMULA DO EXCEL: =(M4/3.9)*100 onde M4=volume_total, 3.9=volume_maximo
            // Assumindo que os volumes estão em m³
            if ($this->barragem->volume_maximo && $this->volume_actual) {
                $this->percentagem_total = ($this->volume_actual / $this->barragem->volume_maximo) * 100;

                // Percentagem de enchimento (excluindo volume morto)
                if ($this->barragem->volume_morto) {
                    $volume_util = $this->volume_actual - $this->barragem->volume_morto;
                    $volume_maximo_util = $this->barragem->volume_maximo - $this->barragem->volume_morto;
                    if ($volume_maximo_util > 0) {
                        $this->percentagem_enchimento = max(0, ($volume_util / $volume_maximo_util) * 100);
                    } else {
                        $this->percentagem_enchimento = 0;
                    }
                } else {
                    $this->percentagem_enchimento = $this->percentagem_total;
                }

                // Limitar percentagens entre 0 e 100
                $this->percentagem_total = max(0, min(100, $this->percentagem_total));
                $this->percentagem_enchimento = max(0, min(100, $this->percentagem_enchimento));
            }

            // Calcular variação de volume primeiro (necessário para descarga)
            $this->calcularVariacaoVolume();

            // Calcular descarga total baseado na fórmula do Excel
            $this->calcularDescargaTotal();

            // Calcular comparação com ano anterior
            $this->calcularComparacaoAnoAnterior();
        }
    }

    private function calcularDescargaTotal()
    {
        // BASEADO NO EXCEL ORIGINAL - Fórmula R4: =N4/3.25
        // Onde N4 = variação de volume
        if ($this->variacao_volume !== null) {
            $this->descarga_total = $this->variacao_volume / 3.25;
        } else {
            // Fallback: soma dos componentes se existirem
            $this->descarga_total =
                ($this->caudal_efluente ?? 0) +
                ($this->descarga_fundo ?? 0) +
                ($this->descarga_superficie ?? 0) +
                ($this->caudal_ecologico ?? 0) +
                ($this->caudal_turbinado ?? 0) +
                ($this->caudal_descarregador ?? 0);
        }
    }

    private function calcularVariacaoVolume()
    {
        $leitura_anterior = static::where('barragem_id', $this->barragem_id)
            ->where('data_leitura', '<', $this->data_leitura)
            ->orderBy('data_leitura', 'desc')
            ->first();

        if ($leitura_anterior && $this->volume_actual && $leitura_anterior->volume_actual) {
            $this->variacao_volume = $this->volume_actual - $leitura_anterior->volume_actual;
        }
    }

    private function calcularComparacaoAnoAnterior()
    {
        $data_ano_anterior = $this->data_leitura->copy()->subYear();
        
        $leitura_ano_anterior = static::where('barragem_id', $this->barragem_id)
            ->whereDate('data_leitura', $data_ano_anterior)
            ->first();

        if ($leitura_ano_anterior && $this->percentagem_enchimento && $leitura_ano_anterior->percentagem_enchimento) {
            $this->comparacao_ano_anterior = $this->percentagem_enchimento - $leitura_ano_anterior->percentagem_enchimento;
        }
    }

    public function getStatusValidacao()
    {
        if ($this->validado) {
            return 'validado';
        }
        
        if ($this->created_at->diffInHours(now()) > 24) {
            return 'pendente';
        }
        
        return 'recente';
    }
}