<?php

namespace App\Models;

use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;
use Spatie\Permission\Traits\HasRoles;

class User extends Authenticatable
{
    use HasApiTokens, HasFactory, Notifiable, HasRoles;

    protected $fillable = [
        'name',
        'email',
        'password',
        'telefone',
        'cargo',
        'departamento',
        'provincia',
        'ativo',
        'notificacao_sms',
        'notificacao_email',
        'notificacao_emergencia',
        'niveis_notificacao',
        'nivel_escalacao',
        'receber_relatorio_diario',
        'horario_relatorio',
    ];

    protected $hidden = [
        'password',
        'remember_token',
    ];

    protected $casts = [
        'email_verified_at' => 'datetime',
        'password' => 'hashed',
        'ativo' => 'boolean',
        'notificacao_sms' => 'boolean',
        'notificacao_email' => 'boolean',
        'notificacao_emergencia' => 'boolean',
        'niveis_notificacao' => 'array',
        'receber_relatorio_diario' => 'boolean',
        'horario_relatorio' => 'datetime:H:i',
    ];

    public function leituras_barragens()
    {
        return $this->hasMany(LeituraBarragem::class, 'operador_id');
    }

    public function leituras_estacoes()
    {
        return $this->hasMany(LeituraEstacao::class, 'operador_id');
    }

    public function boletins_elaborados()
    {
        return $this->hasMany(Boletim::class, 'elaborado_por');
    }

    public function boletins_aprovados()
    {
        return $this->hasMany(Boletim::class, 'aprovado_por');
    }

    public function validacoes_barragens()
    {
        return $this->hasMany(LeituraBarragem::class, 'validado_por');
    }

    public function notificacoesEnviadas()
    {
        return $this->hasMany(NotificacaoEnviada::class);
    }

    /**
     * Verificar se deve receber notificação de um nível específico
     */
    public function deveReceberNotificacao($nivel)
    {
        if ($nivel === 'emergencia' && $this->notificacao_emergencia) {
            return true;
        }

        if ($this->niveis_notificacao) {
            return in_array($nivel, $this->niveis_notificacao);
        }

        // Default: alertas e emergências
        return in_array($nivel, ['alerta', 'emergencia']);
    }

    /**
     * Obter telefone formatado
     */
    public function getTelefoneFormatado()
    {
        if (!$this->telefone) {
            return null;
        }

        $telefone = preg_replace('/[^0-9+]/', '', $this->telefone);
        
        if (strpos($telefone, '+258') === 0) {
            return $telefone;
        }
        
        if (strlen($telefone) === 9 && $telefone[0] === '8') {
            return '+258' . $telefone;
        }
        
        return $this->telefone;
    }

    /**
     * Verificar se telefone é válido
     */
    public function temTelefoneValido()
    {
        $telefone = $this->getTelefoneFormatado();
        return $telefone && (strpos($telefone, '+258') === 0) && strlen($telefone) === 12;
    }

    /**
     * Obter nível de escalação formatado
     */
    public function getNivelEscalacaoFormatado()
    {
        $niveis = [
            'basico' => 'Básico',
            'supervisor' => 'Supervisor',
            'director' => 'Director'
        ];

        return $niveis[$this->nivel_escalacao] ?? 'Básico';
    }

    /**
     * Verificar se é supervisor ou director
     */
    public function isSupervisor()
    {
        return in_array($this->nivel_escalacao, ['supervisor', 'director']);
    }

    /**
     * Obter estatísticas de notificações do usuário
     */
    public function getEstatisticasNotificacoes($dias = 30)
    {
        return [
            'total' => $this->notificacoesEnviadas()
                ->where('created_at', '>=', now()->subDays($dias))
                ->count(),
            'sms' => $this->notificacoesEnviadas()
                ->where('tipo', 'sms')
                ->where('created_at', '>=', now()->subDays($dias))
                ->count(),
            'email' => $this->notificacoesEnviadas()
                ->where('tipo', 'email')
                ->where('created_at', '>=', now()->subDays($dias))
                ->count(),
            'entregues' => $this->notificacoesEnviadas()
                ->where('status', 'entregue')
                ->where('created_at', '>=', now()->subDays($dias))
                ->count(),
            'falhadas' => $this->notificacoesEnviadas()
                ->where('status', 'falhado')
                ->where('created_at', '>=', now()->subDays($dias))
                ->count(),
        ];
    }
}