<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Cache;
use Carbon\Carbon;

class ExternalDataService
{
    private $openWeatherApiKey;
    private $nasaApiKey;

    public function __construct()
    {
        $this->openWeatherApiKey = env('OPENWEATHER_API_KEY');
        $this->nasaApiKey = env('NASA_API_KEY');
    }

    /**
     * Obter dados meteorológicos do OpenWeatherMap para Moçambique
     */
    public function getWeatherData($latitude, $longitude)
    {
        $cacheKey = "weather_data_{$latitude}_{$longitude}";

        return Cache::remember($cacheKey, 3600, function () use ($latitude, $longitude) {
            $response = Http::get("https://api.openweathermap.org/data/2.5/weather", [
                'lat' => $latitude,
                'lon' => $longitude,
                'appid' => $this->openWeatherApiKey,
                'units' => 'metric',
                'lang' => 'pt'
            ]);

            if ($response->successful()) {
                $data = $response->json();
                return [
                    'temperatura' => $data['main']['temp'],
                    'humidade' => $data['main']['humidity'],
                    'precipitacao' => $data['rain']['1h'] ?? 0,
                    'pressao' => $data['main']['pressure'],
                    'vento_velocidade' => $data['wind']['speed'],
                    'vento_direcao' => $data['wind']['deg'] ?? 0,
                    'descricao' => $data['weather'][0]['description'],
                    'timestamp' => now()
                ];
            }

            return null;
        });
    }

    /**
     * Obter dados de precipitação CHIRPS para uma região
     */
    public function getCHIRPSData($latitude, $longitude, $startDate, $endDate)
    {
        // URL do serviço CHIRPS (exemplo)
        $url = "https://climateserv.servirglobal.net/chirps/getDataFromEnsemble/";

        $response = Http::post($url, [
            'datatype' => 'CHIRPS',
            'begintime' => $startDate,
            'endtime' => $endDate,
            'intervaltype' => 0, // Mensal
            'geometry' => [
                'type' => 'Point',
                'coordinates' => [$longitude, $latitude]
            ]
        ]);

        if ($response->successful()) {
            return $response->json();
        }

        return null;
    }

    /**
     * Obter dados NASA GPM para precipitação
     */
    public function getNASAGPMData($latitude, $longitude, $date)
    {
        $cacheKey = "nasa_gpm_{$latitude}_{$longitude}_{$date}";

        return Cache::remember($cacheKey, 86400, function () use ($latitude, $longitude, $date) {
            // NASA GPM IMERG API endpoint
            $url = "https://gpm1.gesdisc.eosdis.nasa.gov/opendap/GPM_L3/GPM_3IMERGM.06/";

            // Exemplo de requisição (necessita autenticação NASA Earthdata)
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->nasaApiKey
            ])->get($url, [
                'lat' => $latitude,
                'lon' => $longitude,
                'time' => $date
            ]);

            if ($response->successful()) {
                return $response->json();
            }

            return null;
        });
    }

    /**
     * Obter dados simulados de nível de rio baseados em precipitação
     */
    public function getSimulatedRiverLevel($baciaId, $precipitacao)
    {
        // Modelo simplificado: nível do rio baseado na precipitação
        $nivelBase = 2.0; // metros
        $fatorResposta = 0.1; // m por mm de chuva

        $nivelCalculado = $nivelBase + ($precipitacao * $fatorResposta);

        return [
            'nivel_atual' => round($nivelCalculado, 2),
            'nivel_base' => $nivelBase,
            'precipitacao_influencia' => $precipitacao,
            'bacia_id' => $baciaId,
            'timestamp' => now(),
            'fonte' => 'simulacao_precipitacao'
        ];
    }

    /**
     * Consolidar dados de múltiplas fontes para uma estação
     */
    public function getConsolidatedData($estacaoId, $latitude, $longitude)
    {
        $weatherData = $this->getWeatherData($latitude, $longitude);
        $precipitacao = $weatherData['precipitacao'] ?? 0;

        return [
            'estacao_id' => $estacaoId,
            'meteorologia' => $weatherData,
            'nivel_rio_simulado' => $this->getSimulatedRiverLevel($estacaoId, $precipitacao),
            'fonte_dados' => 'apis_externas',
            'ultima_atualizacao' => now()
        ];
    }

    /**
     * Atualizar dados automáticamente para todas as estações do norte
     */
    public function updateAllNorthStations()
    {
        $estacoes = [
            ['id' => 1, 'nome' => 'Pemba', 'lat' => -12.97, 'lon' => 40.51],
            ['id' => 2, 'nome' => 'Nampula', 'lat' => -15.12, 'lon' => 39.27],
            ['id' => 3, 'nome' => 'Lichinga', 'lat' => -13.27, 'lon' => 36.64],
            ['id' => 4, 'nome' => 'Nacala', 'lat' => -14.54, 'lon' => 40.67],
        ];

        $results = [];
        foreach ($estacoes as $estacao) {
            $results[] = $this->getConsolidatedData(
                $estacao['id'],
                $estacao['lat'],
                $estacao['lon']
            );
        }

        return $results;
    }
}