<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // Tabela de coletas
        Schema::create('coletas_qa', function (Blueprint $table) {
            $table->id();
            $table->foreignId('estacao_id')->constrained('estacoes')->onDelete('cascade');
            $table->date('data_coleta');
            $table->time('hora_coleta')->nullable();
            $table->string('responsavel', 100)->nullable();
            $table->text('observacoes')->nullable();
            $table->boolean('validado')->default(false);
            $table->foreignId('validado_por')->nullable()->constrained('users');
            $table->timestamp('data_validacao')->nullable();
            $table->timestamps();

            $table->index(['estacao_id', 'data_coleta']);
        });

        // Tabela de parâmetros físico-químicos
        Schema::create('parametros_fisico_quimicos', function (Blueprint $table) {
            $table->id();
            $table->foreignId('coleta_id')->constrained('coletas_qa')->onDelete('cascade');

            // Parâmetros físico-químicos básicos
            $table->decimal('temperatura', 5, 2)->nullable()->comment('°C');
            $table->decimal('ph', 4, 2)->nullable();
            $table->integer('condutividade_eletrica')->nullable()->comment('µS/cm');
            $table->integer('tds')->nullable()->comment('mg/L - Sólidos Dissolvidos Totais');
            $table->decimal('oxigenio_dissolvido', 5, 2)->nullable()->comment('% saturação');
            $table->decimal('turbidez', 6, 2)->nullable()->comment('NTU');

            // Nutrientes
            $table->decimal('nitrato', 8, 4)->nullable()->comment('mg/L');
            $table->decimal('nitrogenio_total', 8, 4)->nullable()->comment('mg/L');
            $table->decimal('fosfato', 8, 4)->nullable()->comment('mg/L');
            $table->decimal('fosfato_total', 8, 4)->nullable()->comment('mg/L');

            // Indicadores orgânicos
            $table->decimal('dbo', 6, 2)->nullable()->comment('mg/L - Demanda Bioquímica Oxigênio');
            $table->integer('coliformes_fecais')->nullable()->comment('NMP/100mL');

            // Sólidos
            $table->integer('solidos_totais')->nullable()->comment('mg/L');

            // Íons
            $table->decimal('cloretos', 8, 4)->nullable()->comment('mg/L');

            $table->timestamps();
        });

        // Tabela de metais pesados
        Schema::create('metais_pesados', function (Blueprint $table) {
            $table->id();
            $table->foreignId('coleta_id')->constrained('coletas_qa')->onDelete('cascade');

            $table->decimal('ferro', 8, 5)->nullable()->comment('mg/L');
            $table->decimal('chumbo', 8, 5)->nullable()->comment('mg/L - Pb');
            $table->decimal('cadmio', 8, 6)->nullable()->comment('mg/L - Cd');
            $table->decimal('zinco', 8, 5)->nullable()->comment('mg/L - Zn');
            $table->decimal('cobre', 8, 5)->nullable()->comment('mg/L - Cu');
            $table->decimal('arsenio', 8, 5)->nullable()->comment('mg/L - As');
            $table->decimal('mercurio', 8, 5)->nullable()->comment('mg/L - Hg');

            $table->timestamps();
        });

        // Tabela de limites de parâmetros
        Schema::create('limites_parametros', function (Blueprint $table) {
            $table->id();
            $table->string('parametro', 50)->unique();
            $table->decimal('limite_minimo', 10, 6)->nullable();
            $table->decimal('limite_maximo', 10, 6)->nullable();
            $table->string('unidade', 20)->nullable();
            $table->string('norma_referencia', 100)->nullable();
            $table->timestamps();
        });

        // Tabela de IQA calculado
        Schema::create('iqa_calculado', function (Blueprint $table) {
            $table->id();
            $table->foreignId('coleta_id')->constrained('coletas_qa')->onDelete('cascade');
            $table->decimal('iqa_valor', 5, 2)->nullable();
            $table->enum('iqa_classificacao', ['Excelente', 'Boa', 'Moderada', 'Má', 'Muito Má'])->nullable();
            $table->string('metodo_calculo', 50)->default('NSF-WQI');
            $table->json('detalhes_calculo')->nullable()->comment('qi de cada parâmetro');
            $table->timestamps();

            $table->index('iqa_classificacao');
        });

        // Inserir limites padrão
        DB::table('limites_parametros')->insert([
            ['parametro' => 'temperatura', 'limite_minimo' => 25, 'limite_maximo' => 40, 'unidade' => '°C', 'norma_referencia' => 'Padrão Moçambique'],
            ['parametro' => 'ph', 'limite_minimo' => 6.5, 'limite_maximo' => 8.5, 'unidade' => null, 'norma_referencia' => 'Padrão Moçambique'],
            ['parametro' => 'condutividade_eletrica', 'limite_minimo' => 50, 'limite_maximo' => 2000, 'unidade' => 'µS/cm', 'norma_referencia' => 'Padrão Moçambique'],
            ['parametro' => 'tds', 'limite_minimo' => null, 'limite_maximo' => 1000, 'unidade' => 'mg/L', 'norma_referencia' => 'Padrão Moçambique'],
            ['parametro' => 'nitrato', 'limite_minimo' => null, 'limite_maximo' => 50, 'unidade' => 'mg/L', 'norma_referencia' => 'Padrão Moçambique'],
            ['parametro' => 'cloretos', 'limite_minimo' => null, 'limite_maximo' => 250, 'unidade' => 'mg/L', 'norma_referencia' => 'Padrão Moçambique'],
            ['parametro' => 'fosfato', 'limite_minimo' => null, 'limite_maximo' => 0.1, 'unidade' => 'mg/L', 'norma_referencia' => 'Padrão Moçambique'],
            ['parametro' => 'ferro', 'limite_minimo' => null, 'limite_maximo' => 0.3, 'unidade' => 'mg/L', 'norma_referencia' => 'Padrão Moçambique'],
            ['parametro' => 'chumbo', 'limite_minimo' => null, 'limite_maximo' => 0.01, 'unidade' => 'mg/L', 'norma_referencia' => 'Padrão Moçambique'],
            ['parametro' => 'cadmio', 'limite_minimo' => null, 'limite_maximo' => 0.003, 'unidade' => 'mg/L', 'norma_referencia' => 'Padrão Moçambique'],
            ['parametro' => 'zinco', 'limite_minimo' => null, 'limite_maximo' => 3, 'unidade' => 'mg/L', 'norma_referencia' => 'Padrão Moçambique'],
            ['parametro' => 'cobre', 'limite_minimo' => null, 'limite_maximo' => 1, 'unidade' => 'mg/L', 'norma_referencia' => 'Padrão Moçambique'],
            ['parametro' => 'arsenio', 'limite_minimo' => null, 'limite_maximo' => 0.01, 'unidade' => 'mg/L', 'norma_referencia' => 'Padrão Moçambique'],
            ['parametro' => 'mercurio', 'limite_minimo' => null, 'limite_maximo' => 0.001, 'unidade' => 'mg/L', 'norma_referencia' => 'Padrão Moçambique'],
        ]);
    }

    public function down(): void
    {
        Schema::dropIfExists('iqa_calculado');
        Schema::dropIfExists('limites_parametros');
        Schema::dropIfExists('metais_pesados');
        Schema::dropIfExists('parametros_fisico_quimicos');
        Schema::dropIfExists('coletas_qa');
    }
};
