#!/usr/bin/env python3
"""
Extrai e documenta as fórmulas específicas de cálculo das barragens
"""

import pandas as pd
import sys
import os
from openpyxl import load_workbook
import re
import warnings
warnings.filterwarnings('ignore')

def extract_percentage_formulas():
    """Extrai fórmulas de percentagem específicas"""
    file_path = "/Users/salumsaidsalum/Desktop/script/ficheiros/Cópia de BARRAGENS ACT 8.8.xlsm"

    print("FÓRMULAS DE CÁLCULO DAS PERCENTAGENS DE ENCHIMENTO")
    print("="*80)

    try:
        # Carrega sem data_only para ver as fórmulas
        wb = load_workbook(file_path, data_only=False)
        wb_data = load_workbook(file_path, data_only=True)

        barragens = ['NAMPULA', 'NACALA', 'MUGICA', 'CHIPEMBE', 'LOCUMUE', 'MITUCUE']

        for sheet_name in barragens:
            if sheet_name in wb.sheetnames:
                print(f"\n{'='*60}")
                print(f"BARRAGEM: {sheet_name}")
                print(f"{'='*60}")

                ws_formula = wb[sheet_name]
                ws_data = wb_data[sheet_name]

                # Analisa os cabeçalhos para entender a estrutura
                print("ESTRUTURA DOS CABEÇALHOS:")
                for row in range(1, 4):
                    headers = []
                    for col in range(1, 31):
                        cell = ws_data.cell(row=row, column=col)
                        if cell.value:
                            headers.append(f"Col{col}:{str(cell.value)[:20]}")
                    if headers:
                        print(f"  Linha {row}: {' | '.join(headers[:10])}")

                # Procura especificamente por fórmulas que calculam percentagens
                percentage_formulas = []
                volume_formulas = []

                for row in range(1, min(100, ws_formula.max_row + 1)):
                    for col in range(1, min(100, ws_formula.max_column + 1)):
                        cell = ws_formula.cell(row=row, column=col)
                        if cell.data_type == 'f' and cell.value:
                            formula = str(cell.value)

                            # Identifica fórmulas de percentagem
                            if '*100' in formula or '/100' in formula:
                                # Obtém o valor calculado
                                data_cell = ws_data.cell(row=row, column=col)
                                percentage_formulas.append({
                                    'cell': cell.coordinate,
                                    'formula': formula,
                                    'value': data_cell.value,
                                    'row': row,
                                    'col': col
                                })

                            # Identifica outras fórmulas importantes (volumes, cotas, etc.)
                            elif any(op in formula for op in ['+', '-', '*', '/', '^']):
                                data_cell = ws_data.cell(row=row, column=col)
                                volume_formulas.append({
                                    'cell': cell.coordinate,
                                    'formula': formula,
                                    'value': data_cell.value,
                                    'row': row,
                                    'col': col
                                })

                # Mostra as fórmulas de percentagem
                if percentage_formulas:
                    print(f"\nFÓRMULAS DE PERCENTAGEM ENCONTRADAS ({len(percentage_formulas)}):")
                    for formula_info in percentage_formulas:
                        print(f"  {formula_info['cell']}: {formula_info['formula']} = {formula_info['value']}")

                        # Tenta identificar o contexto da fórmula
                        context = get_formula_context(ws_data, formula_info['row'], formula_info['col'])
                        if context:
                            print(f"    Contexto: {context}")

                # Mostra outras fórmulas relevantes (primeiras 10)
                if volume_formulas:
                    print(f"\nOUTRAS FÓRMULAS IMPORTANTES ({len(volume_formulas)}):")
                    for formula_info in volume_formulas[:10]:
                        print(f"  {formula_info['cell']}: {formula_info['formula']} = {formula_info['value']}")

                # Identifica padrões específicos
                analyze_formula_patterns(percentage_formulas, sheet_name)

    except Exception as e:
        print(f"Erro: {str(e)}")
        import traceback
        traceback.print_exc()

def get_formula_context(ws, row, col):
    """Obtém o contexto de uma fórmula analisando células próximas"""
    context_parts = []

    # Verifica cabeçalhos da coluna
    for header_row in range(1, 4):
        if header_row < row:
            header_cell = ws.cell(row=header_row, column=col)
            if header_cell.value:
                context_parts.append(str(header_cell.value))

    # Verifica células à esquerda na mesma linha
    for left_col in range(max(1, col-3), col):
        left_cell = ws.cell(row=row, column=left_col)
        if left_cell.value:
            context_parts.append(str(left_cell.value))

    return " | ".join(context_parts) if context_parts else None

def analyze_formula_patterns(formulas, sheet_name):
    """Analisa padrões nas fórmulas"""
    if not formulas:
        return

    print(f"\nANÁLISE DOS PADRÕES DAS FÓRMULAS:")

    # Agrupa fórmulas por padrão
    patterns = {}
    for formula_info in formulas:
        formula = formula_info['formula']

        # Remove referências específicas de células para encontrar padrões
        pattern = re.sub(r'[A-Z]+\d+', 'CELL', formula)
        if pattern not in patterns:
            patterns[pattern] = []
        patterns[pattern].append(formula_info)

    for pattern, formula_list in patterns.items():
        print(f"  Padrão: {pattern}")
        print(f"    Ocorrências: {len(formula_list)}")
        print(f"    Exemplo: {formula_list[0]['formula']} (célula {formula_list[0]['cell']})")

def analyze_constants():
    """Analisa constantes utilizadas nas fórmulas"""
    file_path = "/Users/salumsaidsalum/Desktop/script/ficheiros/Cópia de BARRAGENS ACT 8.8.xlsm"

    print(f"\n{'='*80}")
    print("CONSTANTES E PARÂMETROS DAS BARRAGENS")
    print(f"{'='*80}")

    try:
        wb = load_workbook(file_path, data_only=True)

        barragem_constants = {}

        for sheet_name in ['NAMPULA', 'NACALA']:  # Analisa 2 para exemplo
            if sheet_name in wb.sheetnames:
                ws = wb[sheet_name]

                print(f"\nCONSTANTES DA BARRAGEM {sheet_name}:")

                # Procura por valores constantes nas primeiras linhas (geralmente parâmetros)
                constants = []
                for row in range(1, 10):
                    for col in range(1, 20):
                        cell = ws.cell(row=row, column=col)
                        if cell.value and isinstance(cell.value, (int, float)):
                            if cell.value > 100:  # Valores grandes que podem ser capacidades
                                constants.append({
                                    'cell': cell.coordinate,
                                    'value': cell.value,
                                    'context': get_formula_context(ws, row, col)
                                })

                # Filtra e mostra constantes relevantes
                relevant_constants = []
                for const in constants:
                    if const['context'] and any(keyword in const['context'].lower()
                                                for keyword in ['npa', 'nme', 'morto', 'total', 'volume']):
                        relevant_constants.append(const)

                for const in relevant_constants:
                    print(f"  {const['cell']}: {const['value']} ({const['context']})")

                barragem_constants[sheet_name] = relevant_constants

        return barragem_constants

    except Exception as e:
        print(f"Erro ao analisar constantes: {str(e)}")
        return {}

if __name__ == "__main__":
    extract_percentage_formulas()
    analyze_constants()