import openpyxl
from openpyxl.utils import get_column_letter

def final_analysis():
    """Análise final focada no fluxo de trabalho"""
    wb = openpyxl.load_workbook('ficheiros/Cópia de BARRAGENS ACT 8.8.xlsm', data_only=False)

    print('🎯 ANÁLISE FINAL: FLUXO COMPLETO DO SISTEMA DE BARRAGENS')
    print('=' * 70)
    
    # 1. Examinar uma folha de dados em detalhe
    ws_nampula = wb['NAMPULA']
    print('1. ESTRUTURA DE ENTRADA DE DADOS (Exemplo: NAMPULA)')
    print('-' * 50)
    print(f'   Dimensões: {ws_nampula.max_row} linhas x {ws_nampula.max_column} colunas')
    
    # Mostrar estrutura de cabeçalhos
    print('   Cabeçalhos identificados:')
    for col in range(1, 15):
        cell = ws_nampula.cell(row=2, column=col)
        if cell.value and str(cell.value).strip():
            print(f'     Coluna {get_column_letter(col)}: {cell.value}')
    
    # Mostrar exemplo de dados recentes
    print('\n   Exemplo de dados recentes (últimas entradas):')
    start_row = max(1, ws_nampula.max_row - 3)
    for row in range(start_row, ws_nampula.max_row + 1):
        row_data = []
        for col in range(1, 8):  # Primeiras 7 colunas
            cell = ws_nampula.cell(row=row, column=col)
            if cell.value:
                val_str = str(cell.value)
                if val_str.startswith('='):
                    row_data.append('FÓRMULA')
                else:
                    row_data.append(val_str[:15])
            else:
                row_data.append('')
        print(f'     Linha {row}: {" | ".join([x for x in row_data if x])}')
    
    # 2. Examinar folha de gráficos
    ws_graficos = wb['GRAFICOS DE BARRAGEM']
    print('\n2. ESTRUTURA DOS GRÁFICOS')
    print('-' * 30)
    print(f'   Total de gráficos: {len(ws_graficos._charts)}')
    
    chart_types = {}
    for chart in ws_graficos._charts:
        chart_type = type(chart).__name__
        if chart_type in chart_types:
            chart_types[chart_type] += 1
        else:
            chart_types[chart_type] = 1
    
    for chart_type, count in chart_types.items():
        print(f'   {chart_type}: {count} gráfico(s)')
    
    # 3. Identificar o fluxo de trabalho
    print('\n3. FLUXO DE TRABALHO IDENTIFICADO')
    print('-' * 35)
    print('   📝 INSERÇÃO DE DADOS:')
    print('      • Usuário abre a folha da barragem específica')
    print('      • Localiza a última linha com dados')
    print('      • Insere valores manualmente nas colunas:')
    print('        - Coluna A: Data é gerada automaticamente (fórmula)')
    print('        - Coluna C: NPA (Nível de Pleno Armazenamento)')
    print('        - Coluna D: NME (Nível Máximo de Exploração)')
    print('        - Coluna E: V.Morto (Volume Morto)')
    print('        - Coluna F: Nível atual')
    print('        - Outras colunas: Precipitação, Cota, etc.')
    
    print('\n   📊 ATUALIZAÇÃO AUTOMÁTICA:')
    print('      • Gráficos na folha "GRAFICOS DE BARRAGEM" atualizam automaticamente')
    print('      • Não há macros VBA, mas gráficos referenciam ranges dinâmicos')
    print('      • Sistema baseado em fórmulas e referências de células')
    
    print('\n   🎨 VISUALIZAÇÃO:')
    print('      • 5 gráficos de barras para comparações')
    print('      • 2 gráficos de linha para tendências temporais')
    print('      • Gráficos mostram evolução dos níveis das barragens')
    
    # 4. Experiência do usuário
    print('\n4. EXPERIÊNCIA DO USUÁRIO')
    print('-' * 25)
    print('   🖱️  O QUE O USUÁRIO VÊ:')
    print('      • Folhas separadas para cada barragem')
    print('      • Dados organizados cronologicamente')
    print('      • Folha dedicada para visualizar gráficos')
    
    print('\n   ⌨️  ONDE O USUÁRIO CLICA/DIGITA:')
    print('      • Navega para folha da barragem desejada')
    print('      • Clica na próxima linha vazia após os dados existentes')
    print('      • Digita valores nas colunas C, D, E, F, etc.')
    print('      • Data é preenchida automaticamente')
    
    print('\n   📈 COMO VÊ OS GRÁFICOS:')
    print('      • Navega para folha "GRAFICOS DE BARRAGEM"')
    print('      • Visualiza gráficos atualizados automaticamente')
    print('      • Pode comparar diferentes barragens e períodos')

if __name__ == "__main__":
    try:
        final_analysis()
    except Exception as e:
        print(f'❌ Erro: {e}')
        import traceback
        traceback.print_exc()
