import openpyxl
from openpyxl.utils import get_column_letter

def generate_comprehensive_report():
    """Gerar relatório completo do sistema de barragens"""
    wb = openpyxl.load_workbook('ficheiros/Cópia de BARRAGENS ACT 8.8.xlsm', data_only=False)

    print('🔍 RELATÓRIO COMPLETO: SISTEMA DE MONITORIZAÇÃO DE BARRAGENS')
    print('=' * 80)
    
    # Resumo das folhas
    dam_sheets = ['NAMPULA', 'NACALA', 'MUGICA', 'CHIPEMBE', 'LOCUMUE', 'MITUCUE']
    
    print('\n📋 1. RESUMO DO SISTEMA')
    print('-' * 30)
    print(f'   • Total de folhas no sistema: {len(wb.sheetnames)}')
    print(f'   • Barragens monitoradas: {len(dam_sheets)}')
    print(f'   • Folha de gráficos: "GRAFICOS DE BARRAGEM"')
    print(f'   • Sistema sem macros VBA (baseado em fórmulas)')
    
    # Análise de cada barragem
    print('\n🏗️ 2. ESTRUTURA DAS BARRAGENS MONITORADAS')
    print('-' * 45)
    
    total_records = 0
    for dam_name in dam_sheets:
        if dam_name in wb.sheetnames:
            ws = wb[dam_name]
            total_records += ws.max_row
            print(f'   📊 {dam_name}:')
            print(f'      - Registos: {ws.max_row:,} linhas')
            print(f'      - Colunas: {ws.max_column}')
            
            # Examinar última entrada de dados
            last_data_found = False
            for row in range(ws.max_row, max(1, ws.max_row - 50), -1):
                cell_a = ws.cell(row=row, column=1).value
                if cell_a and str(cell_a) not in ['', 'None']:
                    print(f'      - Última entrada: Linha {row}')
                    if hasattr(cell_a, 'year'):
                        print(f'      - Data mais recente: {cell_a.strftime("%Y-%m-%d") if hasattr(cell_a, "strftime") else cell_a}')
                    last_data_found = True
                    break
            
            if not last_data_found:
                print(f'      - Estado: Dados não identificados claramente')
    
    print(f'\n   📈 TOTAL DE REGISTOS NO SISTEMA: {total_records:,} linhas')
    
    # Sistema de gráficos
    ws_graficos = wb['GRAFICOS DE BARRAGEM']
    print('\n📊 3. SISTEMA DE VISUALIZAÇÃO')
    print('-' * 35)
    print(f'   • Folha: "GRAFICOS DE BARRAGEM"')
    print(f'   • Total de gráficos: {len(ws_graficos._charts)}')
    
    chart_summary = {}
    for chart in ws_graficos._charts:
        chart_type = type(chart).__name__
        chart_summary[chart_type] = chart_summary.get(chart_type, 0) + 1
    
    for chart_type, count in chart_summary.items():
        print(f'     - {chart_type}: {count} gráfico(s)')
    
    # Estrutura de dados
    print('\n📝 4. ESTRUTURA DE ENTRADA DE DADOS')
    print('-' * 38)
    print('   Cada folha de barragem contém:')
    
    # Examinar NAMPULA como exemplo
    ws_exemplo = wb['NAMPULA']
    print(f'   (Exemplo da folha {ws_exemplo.title}):')
    
    headers_found = {}
    for col in range(1, min(20, ws_exemplo.max_column + 1)):
        for row in range(1, 6):  # Primeiras 5 linhas
            cell = ws_exemplo.cell(row=row, column=col)
            if cell.value and isinstance(cell.value, str):
                val = cell.value.strip().upper()
                if any(keyword in val for keyword in ['NPA', 'NME', 'MORTO', 'DATA', 'NIVEL', 'COTA', 'PRECIPITAÇÃO']):
                    headers_found[get_column_letter(col)] = val
    
    for col, header in sorted(headers_found.items())[:8]:  # Primeiros 8
        print(f'     - Coluna {col}: {header}')
    
    return dam_sheets, total_records

def show_workflow_analysis():
    """Mostrar análise do fluxo de trabalho"""
    print('\n🔄 5. FLUXO DE TRABALHO COMPLETO')
    print('-' * 35)
    
    print('\n   📥 INSERÇÃO DE DADOS:')
    print('   ┌─────────────────────────────────────────┐')
    print('   │ 1. Usuário abre folha da barragem       │')
    print('   │ 2. Navega até última linha com dados    │')
    print('   │ 3. Insere nova linha de dados:          │')
    print('   │    • Coluna A: Data (fórmula =A+1)      │')
    print('   │    • Coluna C: NPA (valor manual)       │')
    print('   │    • Coluna D: NME (valor manual)       │')
    print('   │    • Coluna E: V.Morto (valor manual)   │')
    print('   │    • Coluna F: Nível (valor manual)     │')
    print('   │    • Outras colunas: Cota, Precipitação │')
    print('   └─────────────────────────────────────────┘')
    
    print('\n   📊 ATUALIZAÇÃO AUTOMÁTICA:')
    print('   ┌─────────────────────────────────────────┐')
    print('   │ • Gráficos referenciam dados das folhas │')
    print('   │ • Atualização automática ao inserir     │')
    print('   │ • Sem necessidade de macros VBA          │')
    print('   │ • Sistema baseado em referências Excel  │')
    print('   └─────────────────────────────────────────┘')
    
    print('\n   🖥️ EXPERIÊNCIA DO USUÁRIO:')
    print('   ┌─────────────────────────────────────────┐')
    print('   │ PARA INSERIR DADOS:                     │')
    print('   │ 1. Clica na aba da barragem             │')
    print('   │ 2. Rola até ao fim dos dados            │')
    print('   │ 3. Clica na próxima linha vazia         │')
    print('   │ 4. Digite os valores nas colunas        │')
    print('   │ 5. Data é preenchida automaticamente    │')
    print('   │                                         │')
    print('   │ PARA VER GRÁFICOS:                      │')
    print('   │ 1. Clica na aba "GRAFICOS DE BARRAGEM"  │')
    print('   │ 2. Visualiza gráficos atualizados       │')
    print('   │ 3. Pode comparar diferentes barragens   │')
    print('   └─────────────────────────────────────────┘')

def show_technical_details():
    """Mostrar detalhes técnicos"""
    print('\n⚙️ 6. DETALHES TÉCNICOS')
    print('-' * 25)
    print('   📋 FORMATO DO FICHEIRO:')
    print('      • Extensão: .xlsm (Excel Macro-Enabled)')
    print('      • Tamanho: ~2.6MB')
    print('      • Sem macros VBA encontradas')
    print('      • Sistema baseado em fórmulas Excel')
    
    print('\n   📊 TIPOS DE GRÁFICOS:')
    print('      • 5 gráficos de barras (BarChart)')
    print('      • 2 gráficos de linha (LineChart)')
    print('      • Atualização dinâmica baseada em ranges')
    
    print('\n   🔧 AUTOMAÇÃO:')
    print('      • Datas geradas com fórmulas (=A+1)')
    print('      • Referências dinâmicas nos gráficos')
    print('      • Sem botões ou controles visuais')
    print('      • Sistema manual de entrada de dados')

if __name__ == "__main__":
    try:
        dam_sheets, total_records = generate_comprehensive_report()
        show_workflow_analysis()
        show_technical_details()
        
        print(f'\n🎯 RESUMO EXECUTIVO')
        print('=' * 20)
        print(f'Sistema monitora {len(dam_sheets)} barragens com {total_records:,} registos totais.')
        print('Entrada manual de dados, visualização automática via gráficos Excel.')
        print('Fluxo simples: Inserir dados → Gráficos atualizados automaticamente.')
        
    except Exception as e:
        print(f'❌ Erro: {e}')
        import traceback
        traceback.print_exc()
