@extends('layouts.app')

@section('title', 'Sistema de Alertas')

@section('content')
<div class="space-y-6">
    <!-- Header com Dashboard de Estatísticas -->
    <div class="bg-white rounded-lg shadow p-6">
        <div class="flex items-center justify-between mb-6">
            <div>
                <h1 class="text-3xl font-bold text-gray-800">
                    <i class="fas fa-exclamation-triangle mr-3 text-red-600"></i>
                    Sistema de Alertas
                </h1>
                <p class="text-gray-600 mt-2">Monitorização em tempo real dos alertas do sistema</p>
            </div>
            <div class="flex items-center space-x-2">
                <div class="w-3 h-3 bg-green-500 rounded-full animate-pulse"></div>
                <span class="text-sm text-gray-600">Sistema ativo</span>
            </div>
        </div>

        <!-- Dashboard de Estatísticas em Tempo Real -->
        <div class="grid grid-cols-1 md:grid-cols-4 gap-6 mb-6">
            <!-- Total de Alertas -->
            <div class="bg-gradient-to-br from-blue-50 to-blue-100 rounded-xl p-6 border border-blue-200">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-blue-600 text-sm font-medium mb-1">Total de Alertas</p>
                        <p class="text-3xl font-bold text-blue-700">{{ $stats['total'] }}</p>
                        <p class="text-xs text-blue-500 mt-1">Histórico completo</p>
                    </div>
                    <div class="bg-blue-500 rounded-full p-3">
                        <i class="fas fa-bell text-white text-xl"></i>
                    </div>
                </div>
            </div>

            <!-- Alertas Ativos -->
            <div class="bg-gradient-to-br from-orange-50 to-orange-100 rounded-xl p-6 border border-orange-200">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-orange-600 text-sm font-medium mb-1">Alertas Ativos</p>
                        <p class="text-3xl font-bold text-orange-700">{{ $stats['ativos'] }}</p>
                        <p class="text-xs text-orange-500 mt-1">Requerem atenção</p>
                    </div>
                    <div class="bg-orange-500 rounded-full p-3">
                        <i class="fas fa-exclamation-circle text-white text-xl"></i>
                    </div>
                </div>
            </div>

            <!-- Emergências -->
            <div class="bg-gradient-to-br from-red-50 to-red-100 rounded-xl p-6 border border-red-200">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-red-600 text-sm font-medium mb-1">Emergências</p>
                        <p class="text-3xl font-bold text-red-700">{{ $stats['emergencia'] }}</p>
                        <p class="text-xs text-red-500 mt-1">Crítico</p>
                    </div>
                    <div class="bg-red-500 rounded-full p-3 {{ $stats['emergencia'] > 0 ? 'animate-pulse' : '' }}">
                        <i class="fas fa-skull-crossbones text-white text-xl"></i>
                    </div>
                </div>
            </div>

            <!-- Alertas de Nível Alto -->
            <div class="bg-gradient-to-br from-yellow-50 to-yellow-100 rounded-xl p-6 border border-yellow-200">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-yellow-600 text-sm font-medium mb-1">Nível Alerta</p>
                        <p class="text-3xl font-bold text-yellow-700">{{ $stats['alerta'] }}</p>
                        <p class="text-xs text-yellow-500 mt-1">Monitorizar</p>
                    </div>
                    <div class="bg-yellow-500 rounded-full p-3">
                        <i class="fas fa-exclamation text-white text-xl"></i>
                    </div>
                </div>
            </div>
        </div>

        <!-- Indicador de Status do Sistema -->
        @php
            $sistemaStatus = 'normal';
            $sistemaColor = 'green';
            $sistemaIcon = 'check-circle';
            $sistemaTexto = 'Sistema funcionando normalmente';
            
            if ($stats['emergencia'] > 0) {
                $sistemaStatus = 'critico';
                $sistemaColor = 'red';
                $sistemaIcon = 'exclamation-triangle';
                $sistemaTexto = 'Estado crítico - Emergências ativas';
            } elseif ($stats['alerta'] > 3) {
                $sistemaStatus = 'atencao';
                $sistemaColor = 'yellow';
                $sistemaIcon = 'exclamation-circle';
                $sistemaTexto = 'Requer atenção - Múltiplos alertas ativos';
            } elseif ($stats['ativos'] > 5) {
                $sistemaStatus = 'monitorizar';
                $sistemaColor = 'orange';
                $sistemaIcon = 'eye';
                $sistemaTexto = 'Monitorização intensiva recomendada';
            }
        @endphp
        
        <div class="bg-{{ $sistemaColor }}-50 border border-{{ $sistemaColor }}-200 rounded-lg p-4">
            <div class="flex items-center">
                <i class="fas fa-{{ $sistemaIcon }} text-{{ $sistemaColor }}-600 mr-3 text-lg"></i>
                <div>
                    <p class="text-{{ $sistemaColor }}-800 font-semibold">{{ $sistemaTexto }}</p>
                    <p class="text-{{ $sistemaColor }}-600 text-sm">
                        Última verificação: {{ now()->format('d/m/Y H:i:s') }}
                    </p>
                </div>
            </div>
        </div>
    </div>

    <!-- Filtros Avançados -->
    <div class="bg-white rounded-lg shadow">
        <div class="p-6">
            <h3 class="text-lg font-semibold text-gray-900 mb-4">
                <i class="fas fa-filter mr-2 text-gray-600"></i>
                Filtros de Alertas
            </h3>
            <form method="GET" class="grid grid-cols-1 md:grid-cols-4 gap-4">
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">Tipo de Alerta</label>
                    <select name="tipo" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-red-500">
                        <option value="">Todos os tipos</option>
                        <option value="nivel_critico" {{ request('tipo') == 'nivel_critico' ? 'selected' : '' }}>Nível Crítico</option>
                        <option value="descarga" {{ request('tipo') == 'descarga' ? 'selected' : '' }}>Descarga</option>
                        <option value="precipitacao" {{ request('tipo') == 'precipitacao' ? 'selected' : '' }}>Precipitação</option>
                        <option value="emergencia" {{ request('tipo') == 'emergencia' ? 'selected' : '' }}>Emergência</option>
                    </select>
                </div>
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">Nível de Severidade</label>
                    <select name="nivel" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-red-500">
                        <option value="">Todos os níveis</option>
                        <option value="informacao" {{ request('nivel') == 'informacao' ? 'selected' : '' }}>Informação</option>
                        <option value="atencao" {{ request('nivel') == 'atencao' ? 'selected' : '' }}>Atenção</option>
                        <option value="alerta" {{ request('nivel') == 'alerta' ? 'selected' : '' }}>Alerta</option>
                        <option value="emergencia" {{ request('nivel') == 'emergencia' ? 'selected' : '' }}>Emergência</option>
                    </select>
                </div>
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">Estado</label>
                    <select name="estado" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-red-500">
                        <option value="">Todos os estados</option>
                        <option value="activo" {{ request('estado') == 'activo' ? 'selected' : '' }}>Ativos</option>
                        <option value="resolvido" {{ request('estado') == 'resolvido' ? 'selected' : '' }}>Resolvidos</option>
                        <option value="cancelado" {{ request('estado') == 'cancelado' ? 'selected' : '' }}>Cancelados</option>
                    </select>
                </div>
                <div class="flex items-end space-x-2">
                    <button type="submit" class="flex-1 bg-red-600 text-white px-4 py-2 rounded-md hover:bg-red-700 transition-colors">
                        <i class="fas fa-search mr-2"></i>
                        Filtrar
                    </button>
                    <a href="{{ route('alertas.index') }}" class="bg-gray-300 text-gray-700 px-4 py-2 rounded-md hover:bg-gray-400 transition-colors">
                        <i class="fas fa-times"></i>
                    </a>
                </div>
            </form>
        </div>
    </div>

    <!-- Lista de Alertas -->
    <div class="bg-white rounded-lg shadow overflow-hidden">
        <div class="px-6 py-4 border-b border-gray-200 bg-gray-50">
            <h3 class="text-lg font-medium text-gray-900">
                <i class="fas fa-list mr-2"></i>
                Alertas do Sistema ({{ $alertas->total() }})
            </h3>
        </div>
        
        @if($alertas->count() > 0)
            <div class="divide-y divide-gray-200">
                @foreach($alertas as $alerta)
                <div class="p-6 hover:bg-gray-50 transition-colors border-l-4 border-{{ $alerta->getCorNivel() }}-500">
                    <div class="flex items-start justify-between">
                        <div class="flex-1">
                            <!-- Header do Alerta -->
                            <div class="flex items-center space-x-3 mb-3">
                                <!-- Badge de Nível -->
                                <span class="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium bg-{{ $alerta->getCorNivel() }}-100 text-{{ $alerta->getCorNivel() }}-800 border border-{{ $alerta->getCorNivel() }}-200">
                                    @if($alerta->nivel == 'emergencia')
                                        <i class="fas fa-skull-crossbones mr-1"></i>
                                    @elseif($alerta->nivel == 'alerta')
                                        <i class="fas fa-exclamation-triangle mr-1"></i>
                                    @elseif($alerta->nivel == 'atencao')
                                        <i class="fas fa-exclamation-circle mr-1"></i>
                                    @else
                                        <i class="fas fa-info-circle mr-1"></i>
                                    @endif
                                    {{ $alerta->getNivelFormatado() }}
                                </span>

                                <!-- Badge de Tipo -->
                                <span class="inline-flex items-center px-2 py-1 rounded text-xs font-medium bg-gray-100 text-gray-700">
                                    {{ $alerta->getTipoFormatado() }}
                                </span>

                                <!-- Badge de Estado -->
                                @if($alerta->isAtivo())
                                    <span class="inline-flex items-center px-2 py-1 rounded text-xs font-medium bg-green-100 text-green-700 animate-pulse">
                                        <i class="fas fa-circle mr-1 text-green-500"></i>
                                        Ativo
                                    </span>
                                @elseif($alerta->estado == 'resolvido')
                                    <span class="inline-flex items-center px-2 py-1 rounded text-xs font-medium bg-blue-100 text-blue-700">
                                        <i class="fas fa-check-circle mr-1"></i>
                                        Resolvido
                                    </span>
                                @else
                                    <span class="inline-flex items-center px-2 py-1 rounded text-xs font-medium bg-gray-100 text-gray-700">
                                        <i class="fas fa-times-circle mr-1"></i>
                                        Cancelado
                                    </span>
                                @endif

                                <!-- Timestamp -->
                                <span class="text-xs text-gray-500">
                                    <i class="fas fa-clock mr-1"></i>
                                    {{ $alerta->created_at->format('d/m/Y H:i') }}
                                </span>
                            </div>

                            <!-- Título e Descrição -->
                            <h4 class="text-lg font-semibold text-gray-900 mb-2">{{ $alerta->titulo }}</h4>
                            <p class="text-gray-700 mb-3">{{ $alerta->descricao }}</p>

                            <!-- Detalhes Técnicos -->
                            <div class="grid grid-cols-1 md:grid-cols-3 gap-4 mb-4">
                                <div class="bg-gray-50 rounded-lg p-3">
                                    <p class="text-xs text-gray-600 mb-1">Infraestrutura</p>
                                    <p class="font-medium text-gray-900">
                                        <i class="fas fa-{{ $alerta->barragem ? 'dam' : 'satellite-dish' }} mr-1 text-blue-500"></i>
                                        {{ $alerta->getNomeInfrastrutura() }}
                                    </p>
                                </div>
                                @if($alerta->valor_registado)
                                <div class="bg-gray-50 rounded-lg p-3">
                                    <p class="text-xs text-gray-600 mb-1">Valor Registado</p>
                                    <p class="font-medium text-gray-900">{{ $alerta->valor_registado }}</p>
                                </div>
                                @endif
                                @if($alerta->valor_limite)
                                <div class="bg-gray-50 rounded-lg p-3">
                                    <p class="text-xs text-gray-600 mb-1">Limite de Segurança</p>
                                    <p class="font-medium text-gray-900">{{ $alerta->valor_limite }}</p>
                                </div>
                                @endif
                            </div>

                            <!-- Duração -->
                            <div class="flex items-center text-sm text-gray-600">
                                <i class="fas fa-hourglass-half mr-2"></i>
                                <span>{{ $alerta->getDuracaoFormatada() }}</span>
                            </div>
                        </div>

                        <!-- Ações do Alerta -->
                        <div class="ml-6 flex flex-col space-y-2">
                            <a href="{{ route('alertas.show', $alerta) }}" 
                               class="inline-flex items-center px-3 py-1 border border-gray-300 text-xs leading-4 font-medium rounded text-gray-700 bg-white hover:bg-gray-50 transition-colors">
                                <i class="fas fa-eye mr-1"></i>
                                Ver Detalhes
                            </a>
                            
                            @if($alerta->isAtivo())
                                <button onclick="openResolveModal({{ $alerta->id }}, '{{ $alerta->titulo }}')"
                                        class="inline-flex items-center px-3 py-1 border border-transparent text-xs leading-4 font-medium rounded text-white bg-green-600 hover:bg-green-700 transition-colors">
                                    <i class="fas fa-check mr-1"></i>
                                    Resolver
                                </button>
                                
                                <button onclick="openCancelModal({{ $alerta->id }}, '{{ $alerta->titulo }}')"
                                        class="inline-flex items-center px-3 py-1 border border-transparent text-xs leading-4 font-medium rounded text-white bg-red-600 hover:bg-red-700 transition-colors">
                                    <i class="fas fa-times mr-1"></i>
                                    Cancelar
                                </button>
                            @endif
                        </div>
                    </div>
                </div>
                @endforeach
            </div>
            
            <!-- Pagination -->
            @if($alertas->hasPages())
            <div class="px-6 py-4 border-t border-gray-200 bg-gray-50">
                {{ $alertas->withQueryString()->links() }}
            </div>
            @endif
        @else
            <div class="text-center py-12">
                <div class="w-24 h-24 bg-gradient-to-br from-green-100 to-green-200 rounded-full flex items-center justify-center mx-auto mb-4 shadow-lg">
                    <i class="fas fa-shield-alt text-4xl text-green-600"></i>
                </div>
                <h3 class="text-lg font-medium text-gray-900 mb-2">Nenhum alerta encontrado</h3>
                <p class="text-gray-500 mb-6 max-w-md mx-auto">
                    @if(request()->filled('tipo') || request()->filled('nivel') || request()->filled('estado'))
                        Nenhum alerta corresponde aos filtros aplicados.
                    @else
                        Sistema funcionando normalmente. Não há alertas ativos no momento.
                    @endif
                </p>
                @if(request()->filled('tipo') || request()->filled('nivel') || request()->filled('estado'))
                    <a href="{{ route('alertas.index') }}" 
                       class="inline-flex items-center px-4 py-2 border border-gray-300 rounded-md shadow-sm text-sm font-medium text-gray-700 bg-white hover:bg-gray-50 transition-colors">
                        <i class="fas fa-times mr-2"></i>
                        Limpar Filtros
                    </a>
                @endif
            </div>
        @endif
    </div>
</div>

<!-- Modal para Resolução de Alerta -->
<div id="resolveModal" class="hidden fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
    <div class="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
        <div class="mt-3">
            <div class="flex items-center mb-4">
                <div class="mx-auto flex-shrink-0 flex items-center justify-center h-12 w-12 rounded-full bg-green-100">
                    <i class="fas fa-check text-green-600 text-xl"></i>
                </div>
            </div>
            <div class="text-center">
                <h3 class="text-lg leading-6 font-medium text-gray-900 mb-2">Resolver Alerta</h3>
                <p class="text-sm text-gray-500 mb-4" id="resolveAlertTitle"></p>
                <form id="resolveForm" method="POST">
                    @csrf
                    <div class="mb-4">
                        <label class="block text-sm font-medium text-gray-700 mb-2">Ações Tomadas</label>
                        <textarea name="acoes_tomadas" rows="4" required
                                  class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500"
                                  placeholder="Descreva as ações tomadas para resolver este alerta..."></textarea>
                    </div>
                    <div class="flex space-x-3">
                        <button type="submit" class="flex-1 bg-green-600 text-white px-4 py-2 rounded-md hover:bg-green-700 transition-colors">
                            Resolver
                        </button>
                        <button type="button" onclick="closeModal('resolveModal')" class="flex-1 bg-gray-300 text-gray-700 px-4 py-2 rounded-md hover:bg-gray-400 transition-colors">
                            Cancelar
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<!-- Modal para Cancelamento de Alerta -->
<div id="cancelModal" class="hidden fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
    <div class="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
        <div class="mt-3">
            <div class="flex items-center mb-4">
                <div class="mx-auto flex-shrink-0 flex items-center justify-center h-12 w-12 rounded-full bg-red-100">
                    <i class="fas fa-times text-red-600 text-xl"></i>
                </div>
            </div>
            <div class="text-center">
                <h3 class="text-lg leading-6 font-medium text-gray-900 mb-2">Cancelar Alerta</h3>
                <p class="text-sm text-gray-500 mb-4" id="cancelAlertTitle"></p>
                <form id="cancelForm" method="POST">
                    @csrf
                    <div class="mb-4">
                        <label class="block text-sm font-medium text-gray-700 mb-2">Motivo do Cancelamento</label>
                        <textarea name="motivo" rows="4" required
                                  class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-red-500"
                                  placeholder="Explique por que este alerta deve ser cancelado..."></textarea>
                    </div>
                    <div class="flex space-x-3">
                        <button type="submit" class="flex-1 bg-red-600 text-white px-4 py-2 rounded-md hover:bg-red-700 transition-colors">
                            Cancelar Alerta
                        </button>
                        <button type="button" onclick="closeModal('cancelModal')" class="flex-1 bg-gray-300 text-gray-700 px-4 py-2 rounded-md hover:bg-gray-400 transition-colors">
                            Fechar
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<script>
function openResolveModal(alertaId, titulo) {
    document.getElementById('resolveAlertTitle').textContent = titulo;
    document.getElementById('resolveForm').action = `/alertas/${alertaId}/resolve`;
    document.getElementById('resolveModal').classList.remove('hidden');
}

function openCancelModal(alertaId, titulo) {
    document.getElementById('cancelAlertTitle').textContent = titulo;
    document.getElementById('cancelForm').action = `/alertas/${alertaId}/cancel`;
    document.getElementById('cancelModal').classList.remove('hidden');
}

function closeModal(modalId) {
    document.getElementById(modalId).classList.add('hidden');
}

// Auto-refresh das estatísticas a cada 30 segundos
setTimeout(function() {
    location.reload();
}, 30000);

// Fechar modais ao clicar fora
window.onclick = function(event) {
    const resolveModal = document.getElementById('resolveModal');
    const cancelModal = document.getElementById('cancelModal');
    if (event.target == resolveModal) {
        resolveModal.classList.add('hidden');
    }
    if (event.target == cancelModal) {
        cancelModal.classList.add('hidden');
    }
}
</script>
@endsection