@extends('layouts.app')

@section('title', 'Editar Leitura - ' . $barragem->nome)

@section('content')
<div class="space-y-6">
    <!-- Header -->
    <div class="bg-white rounded-lg shadow">
        <!-- Breadcrumb -->
        <div class="px-6 py-3 border-b border-gray-200 bg-gray-50">
            <nav class="flex" aria-label="Breadcrumb">
                <ol class="inline-flex items-center space-x-1 md:space-x-3">
                    <li class="inline-flex items-center">
                        <a href="{{ route('dashboard') }}" class="text-gray-500 hover:text-gray-700">
                            <i class="fas fa-home mr-1"></i>
                            Dashboard
                        </a>
                    </li>
                    <li>
                        <div class="flex items-center">
                            <i class="fas fa-chevron-right text-gray-400 mx-4"></i>
                            <a href="{{ route('barragens.index') }}" class="text-gray-500 hover:text-gray-700">Barragens</a>
                        </div>
                    </li>
                    <li>
                        <div class="flex items-center">
                            <i class="fas fa-chevron-right text-gray-400 mx-4"></i>
                            <a href="{{ route('barragens.show', $barragem) }}" class="text-gray-500 hover:text-gray-700">{{ $barragem->nome }}</a>
                        </div>
                    </li>
                    <li>
                        <div class="flex items-center">
                            <i class="fas fa-chevron-right text-gray-400 mx-4"></i>
                            <a href="{{ route('barragens.leituras', $barragem) }}" class="text-gray-500 hover:text-gray-700">Leituras</a>
                        </div>
                    </li>
                    <li aria-current="page">
                        <div class="flex items-center">
                            <i class="fas fa-chevron-right text-gray-400 mx-4"></i>
                            <span class="text-gray-800 font-medium">Editar Leitura</span>
                        </div>
                    </li>
                </ol>
            </nav>
        </div>

        <!-- Header Principal -->
        <div class="px-6 py-6">
            <div class="flex items-center justify-between">
                <div class="flex items-center">
                    <div class="w-12 h-12 bg-gradient-to-br from-blue-500 to-blue-600 rounded-lg flex items-center justify-center mr-4">
                        <i class="fas fa-edit text-white text-xl"></i>
                    </div>
                    <div>
                        <h1 class="text-2xl font-bold text-gray-900">Editar Leitura</h1>
                        <p class="text-gray-600">{{ $barragem->nome }} - {{ $leitura->data_leitura->format('d/m/Y') }} as {{ $leitura->hora_leitura ? \Carbon\Carbon::parse($leitura->hora_leitura)->format('H:i') : '06:00' }}</p>
                    </div>
                </div>
                <div class="text-right text-sm text-gray-500">
                    <div class="bg-blue-50 border border-blue-200 rounded-lg px-4 py-2">
                        <i class="fas fa-info-circle text-blue-500 mr-1"></i>
                        @if($barragem->escala_npa && $barragem->cota_npa)
                            Insira o <strong>Nivel Hidrometrico</strong> (leitura na regua), Cota e calculada automaticamente
                        @else
                            Insira a <strong>Cota</strong>, os outros valores sao calculados automaticamente
                        @endif
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Formulário -->
    <div class="bg-white rounded-lg shadow p-6">
        <form method="POST" action="{{ route('barragens.leituras.update', [$barragem, $leitura]) }}" id="leituraForm">
            @csrf
            @method('PUT')

            <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
                <!-- COLUNA 1: Dados de Entrada (Principal) -->
                <div class="lg:col-span-1 space-y-4">
                    <div class="bg-green-50 border border-green-200 rounded-lg p-4">
                        <h4 class="font-medium text-green-800 flex items-center border-b border-green-300 pb-2 mb-4">
                            <i class="fas fa-keyboard mr-2"></i>
                            Dados de Entrada
                        </h4>

                        <!-- Data e Hora -->
                        <div class="grid grid-cols-2 gap-3 mb-4">
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-1">Data *</label>
                                <input type="date" name="data_leitura" required
                                       value="{{ old('data_leitura', $leitura->data_leitura->format('Y-m-d')) }}"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500 focus:border-green-500">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-1">Hora *</label>
                                @php
                                    $horaAtual = $leitura->hora_leitura ? \Carbon\Carbon::parse($leitura->hora_leitura)->format('H:i') : '06:00';
                                @endphp
                                <select name="hora_leitura" required
                                        class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500 focus:border-green-500">
                                    <option value="06:00" {{ $horaAtual == '06:00' ? 'selected' : '' }}>06:00h</option>
                                    <option value="12:00" {{ $horaAtual == '12:00' ? 'selected' : '' }}>12:00h</option>
                                    <option value="18:00" {{ $horaAtual == '18:00' ? 'selected' : '' }}>18:00h</option>
                                </select>
                            </div>
                        </div>

                        @php
                            $temEscalaNpa = $barragem->escala_npa && $barragem->cota_npa;
                        @endphp

                        @if($temEscalaNpa)
                        <!-- NIVEL HIDROMETRICO - Campo Principal (quando escala_npa configurada) -->
                        <div class="mb-4">
                            <label class="block text-sm font-bold text-green-700 mb-1">
                                <i class="fas fa-ruler-vertical mr-1"></i>
                                Nivel Hidrometrico (m) *
                            </label>
                            <input type="number" step="0.01" name="nivel_hidrometrico" id="nivel_hidrometrico" required
                                   value="{{ old('nivel_hidrometrico', $leitura->nivel_hidrometrico) }}"
                                   class="w-full px-4 py-3 text-lg font-bold border-2 border-green-400 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500 focus:border-green-500 bg-white"
                                   placeholder="Leitura na regua/escala"
                                   onchange="calcularCota()">
                            <p class="text-xs text-green-600 mt-1">Valor lido na escala - Cota calculada: <span id="cotaCalculada" class="font-bold">{{ number_format($leitura->cota_actual ?? 0, 2) }}</span>m</p>
                            <input type="hidden" name="cota_actual" id="cota_actual" value="{{ old('cota_actual', $leitura->cota_actual) }}">
                        </div>
                        @else
                        <!-- COTA - Campo Principal (entrada direta) -->
                        <div class="mb-4">
                            <label class="block text-sm font-bold text-green-700 mb-1">
                                <i class="fas fa-ruler-vertical mr-1"></i>
                                Cota Atual (m) *
                            </label>
                            <input type="number" step="0.01" name="cota_actual" id="cota_actual" required
                                   value="{{ old('cota_actual', $leitura->cota_actual) }}"
                                   class="w-full px-4 py-3 text-lg font-bold border-2 border-green-400 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500 focus:border-green-500 bg-white"
                                   placeholder="Ex: 123.45">
                            <p class="text-xs text-green-600 mt-1">Este e o valor principal - Volume e % sao calculados a partir dele</p>
                        </div>
                        @endif

                        <!-- Caudal Ecologico -->
                        <div class="mb-4">
                            <label class="block text-sm font-medium text-gray-700 mb-1">Caudal Ecologico (m3/s)</label>
                            <input type="number" step="0.001" name="caudal_ecologico"
                                   value="{{ old('caudal_ecologico', $leitura->caudal_ecologico) }}"
                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500 focus:border-green-500"
                                   placeholder="Ex: 0.500">
                        </div>

                        <!-- Metodo -->
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">Metodo de Leitura</label>
                            <select name="metodo_leitura"
                                    class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500 focus:border-green-500">
                                <option value="manual" {{ old('metodo_leitura', $leitura->metodo_leitura) == 'manual' ? 'selected' : '' }}>Manual</option>
                                <option value="automatico" {{ old('metodo_leitura', $leitura->metodo_leitura) == 'automatico' ? 'selected' : '' }}>Automatico</option>
                                <option value="telemetria" {{ old('metodo_leitura', $leitura->metodo_leitura) == 'telemetria' ? 'selected' : '' }}>Telemetria</option>
                            </select>
                        </div>
                    </div>

                    <!-- Observacoes -->
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-1">Observacoes</label>
                        <textarea name="observacoes" rows="3"
                                  class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                                  placeholder="Notas ou observacoes...">{{ old('observacoes', $leitura->observacoes) }}</textarea>
                    </div>
                </div>

                <!-- COLUNA 2: Valores Calculados -->
                <div class="lg:col-span-1 space-y-4">
                    <div class="bg-gray-50 border border-gray-200 rounded-lg p-4">
                        <h4 class="font-medium text-gray-700 flex items-center border-b border-gray-300 pb-2 mb-4">
                            <i class="fas fa-calculator mr-2 text-gray-500"></i>
                            Valores Calculados
                            <span class="ml-2 text-xs bg-gray-200 px-2 py-1 rounded">Automatico</span>
                        </h4>

                        <div class="space-y-3">
                            <div class="grid grid-cols-2 gap-3">
                                <div>
                                    <label class="block text-xs text-gray-500 mb-1">Volume Total (Mm3)</label>
                                    <input type="number" step="0.01" name="volume_actual" id="volume_actual"
                                           value="{{ old('volume_actual', $leitura->volume_actual) }}"
                                           readonly
                                           class="w-full px-3 py-2 bg-gray-100 border border-gray-300 rounded-md text-gray-600 cursor-not-allowed">
                                </div>
                                <div>
                                    <label class="block text-xs text-gray-500 mb-1">% Total</label>
                                    <input type="number" step="0.01" name="percentagem_total" id="percentagem_total"
                                           value="{{ old('percentagem_total', $leitura->percentagem_total) }}"
                                           readonly
                                           class="w-full px-3 py-2 bg-gray-100 border border-gray-300 rounded-md text-gray-600 cursor-not-allowed">
                                </div>
                            </div>

                            <div class="grid grid-cols-2 gap-3">
                                <div>
                                    <label class="block text-xs text-gray-500 mb-1">% Enchimento</label>
                                    <input type="number" step="0.01" name="percentagem_enchimento" id="percentagem_enchimento"
                                           value="{{ old('percentagem_enchimento', $leitura->percentagem_enchimento) }}"
                                           readonly
                                           class="w-full px-3 py-2 bg-gray-100 border border-gray-300 rounded-md text-gray-600 cursor-not-allowed">
                                </div>
                                <div>
                                    <label class="block text-xs text-gray-500 mb-1">Variacao Volume</label>
                                    <input type="number" step="0.01" name="variacao_volume" id="variacao_volume"
                                           value="{{ old('variacao_volume', $leitura->variacao_volume) }}"
                                           readonly
                                           class="w-full px-3 py-2 bg-gray-100 border border-gray-300 rounded-md text-gray-600 cursor-not-allowed">
                                </div>
                            </div>

                            <div class="grid grid-cols-2 gap-3">
                                <div>
                                    <label class="block text-xs text-gray-500 mb-1">Descarga Total (m3/s)</label>
                                    <input type="number" step="0.001" name="descarga_total" id="descarga_total"
                                           value="{{ old('descarga_total', $leitura->descarga_total) }}"
                                           readonly
                                           class="w-full px-3 py-2 bg-gray-100 border border-gray-300 rounded-md text-gray-600 cursor-not-allowed">
                                </div>
                                <div>
                                    <label class="block text-xs text-gray-500 mb-1">Comp. Ano Anterior (%)</label>
                                    <input type="number" step="0.01" name="comparacao_ano_anterior" id="comparacao_ano_anterior"
                                           value="{{ old('comparacao_ano_anterior', $leitura->comparacao_ano_anterior) }}"
                                           readonly
                                           class="w-full px-3 py-2 bg-gray-100 border border-gray-300 rounded-md text-gray-600 cursor-not-allowed">
                                </div>
                            </div>
                        </div>

                        <p class="text-xs text-green-600 mt-3 italic">
                            <i class="fas fa-sync-alt mr-1"></i>
                            Valores calculados em tempo real
                        </p>
                    </div>

                    <!-- Referencia da Barragem -->
                    <div class="bg-blue-50 border border-blue-200 rounded-lg p-4">
                        <h4 class="font-medium text-blue-800 text-sm mb-2">
                            <i class="fas fa-water mr-1"></i>
                            Parametros da Barragem
                        </h4>
                        <div class="grid grid-cols-2 gap-2 text-xs">
                            <div><span class="text-gray-500">Vol. Maximo:</span> <strong>{{ number_format($barragem->volume_maximo ?? 0, 2) }} Mm3</strong></div>
                            <div><span class="text-gray-500">Vol. Morto:</span> <strong>{{ number_format($barragem->volume_morto ?? 0, 2) }} Mm3</strong></div>
                            <div><span class="text-gray-500">Cota NPA:</span> <strong>{{ number_format($barragem->cota_npa ?? 0, 2) }} m</strong></div>
                            <div><span class="text-gray-500">Cota NME:</span> <strong>{{ number_format($barragem->cota_nme ?? 0, 2) }} m</strong></div>
                            @if($barragem->escala_npa)
                            <div class="col-span-2 mt-1 pt-1 border-t border-blue-200">
                                <span class="text-gray-500">Escala NPA:</span> <strong>{{ number_format($barragem->escala_npa, 2) }} m</strong>
                                <span class="text-gray-400 ml-2">(Cota = NPA - ({{ $barragem->escala_npa }} - Nivel))</span>
                            </div>
                            @endif
                        </div>
                    </div>
                </div>

                <!-- COLUNA 3: Caudais Opcionais -->
                <div class="lg:col-span-1 space-y-4">
                    <div class="bg-cyan-50 border border-cyan-200 rounded-lg p-4">
                        <h4 class="font-medium text-cyan-800 flex items-center border-b border-cyan-300 pb-2 mb-4">
                            <i class="fas fa-water mr-2"></i>
                            Caudais (m3/s)
                            <span class="ml-2 text-xs bg-cyan-200 px-2 py-1 rounded">Opcional</span>
                        </h4>

                        <div class="space-y-3">
                            <div class="grid grid-cols-2 gap-3">
                                <div>
                                    <label class="block text-xs text-gray-600 mb-1">Efluente</label>
                                    <input type="number" step="0.001" name="caudal_efluente"
                                           value="{{ old('caudal_efluente', $leitura->caudal_efluente) }}"
                                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-cyan-500 focus:border-cyan-500">
                                </div>
                                <div>
                                    <label class="block text-xs text-gray-600 mb-1">Captacao</label>
                                    <input type="number" step="0.001" name="caudal_captacao"
                                           value="{{ old('caudal_captacao', $leitura->caudal_captacao) }}"
                                           class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-cyan-500 focus:border-cyan-500">
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Toggle para mostrar campos avancados -->
                    <div class="border border-gray-200 rounded-lg">
                        <button type="button" onclick="toggleAdvanced()"
                                class="w-full px-4 py-3 text-left text-sm font-medium text-gray-700 hover:bg-gray-50 flex items-center justify-between">
                            <span><i class="fas fa-cogs mr-2"></i>Campos Avancados</span>
                            <i class="fas fa-chevron-down" id="advancedIcon"></i>
                        </button>
                        <div id="advancedFields" class="hidden p-4 border-t border-gray-200 space-y-3">
                            <div class="grid grid-cols-2 gap-3">
                                <div>
                                    <label class="block text-xs text-gray-500 mb-1">Afluente</label>
                                    <input type="number" step="0.001" name="caudal_afluente"
                                           value="{{ old('caudal_afluente', $leitura->caudal_afluente) }}"
                                           class="w-full px-2 py-1 text-sm border border-gray-300 rounded">
                                </div>
                                <div>
                                    <label class="block text-xs text-gray-500 mb-1">Turbinado</label>
                                    <input type="number" step="0.001" name="caudal_turbinado"
                                           value="{{ old('caudal_turbinado', $leitura->caudal_turbinado) }}"
                                           class="w-full px-2 py-1 text-sm border border-gray-300 rounded">
                                </div>
                                <div>
                                    <label class="block text-xs text-gray-500 mb-1">Desc. Fundo</label>
                                    <input type="number" step="0.001" name="descarga_fundo"
                                           value="{{ old('descarga_fundo', $leitura->descarga_fundo) }}"
                                           class="w-full px-2 py-1 text-sm border border-gray-300 rounded">
                                </div>
                                <div>
                                    <label class="block text-xs text-gray-500 mb-1">Desc. Superficie</label>
                                    <input type="number" step="0.001" name="descarga_superficie"
                                           value="{{ old('descarga_superficie', $leitura->descarga_superficie) }}"
                                           class="w-full px-2 py-1 text-sm border border-gray-300 rounded">
                                </div>
                                <div>
                                    <label class="block text-xs text-gray-500 mb-1">Descarregador</label>
                                    <input type="number" step="0.001" name="caudal_descarregador"
                                           value="{{ old('caudal_descarregador', $leitura->caudal_descarregador) }}"
                                           class="w-full px-2 py-1 text-sm border border-gray-300 rounded">
                                </div>
                                <div>
                                    <label class="block text-xs text-gray-500 mb-1">Evaporacao</label>
                                    <input type="number" step="0.001" name="evaporacao"
                                           value="{{ old('evaporacao', $leitura->evaporacao) }}"
                                           class="w-full px-2 py-1 text-sm border border-gray-300 rounded">
                                </div>
                                <div>
                                    <label class="block text-xs text-gray-500 mb-1">Infiltracao</label>
                                    <input type="number" step="0.001" name="infiltracao"
                                           value="{{ old('infiltracao', $leitura->infiltracao) }}"
                                           class="w-full px-2 py-1 text-sm border border-gray-300 rounded">
                                </div>
                                <div>
                                    <label class="block text-xs text-gray-500 mb-1">Vol. Morto</label>
                                    <input type="number" step="0.01" name="volume_morto"
                                           value="{{ old('volume_morto', $leitura->volume_morto) }}"
                                           class="w-full px-2 py-1 text-sm border border-gray-300 rounded">
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Botoes -->
            <div class="flex justify-between items-center mt-8 pt-6 border-t">
                <a href="{{ route('barragens.leituras', $barragem) }}"
                   class="px-6 py-2 bg-gray-500 text-white rounded-md hover:bg-gray-600 transition-colors">
                    <i class="fas fa-times mr-2"></i>Cancelar
                </a>
                <button type="submit"
                        class="px-8 py-3 bg-green-600 text-white font-medium rounded-md hover:bg-green-700 transition-colors">
                    <i class="fas fa-save mr-2"></i>
                    Salvar Leitura
                </button>
            </div>
        </form>
    </div>
</div>

<script>
function toggleAdvanced() {
    const fields = document.getElementById('advancedFields');
    const icon = document.getElementById('advancedIcon');
    if (fields.classList.contains('hidden')) {
        fields.classList.remove('hidden');
        icon.classList.remove('fa-chevron-down');
        icon.classList.add('fa-chevron-up');
    } else {
        fields.classList.add('hidden');
        icon.classList.remove('fa-chevron-up');
        icon.classList.add('fa-chevron-down');
    }
}

// Configuracao da barragem para calculos
const barragemConfig = {
    cota_npa: {{ $barragem->cota_npa ?? 0 }},
    cota_nme: {{ $barragem->cota_nme ?? 0 }},
    volume_maximo: {{ $barragem->volume_maximo ?? 0 }},
    volume_morto: {{ $barragem->volume_morto ?? 0 }},
    escala_npa: {{ $barragem->escala_npa ?? 0 }},
    temEscalaNpa: {{ ($barragem->escala_npa && $barragem->cota_npa) ? 'true' : 'false' }}
};

// Calcular volume a partir da cota usando interpolacao linear
function calcularVolume(cota) {
    const { cota_nme, cota_npa, volume_morto, volume_maximo } = barragemConfig;

    if (!cota || cota <= 0) return 0;

    // Se cota <= NME, retorna volume morto
    if (cota <= cota_nme) {
        return volume_morto;
    }

    // Se cota >= NPA, retorna volume maximo
    if (cota >= cota_npa) {
        return volume_maximo;
    }

    // Interpolacao linear entre NME e NPA
    // Formula: Volume = Volume_Morto + (cota - NME) * (Volume_Max - Volume_Morto) / (NPA - NME)
    const deltaVolume = volume_maximo - volume_morto;
    const deltaCota = cota_npa - cota_nme;

    if (deltaCota <= 0) return volume_morto;

    const volume = volume_morto + ((cota - cota_nme) / deltaCota) * deltaVolume;
    return Math.max(volume_morto, Math.min(volume_maximo, volume));
}

// Calcular todos os valores baseados na cota
function calcularValores() {
    let cota = 0;

    // Obter cota (do campo direto ou calculada do nivel)
    if (barragemConfig.temEscalaNpa) {
        const nivelInput = document.getElementById('nivel_hidrometrico');
        if (nivelInput) {
            const nivel = parseFloat(nivelInput.value) || 0;
            cota = barragemConfig.cota_npa - (barragemConfig.escala_npa - nivel);

            // Atualizar campo hidden e display
            document.getElementById('cota_actual').value = cota.toFixed(2);
            const cotaDisplay = document.getElementById('cotaCalculada');
            if (cotaDisplay) cotaDisplay.textContent = cota.toFixed(2);
        }
    } else {
        const cotaInput = document.getElementById('cota_actual');
        if (cotaInput) {
            cota = parseFloat(cotaInput.value) || 0;
        }
    }

    if (cota <= 0) {
        // Limpar campos
        document.getElementById('volume_actual').value = '';
        document.getElementById('percentagem_total').value = '';
        document.getElementById('percentagem_enchimento').value = '';
        return;
    }

    // Calcular volume
    const volume = calcularVolume(cota);

    // Calcular percentagem total: (Volume / Volume_Max) * 100
    let percTotal = 0;
    if (barragemConfig.volume_maximo > 0) {
        percTotal = (volume / barragemConfig.volume_maximo) * 100;
    }

    // Calcular percentagem de enchimento (util): ((Volume - Vol_Morto) / (Vol_Max - Vol_Morto)) * 100
    let percEnchimento = 0;
    const volumeUtilMax = barragemConfig.volume_maximo - barragemConfig.volume_morto;
    if (volumeUtilMax > 0) {
        const volumeUtil = volume - barragemConfig.volume_morto;
        percEnchimento = Math.max(0, (volumeUtil / volumeUtilMax) * 100);
    }

    // Limitar percentagens entre 0 e 100
    percTotal = Math.max(0, Math.min(100, percTotal));
    percEnchimento = Math.max(0, Math.min(100, percEnchimento));

    // Atualizar campos
    document.getElementById('volume_actual').value = volume.toFixed(2);
    document.getElementById('percentagem_total').value = percTotal.toFixed(2);
    document.getElementById('percentagem_enchimento').value = percEnchimento.toFixed(2);
}

// Alias para compatibilidade
function calcularCota() {
    calcularValores();
}

// Inicializar ao carregar a pagina
document.addEventListener('DOMContentLoaded', function() {
    // Adicionar event listeners
    if (barragemConfig.temEscalaNpa) {
        const nivelInput = document.getElementById('nivel_hidrometrico');
        if (nivelInput) {
            nivelInput.addEventListener('input', calcularValores);
            nivelInput.addEventListener('change', calcularValores);
        }
    } else {
        const cotaInput = document.getElementById('cota_actual');
        if (cotaInput) {
            cotaInput.addEventListener('input', calcularValores);
            cotaInput.addEventListener('change', calcularValores);
        }
    }

    // Calcular valores iniciais
    calcularValores();
});
</script>
@endsection