@extends('layouts.app')

@section('title', 'Gráficos - ' . $barragem->nome)

@section('content')
<div class="space-y-6">
    <!-- Header -->
    <div class="bg-white rounded-lg shadow p-6">
        <div class="flex items-center justify-between">
            <div>
                <nav class="flex items-center space-x-2 text-gray-500 text-sm mb-2">
                    <a href="{{ route('charts.index') }}" class="hover:text-blue-600">Gráficos</a>
                    <i class="fas fa-chevron-right text-xs"></i>
                    <a href="{{ route('charts.barragens') }}" class="hover:text-blue-600">Barragens</a>
                    <i class="fas fa-chevron-right text-xs"></i>
                    <span class="text-gray-800">{{ $barragem->nome }}</span>
                </nav>
                <h1 class="text-2xl font-bold text-gray-800 flex items-center">
                    <i class="fas fa-water mr-3 text-blue-600"></i>
                    {{ $barragem->nome }}
                </h1>
                <p class="text-gray-600 mt-2">Análise detalhada dos níveis e enchimento</p>
            </div>
            <div class="text-right">
                @if($barragem->ultima_leitura)
                    <div class="text-3xl font-bold text-blue-600">{{ number_format($barragem->ultima_leitura->percentagem_enchimento, 1) }}%</div>
                    <div class="text-sm text-gray-500">Enchimento Atual</div>
                @else
                    <div class="text-xl text-gray-500">Sem dados</div>
                @endif
            </div>
        </div>
    </div>

    <!-- Filtros de Período -->
    <div class="bg-white rounded-lg shadow p-6">
        <h2 class="text-lg font-semibold text-gray-800 mb-4">Período de Análise</h2>
        <div class="flex flex-wrap gap-4">
            <button onclick="carregarDados(7)" class="px-4 py-2 bg-blue-100 text-blue-800 rounded-lg hover:bg-blue-200 transition-colors" id="btn-7">
                7 dias
            </button>
            <button onclick="carregarDados(30)" class="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors" id="btn-30">
                30 dias
            </button>
            <button onclick="carregarDados(90)" class="px-4 py-2 bg-blue-100 text-blue-800 rounded-lg hover:bg-blue-200 transition-colors" id="btn-90">
                3 meses
            </button>
            <button onclick="carregarDados(365)" class="px-4 py-2 bg-blue-100 text-blue-800 rounded-lg hover:bg-blue-200 transition-colors" id="btn-365">
                1 ano
            </button>
        </div>
    </div>

    <!-- Estatísticas Atuais -->
    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        @if($barragem->ultima_leitura)
        <div class="bg-white rounded-lg shadow p-6">
            <div class="flex items-center">
                <div class="flex-shrink-0">
                    <div class="w-12 h-12 bg-blue-100 rounded-lg flex items-center justify-center">
                        <i class="fas fa-percentage text-blue-600 text-xl"></i>
                    </div>
                </div>
                <div class="ml-4">
                    <p class="text-sm text-gray-500">Enchimento Atual</p>
                    <p class="text-2xl font-bold text-gray-900">{{ number_format($barragem->ultima_leitura->percentagem_enchimento, 1) }}%</p>
                </div>
            </div>
        </div>

        <div class="bg-white rounded-lg shadow p-6">
            <div class="flex items-center">
                <div class="flex-shrink-0">
                    <div class="w-12 h-12 bg-indigo-100 rounded-lg flex items-center justify-center">
                        <i class="fas fa-arrows-alt-v text-indigo-600 text-xl"></i>
                    </div>
                </div>
                <div class="ml-4">
                    <p class="text-sm text-gray-500">Cota Atual</p>
                    <p class="text-2xl font-bold text-gray-900">{{ number_format($barragem->ultima_leitura->cota_actual, 2) }}</p>
                    <p class="text-xs text-gray-500">metros</p>
                </div>
            </div>
        </div>

        <div class="bg-white rounded-lg shadow p-6">
            <div class="flex items-center">
                <div class="flex-shrink-0">
                    <div class="w-12 h-12 bg-green-100 rounded-lg flex items-center justify-center">
                        <i class="fas fa-cube text-green-600 text-xl"></i>
                    </div>
                </div>
                <div class="ml-4">
                    <p class="text-sm text-gray-500">Volume Atual</p>
                    <p class="text-2xl font-bold text-gray-900">{{ number_format($barragem->ultima_leitura->volume_actual, 1) }}</p>
                    <p class="text-xs text-gray-500">Mm³</p>
                </div>
            </div>
        </div>

        <div class="bg-white rounded-lg shadow p-6">
            <div class="flex items-center">
                <div class="flex-shrink-0">
                    @php
                        $status = $barragem->getStatusNivel($barragem->ultima_leitura->cota_actual);
                    @endphp
                    <div class="w-12 h-12 {{ $status == 'normal' ? 'bg-green-100' : 'bg-red-100' }} rounded-lg flex items-center justify-center">
                        <i class="fas {{ $status == 'normal' ? 'fa-check-circle text-green-600' : 'fa-exclamation-triangle text-red-600' }} text-xl"></i>
                    </div>
                </div>
                <div class="ml-4">
                    <p class="text-sm text-gray-500">Status</p>
                    <p class="text-xl font-bold {{ $status == 'normal' ? 'text-green-600' : 'text-red-600' }}">{{ ucfirst($status) }}</p>
                    <p class="text-xs text-gray-500">{{ $barragem->ultima_leitura->data_leitura->format('d/m/Y') }}</p>
                </div>
            </div>
        </div>
        @else
        <div class="col-span-full bg-yellow-50 border border-yellow-200 rounded-lg p-6 text-center">
            <i class="fas fa-exclamation-triangle text-yellow-600 text-2xl mb-2"></i>
            <h3 class="text-lg font-medium text-yellow-800">Sem dados de leitura</h3>
            <p class="text-yellow-700 mt-1">Esta barragem não possui leituras recentes registradas.</p>
        </div>
        @endif
    </div>

    <!-- Gráfico Principal: Evolução do Enchimento -->
    <div class="bg-white rounded-lg shadow p-6">
        <h3 class="text-lg font-semibold text-gray-800 mb-4">Evolução do Enchimento</h3>
        <div class="h-80">
            <canvas id="grafico-enchimento"></canvas>
        </div>
    </div>

    <!-- Gráficos Secundários -->
    <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <!-- Variação da Cota -->
        <div class="bg-white rounded-lg shadow p-6">
            <h3 class="text-lg font-semibold text-gray-800 mb-4">Variação da Cota</h3>
            <div class="h-64">
                <canvas id="grafico-cota"></canvas>
            </div>
        </div>

        <!-- Volume Armazenado -->
        <div class="bg-white rounded-lg shadow p-6">
            <h3 class="text-lg font-semibold text-gray-800 mb-4">Volume Armazenado</h3>
            <div class="h-64">
                <canvas id="grafico-volume"></canvas>
            </div>
        </div>
    </div>

    <!-- Comparação com Anos Anteriores -->
    @if($comparacao_anos->isNotEmpty())
    <div class="bg-white rounded-lg shadow p-6">
        <h3 class="text-lg font-semibold text-gray-800 mb-4">Comparação com Anos Anteriores</h3>
        <div class="h-80">
            <canvas id="grafico-comparacao"></canvas>
        </div>
    </div>
    @endif

    <!-- Tabela de Histórico Recente -->
    <div class="bg-white rounded-lg shadow">
        <div class="px-6 py-4 border-b border-gray-200">
            <h3 class="text-lg font-semibold text-gray-800">Histórico de Leituras</h3>
        </div>
        <div class="overflow-x-auto">
            <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Data</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Cota (m)</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Volume (Mm³)</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Enchimento (%)</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Variação</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200" id="historico-leituras">
                    <!-- Preenchido via JavaScript -->
                </tbody>
            </table>
        </div>
    </div>

    <!-- Informações da Barragem -->
    <div class="bg-white rounded-lg shadow p-6">
        <h3 class="text-lg font-semibold text-gray-800 mb-4">Informações da Barragem</h3>
        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            <div>
                <h4 class="font-medium text-gray-900 mb-2">Dados Técnicos</h4>
                <dl class="space-y-2 text-sm">
                    <div class="flex justify-between">
                        <dt class="text-gray-600">Código:</dt>
                        <dd class="font-medium">{{ $barragem->codigo }}</dd>
                    </div>
                    <div class="flex justify-between">
                        <dt class="text-gray-600">Tipo:</dt>
                        <dd class="font-medium">{{ $barragem->tipo ?? 'N/A' }}</dd>
                    </div>
                    <div class="flex justify-between">
                        <dt class="text-gray-600">Volume Máximo:</dt>
                        <dd class="font-medium">{{ number_format($barragem->volume_maximo ?? 0, 1) }} Mm³</dd>
                    </div>
                    <div class="flex justify-between">
                        <dt class="text-gray-600">Cota Alerta:</dt>
                        <dd class="font-medium text-orange-600">{{ number_format($barragem->cota_alerta ?? 0, 2) }} m</dd>
                    </div>
                </dl>
            </div>
            <div>
                <h4 class="font-medium text-gray-900 mb-2">Localização</h4>
                <dl class="space-y-2 text-sm">
                    <div class="flex justify-between">
                        <dt class="text-gray-600">Província:</dt>
                        <dd class="font-medium">{{ $barragem->provincia ?? 'N/A' }}</dd>
                    </div>
                    <div class="flex justify-between">
                        <dt class="text-gray-600">Distrito:</dt>
                        <dd class="font-medium">{{ $barragem->distrito ?? 'N/A' }}</dd>
                    </div>
                    <div class="flex justify-between">
                        <dt class="text-gray-600">Rio:</dt>
                        <dd class="font-medium">{{ $barragem->rio ?? 'N/A' }}</dd>
                    </div>
                    <div class="flex justify-between">
                        <dt class="text-gray-600">Bacia:</dt>
                        <dd class="font-medium">{{ $barragem->bacia_hidrografica->nome ?? 'N/A' }}</dd>
                    </div>
                </dl>
            </div>
            <div>
                <h4 class="font-medium text-gray-900 mb-2">Status Operacional</h4>
                <dl class="space-y-2 text-sm">
                    <div class="flex justify-between">
                        <dt class="text-gray-600">Estado:</dt>
                        <dd class="font-medium">
                            <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium
                                @if($barragem->estado == 'operacional') bg-green-100 text-green-800
                                @else bg-red-100 text-red-800 @endif">
                                {{ ucfirst($barragem->estado ?? 'N/A') }}
                            </span>
                        </dd>
                    </div>
                    <div class="flex justify-between">
                        <dt class="text-gray-600">Última Leitura:</dt>
                        <dd class="font-medium">
                            @if($barragem->ultima_leitura)
                                {{ $barragem->ultima_leitura->data_leitura->format('d/m/Y') }}
                            @else
                                <span class="text-red-600">Sem dados</span>
                            @endif
                        </dd>
                    </div>
                </dl>
            </div>
        </div>
    </div>
</div>

<!-- Scripts -->
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
let graficos = {
    enchimento: null,
    cota: null,
    volume: null,
    comparacao: null
};

let periodoAtual = 30;

document.addEventListener('DOMContentLoaded', function() {
    carregarDados(30);
});

async function carregarDados(dias) {
    periodoAtual = dias;

    // Atualizar botões
    document.querySelectorAll('[id^="btn-"]').forEach(btn => {
        btn.className = 'px-4 py-2 bg-blue-100 text-blue-800 rounded-lg hover:bg-blue-200 transition-colors';
    });
    document.getElementById(`btn-${dias}`).className = 'px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors';

    try {
        const response = await fetch(`/graficos/api/barragem/{{ $barragem->id }}/data?days=${dias}`);
        const data = await response.json();

        criarGraficos(data);
        atualizarTabelaHistorico(data);
    } catch (error) {
        console.error('Erro ao carregar dados:', error);
    }
}

function criarGraficos(data) {
    const labels = data.map(d => new Date(d.data_leitura).toLocaleDateString('pt-PT'));

    // Gráfico de Enchimento
    const ctxEnchimento = document.getElementById('grafico-enchimento').getContext('2d');
    if (graficos.enchimento) graficos.enchimento.destroy();

    graficos.enchimento = new Chart(ctxEnchimento, {
        type: 'line',
        data: {
            labels: labels,
            datasets: [{
                label: 'Percentagem de Enchimento (%)',
                data: data.map(d => parseFloat(d.percentagem_enchimento) || 0),
                borderColor: '#3B82F6',
                backgroundColor: '#3B82F620',
                fill: true,
                tension: 0.4
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    display: false
                }
            },
            scales: {
                y: {
                    beginAtZero: true,
                    max: 100,
                    ticks: {
                        callback: function(value) {
                            return value + '%';
                        }
                    }
                }
            }
        }
    });

    // Gráfico de Cota
    const ctxCota = document.getElementById('grafico-cota').getContext('2d');
    if (graficos.cota) graficos.cota.destroy();

    graficos.cota = new Chart(ctxCota, {
        type: 'line',
        data: {
            labels: labels,
            datasets: [{
                label: 'Cota (m)',
                data: data.map(d => parseFloat(d.cota_actual) || 0),
                borderColor: '#10B981',
                backgroundColor: '#10B98120',
                fill: true,
                tension: 0.4
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    display: false
                }
            },
            scales: {
                y: {
                    ticks: {
                        callback: function(value) {
                            return value + ' m';
                        }
                    }
                }
            }
        }
    });

    // Gráfico de Volume
    const ctxVolume = document.getElementById('grafico-volume').getContext('2d');
    if (graficos.volume) graficos.volume.destroy();

    graficos.volume = new Chart(ctxVolume, {
        type: 'bar',
        data: {
            labels: labels,
            datasets: [{
                label: 'Volume (Mm³)',
                data: data.map(d => parseFloat(d.volume_actual) || 0),
                backgroundColor: '#F59E0B',
                borderColor: '#F59E0B',
                borderWidth: 1
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    display: false
                }
            },
            scales: {
                y: {
                    beginAtZero: true,
                    ticks: {
                        callback: function(value) {
                            return value + ' Mm³';
                        }
                    }
                }
            }
        }
    });
}

function atualizarTabelaHistorico(data) {
    const tbody = document.getElementById('historico-leituras');
    tbody.innerHTML = '';

    data.slice(-10).reverse().forEach((leitura, index) => {
        const variacao = index < data.length - 1 ?
            parseFloat(leitura.percentagem_enchimento) - parseFloat(data[data.length - index - 2]?.percentagem_enchimento || 0) : 0;

        const row = `
            <tr class="hover:bg-gray-50">
                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                    ${new Date(leitura.data_leitura).toLocaleDateString('pt-PT')}
                </td>
                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                    ${parseFloat(leitura.cota_actual || 0).toFixed(2)}
                </td>
                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                    ${parseFloat(leitura.volume_actual || 0).toFixed(1)}
                </td>
                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                    ${parseFloat(leitura.percentagem_enchimento || 0).toFixed(1)}%
                </td>
                <td class="px-6 py-4 whitespace-nowrap text-sm">
                    <span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium ${
                        variacao > 0 ? 'bg-green-100 text-green-800' :
                        variacao < 0 ? 'bg-red-100 text-red-800' :
                        'bg-gray-100 text-gray-800'
                    }">
                        ${variacao > 0 ? '↗' : variacao < 0 ? '↘' : '→'} ${Math.abs(variacao).toFixed(1)}%
                    </span>
                </td>
                <td class="px-6 py-4 whitespace-nowrap">
                    <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
                        Normal
                    </span>
                </td>
            </tr>
        `;
        tbody.innerHTML += row;
    });
}
</script>
@endsection