@extends('layouts.app')

@section('title', 'Gráficos - ' . $estacao->nome)

@section('content')
<div class="space-y-6">
    <!-- Header -->
    <div class="bg-gradient-to-r from-green-600 to-emerald-600 rounded-lg shadow-lg text-white p-6">
        <div class="flex items-center justify-between">
            <div>
                <nav class="flex items-center space-x-2 text-green-200 text-sm mb-2">
                    <a href="{{ route('charts.index') }}" class="hover:text-white">Gráficos</a>
                    <i class="fas fa-chevron-right text-xs"></i>
                    <a href="{{ route('charts.estacoes') }}" class="hover:text-white">Estações</a>
                    <i class="fas fa-chevron-right text-xs"></i>
                    <span class="text-white">{{ $estacao->nome }}</span>
                </nav>
                <h1 class="text-2xl font-bold flex items-center">
                    <i class="fas fa-satellite-dish mr-3"></i>
                    {{ $estacao->nome }}
                </h1>
                <p class="mt-2 opacity-90">Análise detalhada dos dados da estação {{ strtolower($estacao->getTipoDescricaoAmigavel()) }}</p>
            </div>
            <div class="text-right">
                @if($estacao->ultima_leitura)
                    <div class="text-lg font-bold">
                        @if($estacao->tipo == 'hidrometrica' && $estacao->ultima_leitura->nivel_hidrometrico)
                            {{ number_format($estacao->ultima_leitura->nivel_hidrometrico, 2) }} m
                        @elseif(in_array($estacao->tipo, ['pluviometrica', 'meteorologica']) && $estacao->ultima_leitura->precipitacao_mm)
                            {{ number_format($estacao->ultima_leitura->precipitacao_mm, 1) }} mm
                        @else
                            Dados Disponíveis
                        @endif
                    </div>
                    <div class="text-sm opacity-75">
                        @if($estacao->tipo == 'hidrometrica')
                            Nível Atual
                        @elseif(in_array($estacao->tipo, ['pluviometrica', 'meteorologica']))
                            Precipitação
                        @else
                            Última Leitura
                        @endif
                    </div>
                @else
                    <div class="text-xl">Sem dados</div>
                @endif
            </div>
        </div>
    </div>

    <!-- Filtros de Período -->
    <div class="bg-white rounded-lg shadow p-6">
        <h2 class="text-lg font-semibold text-gray-800 mb-4">Período de Análise</h2>
        <div class="flex flex-wrap gap-4">
            <button onclick="carregarDados(7)" class="px-4 py-2 bg-green-100 text-green-800 rounded-lg hover:bg-green-200 transition-colors" id="btn-7">
                7 dias
            </button>
            <button onclick="carregarDados(30)" class="px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 transition-colors" id="btn-30">
                30 dias
            </button>
            <button onclick="carregarDados(90)" class="px-4 py-2 bg-green-100 text-green-800 rounded-lg hover:bg-green-200 transition-colors" id="btn-90">
                3 meses
            </button>
            <button onclick="carregarDados(365)" class="px-4 py-2 bg-green-100 text-green-800 rounded-lg hover:bg-green-200 transition-colors" id="btn-365">
                1 ano
            </button>
        </div>
    </div>

    <!-- Estatísticas Atuais -->
    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        @if($estacao->ultima_leitura)
        <!-- Valor Principal (varia por tipo de estação) -->
        <div class="bg-white rounded-lg shadow p-6">
            <div class="flex items-center">
                <div class="flex-shrink-0">
                    <div class="w-12 h-12
                        @if($estacao->tipo == 'hidrometrica') bg-blue-100
                        @elseif($estacao->tipo == 'pluviometrica') bg-cyan-100
                        @else bg-green-100 @endif
                        rounded-lg flex items-center justify-center">
                        <i class="
                            @if($estacao->tipo == 'hidrometrica') fas fa-tint text-blue-600
                            @elseif($estacao->tipo == 'pluviometrica') fas fa-cloud-rain text-cyan-600
                            @else fas fa-thermometer-half text-green-600 @endif
                            text-xl"></i>
                    </div>
                </div>
                <div class="ml-4">
                    @if($estacao->tipo == 'hidrometrica')
                        <p class="text-sm text-gray-500">Nível Hidrométrico</p>
                        <p class="text-2xl font-bold text-gray-900">{{ number_format($estacao->ultima_leitura->nivel_hidrometrico ?? 0, 2) }}</p>
                        <p class="text-xs text-gray-500">metros</p>
                    @elseif(in_array($estacao->tipo, ['pluviometrica', 'meteorologica']))
                        <p class="text-sm text-gray-500">Precipitação</p>
                        <p class="text-2xl font-bold text-gray-900">{{ number_format($estacao->ultima_leitura->precipitacao_mm ?? 0, 1) }}</p>
                        <p class="text-xs text-gray-500">mm</p>
                    @else
                        <p class="text-sm text-gray-500">Última Leitura</p>
                        <p class="text-2xl font-bold text-gray-900">{{ $estacao->ultima_leitura->data_leitura->format('d/m') }}</p>
                    @endif
                </div>
            </div>
        </div>

        <!-- Temperatura Máxima -->
        @if(in_array($estacao->tipo, ['meteorologica', 'climatologica']) && $estacao->ultima_leitura->temperatura_max)
        <div class="bg-white rounded-lg shadow p-6">
            <div class="flex items-center">
                <div class="flex-shrink-0">
                    <div class="w-12 h-12 bg-red-100 rounded-lg flex items-center justify-center">
                        <i class="fas fa-thermometer-full text-red-600 text-xl"></i>
                    </div>
                </div>
                <div class="ml-4">
                    <p class="text-sm text-gray-500">Temp. Máxima</p>
                    <p class="text-2xl font-bold text-gray-900">{{ number_format($estacao->ultima_leitura->temperatura_max, 1) }}</p>
                    <p class="text-xs text-gray-500">°C</p>
                </div>
            </div>
        </div>
        @endif

        <!-- Temperatura Mínima -->
        @if(in_array($estacao->tipo, ['meteorologica', 'climatologica']) && $estacao->ultima_leitura->temperatura_min)
        <div class="bg-white rounded-lg shadow p-6">
            <div class="flex items-center">
                <div class="flex-shrink-0">
                    <div class="w-12 h-12 bg-blue-100 rounded-lg flex items-center justify-center">
                        <i class="fas fa-thermometer-empty text-blue-600 text-xl"></i>
                    </div>
                </div>
                <div class="ml-4">
                    <p class="text-sm text-gray-500">Temp. Mínima</p>
                    <p class="text-2xl font-bold text-gray-900">{{ number_format($estacao->ultima_leitura->temperatura_min, 1) }}</p>
                    <p class="text-xs text-gray-500">°C</p>
                </div>
            </div>
        </div>
        @endif

        <!-- Caudal (se for hidrométrica) -->
        @if($estacao->tipo == 'hidrometrica' && $estacao->ultima_leitura->caudal)
        <div class="bg-white rounded-lg shadow p-6">
            <div class="flex items-center">
                <div class="flex-shrink-0">
                    <div class="w-12 h-12 bg-purple-100 rounded-lg flex items-center justify-center">
                        <i class="fas fa-water text-purple-600 text-xl"></i>
                    </div>
                </div>
                <div class="ml-4">
                    <p class="text-sm text-gray-500">Caudal</p>
                    <p class="text-2xl font-bold text-gray-900">{{ number_format($estacao->ultima_leitura->caudal, 2) }}</p>
                    <p class="text-xs text-gray-500">m³/s</p>
                </div>
            </div>
        </div>
        @endif

        <!-- Status da Estação -->
        <div class="bg-white rounded-lg shadow p-6">
            <div class="flex items-center">
                <div class="flex-shrink-0">
                    <div class="w-12 h-12 {{ $estacao->estado == 'activa' ? 'bg-green-100' : 'bg-red-100' }} rounded-lg flex items-center justify-center">
                        <i class="fas {{ $estacao->estado == 'activa' ? 'fa-check-circle text-green-600' : 'fa-times-circle text-red-600' }} text-xl"></i>
                    </div>
                </div>
                <div class="ml-4">
                    <p class="text-sm text-gray-500">Status</p>
                    <p class="text-xl font-bold {{ $estacao->estado == 'activa' ? 'text-green-600' : 'text-red-600' }}">{{ ucfirst($estacao->estado) }}</p>
                    <p class="text-xs text-gray-500">{{ $estacao->ultima_leitura->data_leitura->format('d/m/Y') }}</p>
                </div>
            </div>
        </div>
        @else
        <div class="col-span-full bg-yellow-50 border border-yellow-200 rounded-lg p-6 text-center">
            <i class="fas fa-exclamation-triangle text-yellow-600 text-2xl mb-2"></i>
            <h3 class="text-lg font-medium text-yellow-800">Sem dados de leitura</h3>
            <p class="text-yellow-700 mt-1">Esta estação não possui leituras recentes registradas.</p>
        </div>
        @endif
    </div>

    <!-- Gráficos principais baseados no tipo de estação -->
    @if($estacao->tipo == 'hidrometrica')
        <!-- Gráficos para Estação Hidrométrica -->
        <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
            <!-- Níveis por Horário -->
            <div class="bg-white rounded-lg shadow p-6">
                <h3 class="text-lg font-semibold text-gray-800 mb-4">Níveis por Horário</h3>
                <div class="h-80">
                    <canvas id="grafico-niveis-horarios"></canvas>
                </div>
            </div>

            <!-- Nível Hidrométrico -->
            <div class="bg-white rounded-lg shadow p-6">
                <h3 class="text-lg font-semibold text-gray-800 mb-4">Nível Hidrométrico Médio</h3>
                <div class="h-80">
                    <canvas id="grafico-nivel-hidrometrico"></canvas>
                </div>
            </div>
        </div>

        <!-- Caudal (se disponível) -->
        <div class="bg-white rounded-lg shadow p-6">
            <h3 class="text-lg font-semibold text-gray-800 mb-4">Evolução do Caudal</h3>
            <div class="h-64">
                <canvas id="grafico-caudal"></canvas>
            </div>
        </div>

    @elseif(in_array($estacao->tipo, ['pluviometrica', 'meteorologica', 'climatologica']))
        <!-- Gráficos para Estações Meteorológicas/Pluviométricas -->
        <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
            <!-- Precipitação -->
            <div class="bg-white rounded-lg shadow p-6">
                <h3 class="text-lg font-semibold text-gray-800 mb-4">Precipitação Acumulada</h3>
                <div class="h-80">
                    <canvas id="grafico-precipitacao"></canvas>
                </div>
            </div>

            @if(in_array($estacao->tipo, ['meteorologica', 'climatologica']))
            <!-- Temperatura -->
            <div class="bg-white rounded-lg shadow p-6">
                <h3 class="text-lg font-semibold text-gray-800 mb-4">Variação de Temperatura</h3>
                <div class="h-80">
                    <canvas id="grafico-temperatura"></canvas>
                </div>
            </div>
            @endif
        </div>

        @if(in_array($estacao->tipo, ['meteorologica', 'climatologica']))
        <!-- Gráfico combinado -->
        <div class="bg-white rounded-lg shadow p-6">
            <h3 class="text-lg font-semibold text-gray-800 mb-4">Dados Meteorológicos Combinados</h3>
            <div class="h-80">
                <canvas id="grafico-combinado"></canvas>
            </div>
        </div>
        @endif
    @endif

    <!-- Tabela de Histórico Recente -->
    <div class="bg-white rounded-lg shadow">
        <div class="px-6 py-4 border-b border-gray-200">
            <h3 class="text-lg font-semibold text-gray-800">Histórico de Leituras</h3>
        </div>
        <div class="overflow-x-auto">
            <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Data</th>
                        @if($estacao->tipo == 'hidrometrica')
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Nível 6h</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Nível 12h</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Nível 18h</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Nível Médio</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Caudal</th>
                        @elseif(in_array($estacao->tipo, ['pluviometrica', 'meteorologica', 'climatologica']))
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Precipitação</th>
                            @if(in_array($estacao->tipo, ['meteorologica', 'climatologica']))
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Temp. Máx</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Temp. Mín</th>
                                <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Humidade</th>
                            @endif
                        @endif
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Operador</th>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200" id="historico-leituras">
                    <!-- Preenchido via JavaScript -->
                </tbody>
            </table>
        </div>
    </div>

    <!-- Informações da Estação -->
    <div class="bg-white rounded-lg shadow p-6">
        <h3 class="text-lg font-semibold text-gray-800 mb-4">Informações da Estação</h3>
        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            <div>
                <h4 class="font-medium text-gray-900 mb-2">Dados Técnicos</h4>
                <dl class="space-y-2 text-sm">
                    <div class="flex justify-between">
                        <dt class="text-gray-600">Código:</dt>
                        <dd class="font-medium">{{ $estacao->codigo }}</dd>
                    </div>
                    <div class="flex justify-between">
                        <dt class="text-gray-600">Tipo:</dt>
                        <dd class="font-medium">{{ ucfirst($estacao->tipo) }}</dd>
                    </div>
                    <div class="flex justify-between">
                        <dt class="text-gray-600">Equipamento:</dt>
                        <dd class="font-medium">{{ $estacao->equipamento ?? 'N/A' }}</dd>
                    </div>
                    <div class="flex justify-between">
                        <dt class="text-gray-600">Frequência:</dt>
                        <dd class="font-medium">{{ $estacao->frequencia_leitura ?? 'N/A' }}</dd>
                    </div>
                </dl>
            </div>
            <div>
                <h4 class="font-medium text-gray-900 mb-2">Localização</h4>
                <dl class="space-y-2 text-sm">
                    <div class="flex justify-between">
                        <dt class="text-gray-600">Província:</dt>
                        <dd class="font-medium">{{ $estacao->provincia ?? 'N/A' }}</dd>
                    </div>
                    <div class="flex justify-between">
                        <dt class="text-gray-600">Distrito:</dt>
                        <dd class="font-medium">{{ $estacao->distrito ?? 'N/A' }}</dd>
                    </div>
                    <div class="flex justify-between">
                        <dt class="text-gray-600">Rio:</dt>
                        <dd class="font-medium">{{ $estacao->rio ?? 'N/A' }}</dd>
                    </div>
                    <div class="flex justify-between">
                        <dt class="text-gray-600">Altitude:</dt>
                        <dd class="font-medium">{{ $estacao->altitude ? number_format($estacao->altitude, 0) . ' m' : 'N/A' }}</dd>
                    </div>
                </dl>
            </div>
            <div>
                <h4 class="font-medium text-gray-900 mb-2">Status Operacional</h4>
                <dl class="space-y-2 text-sm">
                    <div class="flex justify-between">
                        <dt class="text-gray-600">Estado:</dt>
                        <dd class="font-medium">
                            <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium
                                @if($estacao->estado == 'activa') bg-green-100 text-green-800
                                @else bg-red-100 text-red-800 @endif">
                                {{ ucfirst($estacao->estado) }}
                            </span>
                        </dd>
                    </div>
                    <div class="flex justify-between">
                        <dt class="text-gray-600">Última Leitura:</dt>
                        <dd class="font-medium">
                            @if($estacao->ultima_leitura)
                                {{ $estacao->ultima_leitura->data_leitura->format('d/m/Y') }}
                            @else
                                <span class="text-red-600">Sem dados</span>
                            @endif
                        </dd>
                    </div>
                    <div class="flex justify-between">
                        <dt class="text-gray-600">Instalação:</dt>
                        <dd class="font-medium">{{ $estacao->data_instalacao ? $estacao->data_instalacao->format('d/m/Y') : 'N/A' }}</dd>
                    </div>
                </dl>
            </div>
        </div>
    </div>
</div>

<!-- Scripts -->
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
let graficos = {};
let periodoAtual = 30;
const tipoEstacao = '{{ $estacao->tipo }}';

document.addEventListener('DOMContentLoaded', function() {
    carregarDados(30);
});

async function carregarDados(dias) {
    periodoAtual = dias;

    // Atualizar botões
    document.querySelectorAll('[id^="btn-"]').forEach(btn => {
        btn.className = 'px-4 py-2 bg-green-100 text-green-800 rounded-lg hover:bg-green-200 transition-colors';
    });
    document.getElementById(`btn-${dias}`).className = 'px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 transition-colors';

    try {
        const response = await fetch(`/graficos/api/estacao/{{ $estacao->id }}/data?days=${dias}`);
        const data = await response.json();

        criarGraficos(data);
        atualizarTabelaHistorico(data);
    } catch (error) {
        console.error('Erro ao carregar dados:', error);
    }
}

function criarGraficos(data) {
    const labels = data.map(d => new Date(d.data_leitura).toLocaleDateString('pt-PT'));

    if (tipoEstacao === 'hidrometrica') {
        criarGraficosHidrometricos(data, labels);
    } else if (['pluviometrica', 'meteorologica', 'climatologica'].includes(tipoEstacao)) {
        criarGraficosMeteorologicos(data, labels);
    }
}

function criarGraficosHidrometricos(data, labels) {
    // Gráfico de Níveis por Horário
    const ctxNiveis = document.getElementById('grafico-niveis-horarios').getContext('2d');
    if (graficos.niveis) graficos.niveis.destroy();

    const horarios = ['6h', '9h', '12h', '15h', '18h'];
    const coresHorarios = ['#3B82F6', '#10B981', '#F59E0B', '#EF4444', '#8B5CF6'];

    const datasetsNiveis = horarios.map((hora, index) => {
        const valores = data.map(d => parseFloat(d[`nivel_${hora}`]) || null);
        return {
            label: `Nível ${hora}`,
            data: valores,
            borderColor: coresHorarios[index],
            backgroundColor: 'transparent',
            fill: false,
            tension: 0.4,
            pointRadius: 3
        };
    });

    graficos.niveis = new Chart(ctxNiveis, {
        type: 'line',
        data: {
            labels: labels,
            datasets: datasetsNiveis
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                title: {
                    display: true,
                    text: 'Níveis de Água por Horário'
                }
            },
            scales: {
                y: {
                    ticks: {
                        callback: function(value) {
                            return value + ' m';
                        }
                    }
                }
            }
        }
    });

    // Gráfico de Nível Hidrométrico Médio
    const ctxHidrometrico = document.getElementById('grafico-nivel-hidrometrico').getContext('2d');
    if (graficos.hidrometrico) graficos.hidrometrico.destroy();

    graficos.hidrometrico = new Chart(ctxHidrometrico, {
        type: 'line',
        data: {
            labels: labels,
            datasets: [{
                label: 'Nível Hidrométrico (m)',
                data: data.map(d => parseFloat(d.nivel_hidrometrico) || 0),
                borderColor: '#3B82F6',
                backgroundColor: '#3B82F620',
                fill: true,
                tension: 0.4
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    display: false
                }
            },
            scales: {
                y: {
                    ticks: {
                        callback: function(value) {
                            return value + ' m';
                        }
                    }
                }
            }
        }
    });

    // Gráfico de Caudal
    const ctxCaudal = document.getElementById('grafico-caudal').getContext('2d');
    if (graficos.caudal) graficos.caudal.destroy();

    graficos.caudal = new Chart(ctxCaudal, {
        type: 'bar',
        data: {
            labels: labels,
            datasets: [{
                label: 'Caudal (m³/s)',
                data: data.map(d => parseFloat(d.caudal) || 0),
                backgroundColor: '#8B5CF6',
                borderColor: '#8B5CF6',
                borderWidth: 1
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    display: false
                }
            },
            scales: {
                y: {
                    beginAtZero: true,
                    ticks: {
                        callback: function(value) {
                            return value + ' m³/s';
                        }
                    }
                }
            }
        }
    });
}

function criarGraficosMeteorologicos(data, labels) {
    // Gráfico de Precipitação
    const ctxPrecipitacao = document.getElementById('grafico-precipitacao').getContext('2d');
    if (graficos.precipitacao) graficos.precipitacao.destroy();

    graficos.precipitacao = new Chart(ctxPrecipitacao, {
        type: 'bar',
        data: {
            labels: labels,
            datasets: [{
                label: 'Precipitação (mm)',
                data: data.map(d => parseFloat(d.precipitacao_mm) || 0),
                backgroundColor: '#06B6D4',
                borderColor: '#06B6D4',
                borderWidth: 1
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    display: false
                }
            },
            scales: {
                y: {
                    beginAtZero: true,
                    ticks: {
                        callback: function(value) {
                            return value + ' mm';
                        }
                    }
                }
            }
        }
    });

    // Gráfico de Temperatura (se aplicável)
    if (['meteorologica', 'climatologica'].includes(tipoEstacao)) {
        const ctxTemperatura = document.getElementById('grafico-temperatura').getContext('2d');
        if (graficos.temperatura) graficos.temperatura.destroy();

        graficos.temperatura = new Chart(ctxTemperatura, {
            type: 'line',
            data: {
                labels: labels,
                datasets: [{
                    label: 'Temperatura Máxima (°C)',
                    data: data.map(d => parseFloat(d.temperatura_max) || null),
                    borderColor: '#EF4444',
                    backgroundColor: 'transparent',
                    fill: false,
                    tension: 0.4
                }, {
                    label: 'Temperatura Mínima (°C)',
                    data: data.map(d => parseFloat(d.temperatura_min) || null),
                    borderColor: '#3B82F6',
                    backgroundColor: 'transparent',
                    fill: false,
                    tension: 0.4
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                scales: {
                    y: {
                        ticks: {
                            callback: function(value) {
                                return value + '°C';
                            }
                        }
                    }
                }
            }
        });

        // Gráfico Combinado
        const ctxCombinado = document.getElementById('grafico-combinado').getContext('2d');
        if (graficos.combinado) graficos.combinado.destroy();

        graficos.combinado = new Chart(ctxCombinado, {
            type: 'line',
            data: {
                labels: labels,
                datasets: [{
                    label: 'Precipitação (mm)',
                    data: data.map(d => parseFloat(d.precipitacao_mm) || 0),
                    borderColor: '#06B6D4',
                    backgroundColor: '#06B6D420',
                    fill: true,
                    tension: 0.4,
                    yAxisID: 'y'
                }, {
                    label: 'Temperatura Média (°C)',
                    data: data.map(d => {
                        const tMax = parseFloat(d.temperatura_max) || 0;
                        const tMin = parseFloat(d.temperatura_min) || 0;
                        return tMax && tMin ? (tMax + tMin) / 2 : null;
                    }),
                    borderColor: '#F59E0B',
                    backgroundColor: 'transparent',
                    fill: false,
                    tension: 0.4,
                    yAxisID: 'y1'
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                scales: {
                    y: {
                        type: 'linear',
                        display: true,
                        position: 'left',
                        ticks: {
                            callback: function(value) {
                                return value + ' mm';
                            }
                        }
                    },
                    y1: {
                        type: 'linear',
                        display: true,
                        position: 'right',
                        ticks: {
                            callback: function(value) {
                                return value + '°C';
                            }
                        },
                        grid: {
                            drawOnChartArea: false,
                        },
                    }
                }
            }
        });
    }
}

function atualizarTabelaHistorico(data) {
    const tbody = document.getElementById('historico-leituras');
    tbody.innerHTML = '';

    data.slice(-10).reverse().forEach((leitura) => {
        let row = `
            <tr class="hover:bg-gray-50">
                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                    ${new Date(leitura.data_leitura).toLocaleDateString('pt-PT')}
                </td>
        `;

        if (tipoEstacao === 'hidrometrica') {
            row += `
                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                    ${leitura.nivel_6h ? parseFloat(leitura.nivel_6h).toFixed(2) + ' m' : '-'}
                </td>
                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                    ${leitura.nivel_12h ? parseFloat(leitura.nivel_12h).toFixed(2) + ' m' : '-'}
                </td>
                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                    ${leitura.nivel_18h ? parseFloat(leitura.nivel_18h).toFixed(2) + ' m' : '-'}
                </td>
                <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                    ${leitura.nivel_hidrometrico ? parseFloat(leitura.nivel_hidrometrico).toFixed(2) + ' m' : '-'}
                </td>
                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                    ${leitura.caudal ? parseFloat(leitura.caudal).toFixed(2) + ' m³/s' : '-'}
                </td>
            `;
        } else {
            row += `
                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                    ${leitura.precipitacao_mm ? parseFloat(leitura.precipitacao_mm).toFixed(1) + ' mm' : '-'}
                </td>
            `;

            if (['meteorologica', 'climatologica'].includes(tipoEstacao)) {
                row += `
                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                        ${leitura.temperatura_max ? parseFloat(leitura.temperatura_max).toFixed(1) + '°C' : '-'}
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                        ${leitura.temperatura_min ? parseFloat(leitura.temperatura_min).toFixed(1) + '°C' : '-'}
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                        ${leitura.humidade_relativa ? parseFloat(leitura.humidade_relativa).toFixed(1) + '%' : '-'}
                    </td>
                `;
            }
        }

        row += `
                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                    ${leitura.operador?.name || 'Sistema'}
                </td>
            </tr>
        `;

        tbody.innerHTML += row;
    });
}
</script>
@endsection